/*
 * Copyright (C) 2018, 2019, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
    3.0.0	Support 'watch for' entries on PurchaseItems
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.time.LocalDateTime;
import java.util.*;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;
import jakarta.json.JsonObjectBuilder;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class ToDoListLister implements IToDoListLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private final DBKeyHandler<IToDoList> useToDoList = new DBKeyHandler<>("toDoListId");

	private final DBKeyHandler<IHusbandry> useHusbandry = new DBKeyHandler<>("husbandryId");

	private final DBKeyHandler<IPurchaseItem> usePurchaseItem = new DBKeyHandler<>("purchaseItemId");

	private final DBKeyHandler<IHusbandryClass> useHusbandryClass = new DBKeyHandler<>("husbandryClassId");

	private final DBKeyHandler<IPlantSpecies> usePlantSpecies = new DBKeyHandler<>("plantSpeciesId");

	private final DBKeyHandler<IPlantVariety> usePlantVariety = new DBKeyHandler<>("plantVarietyId");

	private final DBKeyHandler<IGroundworkActivity> useGroundworkActivity = new DBKeyHandler<>("groundWorkActivityId");

	private boolean useWhere = false;

	ToDoListLister() {}

	@Override
	public List<IToDoList> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		{
			new ReminderLister().buildToDoList();
		}
		List<IToDoList> vals = new ArrayList<>();
		boolean fetchAll = !useWhere;
        String query = "";
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = buildSQL_MySQL();
			case hsqldb -> query = buildSQL_hsqldb();
			case MSSQLServer -> query = buildSQL_MSSQLServer();
			default -> {
				LOGGER.error("fetch(): no known rdbms");
				throw new GNDBException(new IllegalStateException("ToDoListLister: fetch(): no known RDBMS"));
			}
		}
LOGGER.debug("fetch(): query: {}", query);

		try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
		{
			ResultSet rs = stmt.executeQuery(query);
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> vals = processResults_MySQL(rs);
				case hsqldb -> vals = processResults_hsqldb(rs);
				case MSSQLServer -> vals = processResults_MSSQLServer(rs);
				default -> {
					LOGGER.error("fetch(): no known rdbms");
					throw new GNDBException(new IllegalStateException("ToDoListLister: fetch(): no known RDBMS"));
				}
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("fetch(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

		if (vals.isEmpty()) return Collections.emptyList();

		for (IToDoList vx : vals)
		{
			MySQLCache.cacheToDoList.putIfAbsent(vx.getKey(), vx);
		}
		if (fetchAll) { MySQLCache.completeToDoList = true; }

		populateHusbandryClass();

		populateHusbandry(vals, false);

		populatePurchaseItem(vals, false);

		populatePlantSpecies(vals, false);

		populatePlantVariety(vals, false);

		populateGroundworkActivity();

LOGGER.traceExit(log4jEntryMsg);
		return vals;
	}	// fetch()

    private String buildSQL_MySQL()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from todolist as d ");
        query.append("left join (select * from comment where ownerType = 'TD') as c ");
        query.append("on d.toDoListId = c.ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        return query.toString();
    }

    private String buildSQL_hsqldb()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from todolist as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'TD') as c ");
        query.append("on d.toDoListId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        return query.toString();
    }

    private String buildSQL_MSSQLServer()
    {
        StringBuilder query = new StringBuilder("select d.toDoListId as d_toDoListId, " +
                                                        "d.plantSpeciesId as d_plantSpeciesId, " +
                                                        "d.plantVarietyId as d_plantVarietyId, " +
                                                        "d.husbandryClassId as d_husbandryClassId, " +
                                                        "d.groundWorkActivityId as d_groundWorkActivityId, " +
                                                        "d.husbandryId as d_husbandryId, " +
														"d.purchaseItemId as d_purchaseItemId, " +
                                                        "d.description as d_description, " +
                                                        "d.lastUpdated as d_lastUpdated, " +
                                                        "d.created as d_created, " +
                                                        " c.* from todolist as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'TD') as c ");
        query.append("on d.toDoListId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        return query.toString();
    }

    private void buildCommonSQL(StringBuilder query)
    {
        boolean first = true;
		if (useToDoList.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(useToDoList.getSQLClause("d"));
			first = false;
			useToDoList.clear();
		}

		if (useHusbandry.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and ");
			query.append(useHusbandry.getSQLClause("d"));
			first = false;
			useHusbandry.clear();
		}

		if (usePurchaseItem.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and ");
			query.append(usePurchaseItem.getSQLClause("d"));
			first = false;
			usePurchaseItem.clear();
		}

		if (useHusbandryClass.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and ");
			query.append(useHusbandryClass.getSQLClause("d"));
			first = false;
			useHusbandryClass.clear();
		}

//        if (this.usePlantSpeciesId || this.usePlantVarietyId)
		if (this.usePlantSpecies.isUsed() || this.usePlantVariety.isUsed())
        {
            if (first)
                query.append(" where ");
            else
                query.append(" and ");

			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" ( ");

			if (this.usePlantSpecies.isUsed())
			{
				query.append(usePlantSpecies.getSQLClause("d"));
			}

			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" or ");

			if (this.usePlantVariety.isUsed())
			{
				query.append(usePlantVariety.getSQLClause("d"));
			}

			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" ) ");

			first = false;
			usePlantSpecies.clear();
			usePlantVariety.clear();
        }

		if (useGroundworkActivity.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and ");
			query.append(useGroundworkActivity.getSQLClause("d"));
			first = false;
			useGroundworkActivity.clear();
		}
    }   //  buildCommonSQL()

	private List<IToDoList> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<IToDoList> tempList = new ArrayList<>();

		ToDoList item = null;

		while (rs.next())
		{
			int toDoListId = rs.getInt("d.toDoListId");
			int tmp_plantSpeciesId = rs.getInt("d.plantSpeciesId");
			Integer plantSpeciesId = rs.wasNull() ? null : tmp_plantSpeciesId;
			int tmp_plantVarietyId = rs.getInt("d.plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			int tmp_husbandryClassId = rs.getInt("d.husbandryClassId");
			Integer husbandryClassId = rs.wasNull() ? null : tmp_husbandryClassId;
			int tmp_groundWorkActivityId = rs.getInt("d.groundWorkActivityId");
			Integer groundWorkActivityId = rs.wasNull() ? null : tmp_groundWorkActivityId;

			int tmp_husbandryId = rs.getInt("d.husbandryId");
			Integer husbandryId = rs.wasNull() ? null : tmp_husbandryId;

			int tmp_purchaseItemId = rs.getInt("d.purchaseItemId");
			Integer purchaseItemId = rs.wasNull() ? null : tmp_purchaseItemId;

			String description = rs.getString("d.description");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("toDoListId: {}, plantSpeciesId: {}, plantVarietyId: {}, husbandryClassId: {}, groundWorkActivityId: {}, husbandryId: {}, purchaseItemId: {}, description: {}, lastUpdated: {}, created: {}",
                toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, purchaseItemId, description, lastUpdated, created);
			if (item != null && toDoListId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_MySQL(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("commentId"),
					rs.getInt("ownerId"),
					"TD",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): extra comment is: {}", comm);
				item = new ToDoList(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
//					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, description, lastUpdated, created);
					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, purchaseItemId, description, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						toDoListId,
						"TD",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): first comment is: {}", comm);
//					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, description, lastUpdated, created, comm);
					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, purchaseItemId, description, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<IToDoList> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<IToDoList> tempList = new ArrayList<>();

		ToDoList item = null;

		while (rs.next())
		{
			int toDoListId = rs.getInt("todolist.toDoListId");
			int tmp_plantSpeciesId = rs.getInt("todolist.plantSpeciesId");
			Integer plantSpeciesId = rs.wasNull() ? null : tmp_plantSpeciesId;
			int tmp_plantVarietyId = rs.getInt("todolist.plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			int tmp_husbandryClassId = rs.getInt("todolist.husbandryClassId");
			Integer husbandryClassId = rs.wasNull() ? null : tmp_husbandryClassId;
			int tmp_groundWorkActivityId = rs.getInt("todolist.groundWorkActivityId");
			Integer groundWorkActivityId = rs.wasNull() ? null : tmp_groundWorkActivityId;

			int tmp_husbandryId = rs.getInt("todolist.husbandryId");
			Integer husbandryId = rs.wasNull() ? null : tmp_husbandryId;

			int tmp_purchaseItemId = rs.getInt("todolist.purchaseItemId");
			Integer purchaseItemId = rs.wasNull() ? null : tmp_purchaseItemId;

			String description = rs.getString("todolist.description");
			LocalDateTime lastUpdated = rs.getTimestamp("todolist.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("todolist.created").toLocalDateTime();
LOGGER.debug("toDoListId: {}, plantSpeciesId: {}, plantVarietyId: {}, husbandryClassId: {}, groundWorkActivityId: {}, husbandryId: {}, purchaseItemId: {}, description: {}, lastUpdated: {}, created: {}",
                toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, purchaseItemId, description, lastUpdated, created);
			if (item != null && toDoListId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"TD",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new ToDoList(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
//					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, description, lastUpdated, created);
					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, purchaseItemId, description, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						toDoListId,
						"TD",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
//					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, description, lastUpdated, created, comm);
					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, purchaseItemId, description, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<IToDoList> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<IToDoList> tempList = new ArrayList<>();

		ToDoList item = null;

		while (rs.next())
		{
			int toDoListId = rs.getInt("d_toDoListId");
			int tmp_plantSpeciesId = rs.getInt("d_plantSpeciesId");
			Integer plantSpeciesId = rs.wasNull() ? null : tmp_plantSpeciesId;
			int tmp_plantVarietyId = rs.getInt("d_plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			int tmp_husbandryClassId = rs.getInt("d_husbandryClassId");
			Integer husbandryClassId = rs.wasNull() ? null : tmp_husbandryClassId;
			int tmp_groundWorkActivityId = rs.getInt("d_groundWorkActivityId");
			Integer groundWorkActivityId = rs.wasNull() ? null : tmp_groundWorkActivityId;

			int tmp_husbandryId = rs.getInt("d_husbandryId");
			Integer husbandryId = rs.wasNull() ? null : tmp_husbandryId;

			int tmp_purchaseItemId = rs.getInt("d_purchaseItemId");
			Integer purchaseItemId = rs.wasNull() ? null : tmp_purchaseItemId;

			String description = rs.getString("d_description");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("toDoListId: {}, plantSpeciesId: {}, plantVarietyId: {}, husbandryClassId: {}, groundWorkActivityId: {}, husbandryId: {}, purchaseItemId: {}, description: {}, lastUpdated: {}, created: {}",
                toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, purchaseItemId, description, lastUpdated, created);
			if (item != null && toDoListId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"TD",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new ToDoList(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
//					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, description, lastUpdated, created);
					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, purchaseItemId, description, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						toDoListId,
						"TD",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
//					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, description, lastUpdated, created, comm);
					item = new ToDoList(toDoListId, plantSpeciesId, plantVarietyId, husbandryClassId, groundWorkActivityId, husbandryId, purchaseItemId, description, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	/*
	*	Populate the parents slot(s)
	*/
	private void populateHusbandry(List<IToDoList> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: NL
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populateHusbandry()");

		if (fetchAll)
		{
			new HusbandryLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			map(c -> ((ToDoList)c).getHusbandryId()).filter(Objects::nonNull).
			mapToInt(Integer::intValue).filter(c -> c>0).distinct().
			toArray();

		if (keys.length > 0)
		{
			new HusbandryLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	private void populatePurchaseItem(List<IToDoList> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: NL
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populatePurchaseItem()");

		if (fetchAll)
		{
			new PurchaseLister().fetch();
			return;
		}
		int[] keys = vals.stream().
				map(c -> ((ToDoList)c).getPurchaseItemId()).filter(Objects::nonNull).
				mapToInt(Integer::intValue).filter(c -> c>0).distinct().
				toArray();

		if (keys.length > 0)
		{
			new PurchaseItemLister().id(keys).fetch();
		}
		LOGGER.traceExit(log4jEntryMsg);
	}

	private void populateHusbandryClass() throws GNDBException
	{
// parent table type: TF
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populateHusbandryClass()");

// just make sure everything is cached
		new HusbandryClassLister().load();
LOGGER.traceExit(log4jEntryMsg);
	}

	private void populatePlantSpecies(List<IToDoList> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: TD
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populatePlantSpecies()");

		if (fetchAll)
		{
			new PlantSpeciesLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			map(c -> ((ToDoList)c).getPlantSpeciesId()).filter(Objects::nonNull).
			mapToInt(Integer::intValue).filter(c -> c>0).distinct().
			toArray();

		if (keys.length > 0)
		{
			new PlantSpeciesLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	private void populatePlantVariety(List<IToDoList> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: NL
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populatePlantVariety()");

		if (fetchAll)
		{
			new PlantVarietyLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			map(c -> ((ToDoList)c).getPlantVarietyId()).filter(Objects::nonNull).
			mapToInt(Integer::intValue).filter(c -> c>0).distinct().
			toArray();

		if (keys.length > 0)
		{
			new PlantVarietyLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	private void populateGroundworkActivity() throws GNDBException
	{
// parent table type: TF
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populateGroundworkActivity()");

// just make sure everything is cached
		new GroundworkActivityLister().load();
LOGGER.traceExit(log4jEntryMsg);
	}

	void clear()
	{
		MySQLCache.cacheToDoList.clear();
		MySQLCache.completeToDoList = false;
	}

	/**
	*
	*Select only the ToDoList entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	ToDoListLister id(int... vals)
	{
		useToDoList.id(vals);
		useWhere = useWhere || useToDoList.isUsed();
		return this;
	}

	@Override
	public IToDoListLister toDoList(IToDoList... items)
	{
		useToDoList.item(items);
		useWhere = useWhere || useToDoList.isUsed();
		return this;
	}

	@Override
	public IToDoListLister toDoList(List<IToDoList> items)
	{
		useToDoList.item(items);
		useWhere = useWhere || useToDoList.isUsed();
		return this;
	}

	@Override
	public IToDoListLister husbandry(IHusbandry... items)
	{
		useHusbandry.item(items);
		useWhere = useWhere || useHusbandry.isUsed();
		return this;
	}

	@Override
	public IToDoListLister husbandry(List<IHusbandry> items)
	{
		useHusbandry.item(items);
		useWhere = useWhere || useHusbandry.isUsed();
		return this;
	}

	@Override
	public IToDoListLister purchaseItem(IPurchaseItem... items)
	{
		usePurchaseItem.item(items);
		useWhere = useWhere || usePurchaseItem.isUsed();
		return this;
	}

	@Override
	public IToDoListLister purchaseItem(List<IPurchaseItem> items)
	{
		usePurchaseItem.item(items);
		useWhere = useWhere || usePurchaseItem.isUsed();
		return this;
	}

	@Override
	public IToDoListLister husbandryClass(IHusbandryClass... items)
	{
		useHusbandryClass.item(items);
		useWhere = useWhere || useHusbandryClass.isUsed();
		return this;
	}

	@Override
	public IToDoListLister husbandryClass(List<IHusbandryClass> items)
	{
		useHusbandryClass.item(items);
		useWhere = useWhere || useHusbandryClass.isUsed();
		return this;
	}

	@Override
	public IToDoListLister plantSpecies(IPlantSpecies... items)
	{
		usePlantSpecies.item(items);
		useWhere = useWhere || usePlantSpecies.isUsed();
		return this;
	}

	@Override
	public IToDoListLister plantSpecies(List<IPlantSpecies> items)
	{
		usePlantSpecies.item(items);
		useWhere = useWhere || usePlantSpecies.isUsed();
		return this;
	}

	@Override
	public IToDoListLister plantVariety(IPlantVariety... items)
	{
		usePlantVariety.item(items);
		useWhere = useWhere || usePlantVariety.isUsed();
		return this;
	}

	@Override
	public IToDoListLister plantVariety(List<IPlantVariety> items)
	{
		usePlantVariety.item(items);
		useWhere = useWhere || usePlantVariety.isUsed();
		return this;
	}

	@Override
	public IToDoListLister groundworkActivity(IGroundworkActivity... items)
	{
		useGroundworkActivity.item(items);
		useWhere = useWhere || useGroundworkActivity.isUsed();
		return this;
	}

	@Override
	public IToDoListLister groundworkActivity(List<IGroundworkActivity> items)
	{
		useGroundworkActivity.item(items);
		useWhere = useWhere || useGroundworkActivity.isUsed();
		return this;
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		useWhere = false;
		fetch();

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (IToDoList ihc : MySQLCache.cacheToDoList.values())
		{
			ToDoList hc = (ToDoList)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "ToDoList");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "ToDoList.json"), false)))
		{
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJson(): IOException", ex);
		}
	}	// toJson

}
