/*
 * Copyright (C) 2018, 2019, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
    2.5.0   First proper implementation
            Support selection by PlantSpecies and Variety
	2.9.5	Fixed bug in selection by plant species/variety
	3.0.0	Use DBKeyHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;
import java.util.ArrayList;
import java.util.Collections;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;
import jakarta.json.JsonObjectBuilder;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class SaleLister implements ISaleLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private final DBKeyHandler<ISale> useSale = new DBKeyHandler<>("saleId");

	private final DBKeyHandler<IPlantSpecies> usePlantSpecies = new DBKeyHandler<>("plantSpeciesId");

	private final DBKeyHandler<IPlantVariety> usePlantVariety = new DBKeyHandler<>("plantVarietyId");

	private boolean useFromDate = false;
	private LocalDate fromDate;
	private boolean useToDate = false;
	private LocalDate toDate;

	private boolean useWhere = false;

	SaleLister() {}

	@Override
	public List<ISale> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		List<ISale> vals = new ArrayList<>();
		boolean fetchAll = !useWhere;
        String query = "";
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = buildSQL_MySQL();
			case hsqldb -> query = buildSQL_hsqldb();
			case MSSQLServer -> query = buildSQL_MSSQLServer();
			default -> {
				LOGGER.error("fetch(): no known rdbms");
				throw new GNDBException(new IllegalStateException("SaleLister: fetch(): no known RDBMS"));
			}
		}
LOGGER.debug("fetch(): query: {}", query);

		try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
		{
			ResultSet rs = stmt.executeQuery(query);
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> vals = processResults_MySQL(rs);
				case hsqldb -> vals = processResults_hsqldb(rs);
				case MSSQLServer -> vals = processResults_MSSQLServer(rs);
				default -> {
					LOGGER.error("fetch(): no known rdbms");
					throw new GNDBException(new IllegalStateException("SaleLister: fetch(): no known RDBMS"));
				}
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("fetch(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

		if (vals.isEmpty()) return Collections.emptyList();

		for (ISale vx : vals)
		{
			MySQLCache.cacheSale.putIfAbsent(vx.getKey(), vx);
		}
		if (fetchAll) { MySQLCache.completeSale = true; }

LOGGER.traceExit(log4jEntryMsg);
		return vals;
	}	// fetch()

    private String buildSQL_MySQL()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from sale as d ");
        query.append("left join (select * from comment where ownerType = 'SA') as c ");
        query.append("on d.saleId = c.ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.saleId, c.date");
        return query.toString();
    }   //  buildSQL_MySQL()

    private String buildSQL_hsqldb()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from sale as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'SA') as c ");
        query.append("on d.saleId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.saleId, c_date");
        return query.toString();
    }   //  buildSQL_hsqldb()

    private String buildSQL_MSSQLServer()
    {
        StringBuilder query = new StringBuilder("select d.saleId as d_saleId, " +
                                                        "d.date as d_date, " +
                                                        "d.purchasedBy as d_purchasedBy, " +
                                                        "d.totalCost as d_totalCost, " +
                                                        "d.currency as d_currency, " +
                                                        "d.lastUpdated as d_lastUpdated, " +
                                                        "d.created as d_created," +
                                                        " c.* from sale as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'SA') as c ");
        query.append("on d.saleId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.saleId, c_date");
        return query.toString();
    }   //  buildSQL_MSSQLServer()

    private void buildCommonSQL(StringBuilder query)
    {
        boolean first = true;
		if (useSale.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(useSale.getSQLClause("d"));
			first = false;
			useSale.clear();
		}

        //  2.5.0
//        if (this.usePlantSpeciesId || this.usePlantVarietyId)
		if (this.usePlantSpecies.isUsed() || this.usePlantVariety.isUsed())
        {
            if (first) query.append(" where ");
            else query.append(" and ");
            first = false;

            query.append(" d.saleId in (select saleid from saleitem as pi where ");
			if (this.usePlantSpecies.isUsed())
			{
				query.append(usePlantSpecies.getSQLClause("pi"));
			}

			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" or ");

			if (this.usePlantVariety.isUsed())
			{
				query.append(usePlantVariety.getSQLClause("pi"));
			}
            query.append(" )");

			usePlantSpecies.clear();
			usePlantVariety.clear();
        }

        if (this.useFromDate)
        {
            if (first) query.append(" where ");
            else query.append(" and");

            query.append(" d.date >= '").append(this.fromDate).append("'");
            first = false;
            this.useFromDate = false;
        }
        if (this.useToDate)
        {
            if (first) query.append(" where ");
            else query.append(" and");

            query.append(" d.date <= '").append(this.toDate).append("'");
            this.useToDate = false;
        }
    }   //  buildCommonSQL()

	private List<ISale> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<ISale> tempList = new ArrayList<>();

		Sale item = null;

		while (rs.next())
		{
			int saleId = rs.getInt("d.saleId");
			LocalDate date = rs.getDate("d.date").toLocalDate();
			String purchasedBy = rs.getString("d.purchasedBy");
			BigDecimal totalCost = rs.getBigDecimal("d.totalCost");
			String currency = rs.getString("d.currency");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("saleId: {}, date: {}, purchasedBy: {}, totalCost: {}, currency: {}, lastUpdated: {}, created: {}",
                saleId, date, purchasedBy, totalCost, currency, lastUpdated, created);
			if (item != null && saleId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_MySQL(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("commentId"),
					rs.getInt("ownerId"),
					"SA",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): extra comment is: {}", comm);
				item = new Sale(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
					item = new Sale(saleId, date, purchasedBy, totalCost, currency, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						saleId,
						"SA",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): first comment is: {}", comm);
					item = new Sale(saleId, date, purchasedBy, totalCost, currency, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<ISale> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<ISale> tempList = new ArrayList<>();

		Sale item = null;

		while (rs.next())
		{
			int saleId = rs.getInt("sale.saleId");
			LocalDate date = rs.getDate("sale.date").toLocalDate();
			String purchasedBy = rs.getString("sale.purchasedBy");
			BigDecimal totalCost = rs.getBigDecimal("sale.totalCost");
			String currency = rs.getString("sale.currency");
			LocalDateTime lastUpdated = rs.getTimestamp("sale.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("sale.created").toLocalDateTime();
LOGGER.debug("saleId: {}, date: {}, purchasedBy: {}, totalCost: {}, currency: {}, lastUpdated: {}, created: {}",
                saleId, date, purchasedBy, totalCost, currency, lastUpdated, created);
			if (item != null && saleId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"SA",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new Sale(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new Sale(saleId, date, purchasedBy, totalCost, currency, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						saleId,
						"SA",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new Sale(saleId, date, purchasedBy, totalCost, currency, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<ISale> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<ISale> tempList = new ArrayList<>();

		Sale item = null;

		while (rs.next())
		{
			int saleId = rs.getInt("d_saleId");
			LocalDate date = rs.getDate("d_date").toLocalDate();
			String purchasedBy = rs.getString("d_purchasedBy");
			BigDecimal totalCost = rs.getBigDecimal("d_totalCost");
			String currency = rs.getString("d_currency");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("saleId: {}, date: {}, purchasedBy: {}, totalCost: {}, currency: {}, lastUpdated: {}, created: {}",
                saleId, date, purchasedBy, totalCost, currency, lastUpdated, created);
			if (item != null && saleId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"SA",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new Sale(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new Sale(saleId, date, purchasedBy, totalCost, currency, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						saleId,
						"SA",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new Sale(saleId, date, purchasedBy, totalCost, currency, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	void clear()
	{
		MySQLCache.cacheSale.clear();
		MySQLCache.completeSale = false;
	}

	/**
	*
	*Select only the Sale entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	SaleLister id(int... vals)
	{
		useSale.id(vals);
		useWhere = useWhere || useSale.isUsed();
		return this;
	}

	@Override
	public ISaleLister sale(ISale... items)
	{
		useSale.item(items);
		useWhere = useWhere || useSale.isUsed();
		return this;
	}

	@Override
	public ISaleLister sale(List<ISale> items)
	{
		useSale.item(items);
		useWhere = useWhere || useSale.isUsed();
		return this;
	}

	@Override
	public ISaleLister plantSpecies(IPlantSpecies... items)
	{
		usePlantSpecies.item(items);
		useWhere = useWhere || usePlantSpecies.isUsed();
		return this;
	}

	@Override
	public ISaleLister plantSpecies(List<IPlantSpecies> items)
	{
		usePlantSpecies.item(items);
		useWhere = useWhere || usePlantSpecies.isUsed();
		return this;
	}

	@Override
	public ISaleLister plantVariety(IPlantVariety... items)
	{
		usePlantVariety.item(items);
		useWhere = useWhere || usePlantVariety.isUsed();
		return this;
	}

	@Override
	public ISaleLister plantVariety(List<IPlantVariety> items)
	{
		usePlantVariety.item(items);
		useWhere = useWhere || usePlantVariety.isUsed();
		return this;
	}

	@Override
	public ISaleLister fromDate(LocalDate item)
	{
		if (item == null) return this;
		this.fromDate = item;
		this.useFromDate = true;
		this.useWhere = true;
		return this;
	}

	@Override
	public ISaleLister toDate(LocalDate item)
	{
		if (item == null) return this;
		this.toDate = item;
		this.useToDate = true;
		this.useWhere = true;
		return this;
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		useWhere = false;
		fetch();

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (ISale ihc : MySQLCache.cacheSale.values())
		{
			Sale hc = (Sale)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "Sale");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "Sale.json"), false)))
		{
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJson(): IOException", ex);
		}
	}	// toJson

}
