/*
 * Copyright (C) 2018, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Use FlagHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import java.beans.PropertyChangeListener;

import uk.co.gardennotebook.spi.*;

import uk.co.gardennotebook.util.SimpleMoney;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;

import java.time.*;
import java.math.BigDecimal;

import java.util.Optional;

import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonArray;
import jakarta.json.JsonObjectBuilder;
import jakarta.json.JsonObject;

/**
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/

final class Sale implements ISale
{
	private final FlagHandler<ISale> flagHandler;
	{
		flagHandler = new FlagHandler<>(this);
	}

	private final int id;
	private final LocalDate date;
	private final String purchasedBy;
	private final BigDecimal totalCost;

	/*
	*	ISO 4217 standard currency code (GBP, USD, EUR, etc).  Null means the local currency.
	*/
	private final String currency;
	private final LocalDateTime lastUpdated;
	private final LocalDateTime created;
	private final List<Comment> commentsList;

	/**
	*	Build an immutable Sale entry one field at a time
	*/
	Sale(
		final int id,
		final LocalDate date,
		final String purchasedBy,
		final BigDecimal totalCost,
		final String currency,
		final LocalDateTime lastUpdated,
		final LocalDateTime created,
		final Comment... comments)
	{
		this.id = id;
		this.date = date;
		this.purchasedBy = purchasedBy;
		this.totalCost = totalCost;
		this.currency = currency;
		this.lastUpdated = lastUpdated;
		this.created = created;
		if (comments != null && comments.length>0)
		{
			this.commentsList = new ArrayList<>(Arrays.asList(comments));
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}

	/**
	*	Build an immutable Sale entry cloning the given Sale entry but adding the comments list
	*/
	Sale(
		final Sale toCopy,
		final Comment... comments)
	{
		this(toCopy, Arrays.asList(comments));
	}

	/**
	*	Build an immutable Sale entry cloning the given Sale entry but adding the comments list
	*/
	Sale(
		final Sale toCopy,
		final List<Comment> comments)
	{
		this.id = toCopy.id;
		this.date = toCopy.date;
		this.purchasedBy = toCopy.purchasedBy;
		this.totalCost = toCopy.totalCost;
		this.currency = toCopy.currency;
		this.lastUpdated = toCopy.lastUpdated;
		this.created = toCopy.created;
		if (comments != null && comments.size()>0)
		{
			if (toCopy.commentsList.size()>0)
			{
				// append new comments to previous list
				this.commentsList = new ArrayList<>(toCopy.commentsList);
				this.commentsList.addAll(comments);
			}
			else
			{	// no comments on original item
				this.commentsList = new ArrayList<>(comments);
			}
		}
		else
		{	// no new comments to add
			this.commentsList = toCopy.commentsList;
		}
	}

	/**
	*	Build an immutable Sale entry from a JSON dump.
	*	The dumped object must be complete (all non-nullable fields must have values) except
	*	the id field can be null or absent to indicate that this is a new item to be inserted.
	*
	*	@param	json	a JsonObject holding all the fields for a full initialisation.
	*/
	Sale(JsonObject json)
	{
		this.id = json.getInt("id", -1);
		this.date = LocalDate.parse(json.getString("date"));
		if (json.containsKey("purchasedBy") && !json.isNull("purchasedBy"))
		{
			this.purchasedBy = json.getString("purchasedBy");
		}
		else
		{
			this.purchasedBy = null;
		}

		if (json.containsKey("totalCost") && !json.isNull("totalCost"))
		{
			this.totalCost = json.getJsonNumber("totalCost").bigDecimalValue();
		}
		else
		{
			this.totalCost = null;
		}

		if (json.containsKey("currency") && !json.isNull("currency"))
		{
			this.currency = json.getString("currency");
		}
		else
		{
			this.currency = null;
		}

		this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
		this.created = LocalDateTime.parse(json.getString("created"));
		JsonArray jsonComments = json.getJsonArray("comments");
		if (jsonComments != null && !jsonComments.isEmpty())
		{// there is probably only one comment
			this.commentsList = new ArrayList<>(jsonComments.size());
			for (JsonObject ct : jsonComments.getValuesAs(JsonObject.class))
			{
				this.commentsList.add(new Comment(ct));
			}
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}	//constructor from JSON

	int getId()
	{
		return id;
	}
	@Override
	public Integer getKey()
	{
		return id;
	}

	@Override
	public NotebookEntryType getType()
	{
		return NotebookEntryType.SALE;
	}

	@Override
	public LocalDate getDate()
	{
		return date;
	}

	@Override
	public Optional<String> getPurchasedBy()
	{
		return Optional.ofNullable(purchasedBy);
	}

	@Override
	public Optional<BigDecimal> getTotalCost()
	{
		return Optional.ofNullable(totalCost);
	}

	@Override
	public Optional<String> getCurrency()
	{
		return Optional.ofNullable(currency);
	}

	@Override
	public LocalDateTime getLastUpdated()
	{
		return lastUpdated;
	}

	@Override
	public LocalDateTime getCreated()
	{
		return created;
	}

	/**
	*	get the value of totalCost as a Money value
	*
	*	@return	 the value of totalCost
	*/
	public SimpleMoney getTotalPrice()

	{
		if (currency == null)
		{
			return new SimpleMoney(totalCost);
		}
		return new SimpleMoney(currency, totalCost);
	}

	@Override
	public boolean sameAs(INotebookEntry other)
	{
		if (other == null || other.getType() != this.getType())
		{
			return false;
		}
		return other.getKey().equals(this.getKey());
	}

	@Override
	public ISaleItemLister getSaleItem()
	{
		return new SaleItemLister().sale(this);
	}

	@Override
	public List<IComment> getComments() {
		return new ArrayList<>(this.commentsList);
	}

	JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
	{
		JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
		jsonBuilder.add("id", id);
		jsonBuilder.add("date", date.toString());
		if (purchasedBy != null)
		{
			jsonBuilder.add("purchasedBy", purchasedBy);
		}
		else
		{
			jsonBuilder.addNull("purchasedBy");
		}
		if (totalCost != null)
		{
			jsonBuilder.add("totalCost", totalCost);
		}
		else
		{
			jsonBuilder.addNull("totalCost");
		}
		if (currency != null)
		{
			jsonBuilder.add("currency", currency);
		}
		else
		{
			jsonBuilder.addNull("currency");
		}
		jsonBuilder.add("lastUpdated", lastUpdated.toString());
		jsonBuilder.add("created", created.toString());
		if (commentsList != null && !commentsList.isEmpty())
		{// no point writing an empty comments array (the loaders handle this)
			JsonArrayBuilder jsonComments = jsonFactory.createArrayBuilder();
			for (Comment ct : commentsList)
			{
				jsonComments.add(ct.toJson(jsonFactory));
			}
			jsonBuilder.add("comments", jsonComments);
		}
		jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "Sale");
		return jsonBuilder;
	}	//	toJson

	@Override
	public void addPropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		flagHandler.addPropertyChangeListener(propertyName, listener);
	}

	@Override
	public void removePropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		flagHandler.removePropertyChangeListener(propertyName, listener);
	}

	@Override
	public void flagDeleted()
	{
		flagHandler.flagDeleted();
	}

	@Override
	public void flagReplaced(final ISale newValue)
	{
		flagHandler.flagReplaced(newValue, newValue::addPropertyChangeListener);
	}

	@Override
	public void flagChildDeleted(final ISaleItem child)
	{
		flagHandler.flagChildDeleted("SaleItem", child);
	}

	@Override
	public void flagChildAdded(final ISaleItem child)
	{
		flagHandler.flagChildAdded("SaleItem", child);
	}


	@Override
	public String toString() {
		return "Sale: " + "id: " + id + ", " +
				"date: " + date + ", " +
				"purchasedBy: " + purchasedBy + ", " +
				"totalCost: " + totalCost + ", " +
				"currency: " + currency + ", " +
				"lastUpdated: " + lastUpdated + ", " +
				"created: " + created;
	}

}
