/*
 * Copyright (C) 2018, 2019, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
    3.0.0	Use DBKeyHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;
import java.util.ArrayList;
import java.util.Collections;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;
import jakarta.json.JsonObjectBuilder;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class PurchaseItemLister implements IPurchaseItemLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private final DBKeyHandler<IPurchaseItem> usePurchaseItem = new DBKeyHandler<>("purchaseItemId");

	private final DBKeyHandler<IProduct> useProduct = new DBKeyHandler<>("productId");

	private final DBKeyHandler<IPurchase> usePurchase = new DBKeyHandler<>("purchaseId");

	private boolean useWhere = false;

	PurchaseItemLister() {}

	@Override
	public List<IPurchaseItem> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		List<IPurchaseItem> vals = new ArrayList<>();
		boolean fetchAll = !useWhere;
        String query = "";
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = buildSQL_MySQL();
			case hsqldb -> query = buildSQL_hsqldb();
			case MSSQLServer -> query = buildSQL_MSSQLServer();
			default -> {
				LOGGER.error("fetch(): no known rdbms");
				throw new GNDBException(new IllegalStateException("PurchaseItemLister: fetch(): no known RDBMS"));
			}
		}
LOGGER.debug("fetch(): query: {}", query);

		try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
		{
			ResultSet rs = stmt.executeQuery(query);
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> vals = processResults_MySQL(rs);
				case hsqldb -> vals = processResults_hsqldb(rs);
				case MSSQLServer -> vals = processResults_MSSQLServer(rs);
				default -> {
					LOGGER.error("fetch(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PurchaseItemLister: fetch(): no known RDBMS"));
				}
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("fetch(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

		if (vals.isEmpty()) return Collections.emptyList();

		for (IPurchaseItem vx : vals)
		{
			MySQLCache.cachePurchaseItem.putIfAbsent(vx.getKey(), vx);
		}
		if (fetchAll) { MySQLCache.completePurchaseItem = true; }

		populateProduct(vals, fetchAll);

		populatePurchase(vals, fetchAll);

LOGGER.traceExit(log4jEntryMsg);
		return vals;
	}	// fetch()

    private String buildSQL_MySQL()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from purchaseitem as d ");
        query.append("left join (select * from comment where ownerType = \"PI\") as c ");
        query.append("on d.purchaseItemId = c.ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.purchaseId, d.productId, d.purchaseItemId, c.date");
        return query.toString();
    }   //  buildSQL_MySQL()

    private String buildSQL_hsqldb()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from purchaseitem as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'PI') as c ");
        query.append("on d.purchaseItemId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.purchaseId, d.productId, d.purchaseItemId, c_date");
        return query.toString();
    }   //  buildSQL_hsqldb()

    private String buildSQL_MSSQLServer()
    {
        StringBuilder query = new StringBuilder("select d.purchaseItemId as d_purchaseItemId, " +
                                                        "d.purchaseId as d_purchaseId, " +
                                                        "d.productId as d_productId, " +
                                                        "d.quantity as d_quantity, " +
                                                        "d.unit as d_unit, " +
                                                        "d.itemCost as d_itemCost, " +
                                                        "d.currency as d_currency, " +
                                                        "d.lastUpdated as d_lastUpdated, " +
                                                        "d.created as d_created," +
                                                        " c.* from purchaseitem as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'PI') as c ");
        query.append("on d.purchaseItemId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.purchaseId, d.productId, d.purchaseItemId, c_date");
        return query.toString();
    }   //  buildSQL_MSSQLServer()

    private void buildCommonSQL(StringBuilder query)
    {
        boolean first = true;
		if (usePurchaseItem.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(usePurchaseItem.getSQLClause("d"));
			first = false;
			usePurchaseItem.clear();
		}

		if (useProduct.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(useProduct.getSQLClause("d"));
			first = false;
			useProduct.clear();
		}

		if (usePurchase.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(usePurchase.getSQLClause("d"));
			first = false;
			usePurchase.clear();
		}

	}   //  buildCommonSQL()

	private List<IPurchaseItem> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<IPurchaseItem> tempList = new ArrayList<>();

		PurchaseItem item = null;

		while (rs.next())
		{
			int purchaseItemId = rs.getInt("d.purchaseItemId");
			int purchaseId = rs.getInt("d.purchaseId");
			int productId = rs.getInt("d.productId");
			BigDecimal quantity = rs.getBigDecimal("d.quantity");
			String unit = rs.getString("d.unit");
			BigDecimal itemCost = rs.getBigDecimal("d.itemCost");
			String currency = rs.getString("d.currency");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("purchaseItemId: {}, purchaseId: {}, productId: {}, quantity: {}, unit: {}, itemCost: {}, currency: {}, lastUpdated: {}, created: {}",
                purchaseItemId, purchaseId, productId, quantity, unit, itemCost, currency, lastUpdated, created);
			if (item != null && purchaseItemId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_MySQL(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("commentId"),
					rs.getInt("ownerId"),
					"PI",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): extra comment is: {}", comm);
				item = new PurchaseItem(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
					item = new PurchaseItem(purchaseItemId, purchaseId, productId, quantity, unit, itemCost, currency, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						purchaseItemId,
						"PI",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): first comment is: {}", comm);
					item = new PurchaseItem(purchaseItemId, purchaseId, productId, quantity, unit, itemCost, currency, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<IPurchaseItem> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<IPurchaseItem> tempList = new ArrayList<>();

		PurchaseItem item = null;

		while (rs.next())
		{
			int purchaseItemId = rs.getInt("purchaseitem.purchaseItemId");
			int purchaseId = rs.getInt("purchaseitem.purchaseId");
			int productId = rs.getInt("purchaseitem.productId");
			BigDecimal quantity = rs.getBigDecimal("purchaseitem.quantity");
			String unit = rs.getString("purchaseitem.unit");
			BigDecimal itemCost = rs.getBigDecimal("purchaseitem.itemCost");
			String currency = rs.getString("purchaseitem.currency");
			LocalDateTime lastUpdated = rs.getTimestamp("purchaseitem.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("purchaseitem.created").toLocalDateTime();
LOGGER.debug("purchaseItemId: {}, purchaseId: {}, productId: {}, quantity: {}, unit: {}, itemCost: {}, currency: {}, lastUpdated: {}, created: {}",
                purchaseItemId, purchaseId, productId, quantity, unit, itemCost, currency, lastUpdated, created);
			if (item != null && purchaseItemId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"PI",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new PurchaseItem(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new PurchaseItem(purchaseItemId, purchaseId, productId, quantity, unit, itemCost, currency, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						purchaseItemId,
						"PI",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new PurchaseItem(purchaseItemId, purchaseId, productId, quantity, unit, itemCost, currency, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<IPurchaseItem> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<IPurchaseItem> tempList = new ArrayList<>();

		PurchaseItem item = null;

		while (rs.next())
		{
			int purchaseItemId = rs.getInt("d_purchaseItemId");
			int purchaseId = rs.getInt("d_purchaseId");
			int productId = rs.getInt("d_productId");
			BigDecimal quantity = rs.getBigDecimal("d_quantity");
			String unit = rs.getString("d_unit");
			BigDecimal itemCost = rs.getBigDecimal("d_itemCost");
			String currency = rs.getString("d_currency");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("purchaseItemId: {}, purchaseId: {}, productId: {}, quantity: {}, unit: {}, itemCost: {}, currency: {}, lastUpdated: {}, created: {}",
                purchaseItemId, purchaseId, productId, quantity, unit, itemCost, currency, lastUpdated, created);
			if (item != null && purchaseItemId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"PI",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new PurchaseItem(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new PurchaseItem(purchaseItemId, purchaseId, productId, quantity, unit, itemCost, currency, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						purchaseItemId,
						"PI",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new PurchaseItem(purchaseItemId, purchaseId, productId, quantity, unit, itemCost, currency, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	/*
	*	Populate the parents slot(s)
	*/
	private void populateProduct(List<IPurchaseItem> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: NL
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populateProduct()");

		if (fetchAll)
		{
			new ProductLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			mapToInt(c -> ((PurchaseItem)c).getProductId()).filter(c -> c>0).distinct().
			filter(c -> !MySQLCache.cacheProduct.containsKey(c)).
			toArray();

		if (keys.length > 0)
		{
			new ProductLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	private void populatePurchase(List<IPurchaseItem> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: NL
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populatePurchase()");

		if (fetchAll)
		{
			new PurchaseLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			mapToInt(c -> ((PurchaseItem)c).getPurchaseId()).filter(c -> c>0).distinct().
			filter(c -> !MySQLCache.cachePurchase.containsKey(c)).
			toArray();

		if (keys.length > 0)
		{
			new PurchaseLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	void clear()
	{
		MySQLCache.cachePurchaseItem.clear();
		MySQLCache.completePurchaseItem = false;
	}

	/**
	*
	*Select only the PurchaseItem entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	PurchaseItemLister id(int... vals)
	{
		usePurchaseItem.id(vals);
		useWhere = useWhere || usePurchaseItem.isUsed();
		return this;
	}

	@Override
	public IPurchaseItemLister purchaseItem(IPurchaseItem... items)
	{
		usePurchaseItem.item(items);
		useWhere = useWhere || usePurchaseItem.isUsed();
		return this;
	}

	@Override
	public IPurchaseItemLister purchaseItem(List<IPurchaseItem> items)
	{
		usePurchaseItem.item(items);
		useWhere = useWhere || usePurchaseItem.isUsed();
		return this;
	}

	@Override
	public IPurchaseItemLister product(IProduct... items)
	{
		useProduct.item(items);
		useWhere = useWhere || useProduct.isUsed();
		return this;
	}

	@Override
	public IPurchaseItemLister product(List<IProduct> items)
	{
		useProduct.item(items);
		useWhere = useWhere || useProduct.isUsed();
		return this;
	}

	@Override
	public IPurchaseItemLister purchase(IPurchase... items)
	{
		LOGGER.debug("purchase(): items: {}", items);

		usePurchase.item(items);
		useWhere = useWhere || usePurchase.isUsed();
		return this;
	}

	@Override
	public IPurchaseItemLister purchase(List<IPurchase> items)
	{
		LOGGER.debug("purchase(): items: {}", items);

		usePurchase.item(items);
		useWhere = useWhere || usePurchase.isUsed();
		return this;
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		useWhere = false;
		fetch();

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (IPurchaseItem ihc : MySQLCache.cachePurchaseItem.values())
		{
			PurchaseItem hc = (PurchaseItem)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "PurchaseItem");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "PurchaseItem.json"), false)))
		{
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJson(): IOException", ex);
		}
	}	// toJson

}
