/*
 * Copyright (C) 2018, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Support setting 'watch for' items on PIs
			Use FlagHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import java.beans.PropertyChangeListener;

import uk.co.gardennotebook.spi.*;

import uk.co.gardennotebook.util.SimpleMoney;

import java.time.LocalDateTime;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;

//import java.time.*;
import java.math.BigDecimal;

import java.util.Optional;

import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonArray;
import jakarta.json.JsonObjectBuilder;
import jakarta.json.JsonObject;

/**
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/

final class PurchaseItem implements IPurchaseItem
{
	private final FlagHandler<IPurchaseItem> flagHandler;
	{
		flagHandler = new FlagHandler<>(this);
	}
	private final int id;
	private final int purchaseId;
	private final int productId;
	private final BigDecimal quantity;
	private final String unit;
	private final BigDecimal itemCost;

	/*
	*	ISO 4217 standard currency code (GBP, USD, EUR, etc).  Null means the local currency.
	*/
	private final String currency;
	private final LocalDateTime lastUpdated;
	private final LocalDateTime created;
	private final List<Comment> commentsList;

	/**
	*	Build an immutable PurchaseItem entry one field at a time
	*/
	PurchaseItem(
		final int id,
		final int purchaseId,
		final int productId,
		final BigDecimal quantity,
		final String unit,
		final BigDecimal itemCost,
		final String currency,
		final LocalDateTime lastUpdated,
		final LocalDateTime created,
		final Comment... comments)
	{
		this.id = id;
		this.purchaseId = purchaseId;
		this.productId = productId;
		this.quantity = quantity;
		this.unit = unit;
		this.itemCost = itemCost;
		this.currency = currency;
		this.lastUpdated = lastUpdated;
		this.created = created;
		if (comments != null && comments.length>0)
		{
			this.commentsList = new ArrayList<>(Arrays.asList(comments));
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}

	/**
	*	Build an immutable PurchaseItem entry cloning the given PurchaseItem entry but adding the comments list
	*/
	PurchaseItem(
		final PurchaseItem toCopy,
		final Comment... comments)
	{
		this(toCopy, Arrays.asList(comments));
	}

	/**
	*	Build an immutable PurchaseItem entry cloning the given PurchaseItem entry but adding the comments list
	*/
	PurchaseItem(
		final PurchaseItem toCopy,
		final List<Comment> comments)
	{
		this.id = toCopy.id;
		this.purchaseId = toCopy.purchaseId;
		this.productId = toCopy.productId;
		this.quantity = toCopy.quantity;
		this.unit = toCopy.unit;
		this.itemCost = toCopy.itemCost;
		this.currency = toCopy.currency;
		this.lastUpdated = toCopy.lastUpdated;
		this.created = toCopy.created;
		if (comments != null && comments.size()>0)
		{
			if (toCopy.commentsList.size()>0)
			{
				// append new comments to previous list
				this.commentsList = new ArrayList<>(toCopy.commentsList);
				this.commentsList.addAll(comments);
			}
			else
			{	// no comments on original item
				this.commentsList = new ArrayList<>(comments);
			}
		}
		else
		{	// no new comments to add
			this.commentsList = toCopy.commentsList;
		}
	}

	/**
	*	Build an immutable PurchaseItem entry from a JSON dump.
	*	The dumped object must be complete (all non-nullable fields must have values) except
	*	the id field can be null or absent to indicate that this is a new item to be inserted.
	*
	*	@param	json	a JsonObject holding all the fields for a full initialisation.
	*/
	PurchaseItem(JsonObject json)
	{
		this.id = json.getInt("id", -1);
		this.purchaseId = json.getInt("purchaseId");
		this.productId = json.getInt("productId");
		if (json.containsKey("quantity") && !json.isNull("quantity"))
		{
			this.quantity = json.getJsonNumber("quantity").bigDecimalValue();
		}
		else
		{
			this.quantity = null;
		}

		if (json.containsKey("unit") && !json.isNull("unit"))
		{
			this.unit = json.getString("unit");
		}
		else
		{
			this.unit = null;
		}

		if (json.containsKey("itemCost") && !json.isNull("itemCost"))
		{
			this.itemCost = json.getJsonNumber("itemCost").bigDecimalValue();
		}
		else
		{
			this.itemCost = null;
		}

		if (json.containsKey("currency") && !json.isNull("currency"))
		{
			this.currency = json.getString("currency");
		}
		else
		{
			this.currency = null;
		}

		this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
		this.created = LocalDateTime.parse(json.getString("created"));
		JsonArray jsonComments = json.getJsonArray("comments");
		if (jsonComments != null && !jsonComments.isEmpty())
		{// there is probably only one comment
			this.commentsList = new ArrayList<>(jsonComments.size());
			for (JsonObject ct : jsonComments.getValuesAs(JsonObject.class))
			{
				this.commentsList.add(new Comment(ct));
			}
		}
		else
		{
			this.commentsList = Collections.emptyList();
		}
	}	//constructor from JSON

	int getId()
	{
		return id;
	}
	@Override
	public Integer getKey()
	{
		return id;
	}

	@Override
	public NotebookEntryType getType()
	{
		return NotebookEntryType.PURCHASEITEM;
	}

	int getPurchaseId()
	{
		return purchaseId;
	}
	@Override
	public IPurchase getPurchase()
	{
		return MySQLCache.cachePurchase.get(purchaseId);
	}

	int getProductId()
	{
		return productId;
	}
	@Override
	public IProduct getProduct()
	{
		return MySQLCache.cacheProduct.get(productId);
	}

	@Override
	public Optional<BigDecimal> getQuantity()
	{
		return Optional.ofNullable(quantity);
	}

	@Override
	public Optional<String> getUnit()
	{
		return Optional.ofNullable(unit);
	}

	@Override
	public Optional<BigDecimal> getItemCost()
	{
		return Optional.ofNullable(itemCost);
	}

	@Override
	public Optional<String> getCurrency()
	{
		return Optional.ofNullable(currency);
	}

	@Override
	public LocalDateTime getLastUpdated()
	{
		return lastUpdated;
	}

	@Override
	public LocalDateTime getCreated()
	{
		return created;
	}

    @Override
	public SimpleMoney getItemPrice()
	{
		return new SimpleMoney(currency, itemCost);
	}

	@Override
	public boolean sameAs(INotebookEntry other)
	{
		if (other == null || other.getType() != this.getType())
		{
			return false;
		}
		return other.getKey().equals(this.getKey());
	}

	@Override
	public IToDoListLister getToDoList()
	{
		return new ToDoListLister().purchaseItem(this);
	}

	@Override
	public IReminderLister getReminder()
	{
		return new ReminderLister().purchaseItem(this);
	}

	@Override
	public List<IComment> getComments() {
		return new ArrayList<>(this.commentsList);
	}

	JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
	{
		JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
		jsonBuilder.add("id", id);
		jsonBuilder.add("purchaseId", purchaseId);
		jsonBuilder.add("productId", productId);
		if (quantity != null)
		{
			jsonBuilder.add("quantity", quantity);
		}
		else
		{
			jsonBuilder.addNull("quantity");
		}
		if (unit != null)
		{
			jsonBuilder.add("unit", unit);
		}
		else
		{
			jsonBuilder.addNull("unit");
		}
		if (itemCost != null)
		{
			jsonBuilder.add("itemCost", itemCost);
		}
		else
		{
			jsonBuilder.addNull("itemCost");
		}
		if (currency != null)
		{
			jsonBuilder.add("currency", currency);
		}
		else
		{
			jsonBuilder.addNull("currency");
		}
		jsonBuilder.add("lastUpdated", lastUpdated.toString());
		jsonBuilder.add("created", created.toString());
		if (commentsList != null && !commentsList.isEmpty())
		{// no point writing an empty comments array (the loaders handle this)
			JsonArrayBuilder jsonComments = jsonFactory.createArrayBuilder();
			for (Comment ct : commentsList)
			{
				jsonComments.add(ct.toJson(jsonFactory));
			}
			jsonBuilder.add("comments", jsonComments);
		}
		jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "PurchaseItem");
		return jsonBuilder;
	}	//	toJson

	@Override
	public void addPropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		flagHandler.addPropertyChangeListener(propertyName, listener);
	}

	@Override
	public void removePropertyChangeListener(final String propertyName, final PropertyChangeListener listener)
	{
		flagHandler.removePropertyChangeListener(propertyName, listener);
	}

	@Override
	public void flagDeleted()
	{
		flagHandler.flagDeleted();
	}

	@Override
	public void flagReplaced(final IPurchaseItem newValue)
	{
		flagHandler.flagReplaced(newValue, newValue::addPropertyChangeListener);
	}

	@Override
	public void flagChildDeleted(final IToDoList child) {
		flagHandler.flagChildDeleted("ToDoList", child);
	}

	@Override
	public void flagChildAdded(final IToDoList child) {
		flagHandler.flagChildAdded("ToDoList", child);
	}

	@Override
	public void flagChildDeleted(final IReminder child) {
		flagHandler.flagChildDeleted("Reminder", child);
	}

	@Override
	public void flagChildAdded(final IReminder child) {
		flagHandler.flagChildAdded("Reminder", child);
	}


	@Override
	public String toString() {
		return "PurchaseItem: " + "id: " + id + ", " +
				"purchaseId: " + purchaseId + ", " +
				"productId: " + productId + ", " +
				"quantity: " + quantity + ", " +
				"unit: " + unit + ", " +
				"itemCost: " + itemCost + ", " +
				"currency: " + currency + ", " +
				"lastUpdated: " + lastUpdated + ", " +
				"created: " + created;
	}

}
