/*
 * Copyright (C) 2018, 2019, 2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.3.0   Retrieve generated keys properly!
    2.4.0   Support MS SQLServer
    3.0.0	Use DBCommentHandler
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import uk.co.gardennotebook.util.StoryLineTree;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import org.apache.logging.log4j.Level;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.0.0
*	@since	1.0
*/
final class ProductBrandBuilder implements IProductBrandBuilder
{
	private static final Logger LOGGER = LogManager.getLogger();

	private IProductBrand oldInstance = null;

	private final boolean newInstance;

	private boolean changedAncestor = false;

	private int id;

	/*
	*	If this Brand is a retailer's 'own brand'.
	*/
	private Integer ownBrandRetailerId;
	private boolean changedOwnBrandRetailerId = false;
	private String name;
	private boolean changedName = false;
	private String description;
	private boolean changedDescription = false;
	private LocalDateTime lastUpdated;
	private LocalDateTime created;
	private boolean somethingChanged = false;

	private boolean changedComments = false;
	private DBCommentHandler commentHandler;	//	compiler will not allow declaration as final

	/**
	*	constructor to use for a new entry
	*/
	ProductBrandBuilder()
	{
		this(null);
	}

	/**
	*	constructor to use to edit or delete an existing entry
	*
	*	@param	oldVal	the existing item to modify or delete; if null a new entry will be created
	*/
	ProductBrandBuilder(final IProductBrand oldVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("constructor(): oldVal={}", oldVal);
		if (oldVal == null || oldVal.getKey() == null)
		{
			newInstance = true;
			oldInstance = null;
			this.id = -1;
			commentHandler = new DBCommentHandler(NotebookEntryType.PRODUCTBRAND);
			return;
		}

		newInstance = false;
		oldInstance = oldVal;

		ProductBrand baseObj;
		if (oldVal instanceof ProductBrand)
		{
			baseObj = (ProductBrand)oldVal;
			this.id = baseObj.getId();
			this.ownBrandRetailerId = baseObj.getOwnBrandRetailerId();
			this.name = baseObj.getName();
			this.description = baseObj.getDescription().orElse(null);
			this.lastUpdated = baseObj.getLastUpdated();
			this.created = baseObj.getCreated();
		}
		else
		{
			Object ky = oldVal.getKey();
			if (ky == null) return;
			this.id = (Integer)ky;
			ky = oldVal.getRetailer().orElse(null);
			if (ky == null)
			{
				this.ownBrandRetailerId = null;
			}
			else
			{
				this.ownBrandRetailerId = ((IRetailer)ky).getKey();
			}
			this.name = oldVal.getName();
			this.description = oldVal.getDescription().orElse(null);
			this.lastUpdated = oldVal.getLastUpdated();
			this.created = oldVal.getCreated();
		}
		commentHandler = new DBCommentHandler(NotebookEntryType.PRODUCTBRAND, this.id);
		LOGGER.traceExit();
	}	//	constructor()

	/**
	*	give the (new) value of ownBrandRetailerId
	*
	*	@param	newVal	the new value
	*	@return	this Builder
	*/
	IProductBrandBuilder ownBrandRetailerId(final int newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("ownBrandRetailerId(): oldVal={}, newVal={}", this.ownBrandRetailerId, newVal);
		if (this.ownBrandRetailerId == newVal) return this;
		this.ownBrandRetailerId = newVal;
		changedOwnBrandRetailerId = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}
	@Override
	public IProductBrandBuilder retailer(final IRetailer newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("retailer(): oldVal={}, newVal={}", this.ownBrandRetailerId, newVal);
		if ((newVal == null) && (this.ownBrandRetailerId == null)) return this;
		if ((newVal != null) && (this.ownBrandRetailerId != null) && (this.ownBrandRetailerId.equals(newVal.getKey()))) return this;
		if (newVal == null)
		{
			this.ownBrandRetailerId = null;
		}
		else
		{	//	non-null value
			this.ownBrandRetailerId = newVal.getKey();
		}
		changedOwnBrandRetailerId = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public boolean isNameDuplicate(final String newVal) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("isNameDuplicate({})", newVal);
        
        if (newVal == null || newVal.isBlank())
            return false;
        
        if (!newInstance && !changedName)
            return false;

        if (!newInstance && newVal.equals(oldInstance.getName()))
            return false;

		boolean duplicateFound = false;
		String query;
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = "select exists (select 1 from productbrand where name = ? ) as numvals";
			case hsqldb -> query = "select exists (select 1 from productbrand where name = ? ) as numvals from (values(99))";
			case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from productbrand where name = ? ) THEN 1 ELSE 0 END as numvals";
			default -> {
				LOGGER.error("isNameDuplicate(): no known rdbms");
				throw new GNDBException(new IllegalStateException("ProductBrandBuilder: isNameDuplicate(): no known RDBMS"));
			}
		}
LOGGER.debug("isNameDuplicate(): query: {}", query);
		try (	Connection conn = DBConnection.getConnection();
				PreparedStatement stmt = conn.prepareStatement(query); )
		{
			stmt.setString(1, newVal);
			ResultSet rs = stmt.executeQuery();
            rs.next();
            duplicateFound = rs.getBoolean("numvals");
LOGGER.debug("isNameDuplicate(): result: {}", duplicateFound);
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("isNameDuplicate(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
		return LOGGER.traceExit(log4jEntryMsg, duplicateFound);
	}	// isNameDuplicate()

	@Override
	public IProductBrandBuilder name(final String newVal) throws IllegalArgumentException, GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("name(): oldVal={}, newVal={}", this.name, newVal);

		if (newVal == null) return this;
		if (newVal.equals(this.name)) return this;
		this.name = newVal;
		changedName = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IProductBrandBuilder description(final String newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("description(): oldVal={}, newVal={}", this.description, newVal);

		if (newVal == null && this.description == null) return this;
		if (newVal != null && this.description != null && newVal.equals(this.description)) return this;
		this.description = newVal;
		changedDescription = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IProductBrandBuilder addComment(final String... newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment[array]()");

		commentHandler.addComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit("addComment");
		return this;
	}

	@Override
	public IProductBrandBuilder addComment(final List<String> newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment<list>()");

		commentHandler.addComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit("addComment");
		return this;
	}

	/**
	*	remove a comment from this item
	*
	*	@param	newVals	the comment to remove.  If the comment does not exist, this is a null-op
	*	@return	 this Builder
	*/
	IProductBrandBuilder deleteComment(int... newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

		commentHandler.deleteComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IProductBrandBuilder deleteComment(final IComment... newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

		commentHandler.deleteComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IProductBrandBuilder deleteComment(final List<IComment> vals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

		commentHandler.deleteComment(vals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IProductBrandBuilder changeComment(final IComment base, final String comment)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

		commentHandler.changeComment(base, comment);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IProductBrandBuilder changeComment(final IComment base, final LocalDate date)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

		commentHandler.changeComment(base, date);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IProductBrandBuilder changeComment(final IComment base, final LocalDate date, final String comment)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

		commentHandler.changeComment(base, date, comment);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IProductBrand save() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("save(): somethingChanged: {}, newInstance: {}, changedComments: {}, changedAncestor: {}",
                                                                somethingChanged, newInstance, changedComments, changedAncestor);

		if (!somethingChanged && !newInstance && !changedComments && !changedAncestor)
		{
			LOGGER.traceExit("nothing changed");
			return MySQLCache.cacheProductBrand.get(this.id);
		}

		if (newInstance)
		{
			doInsert();
		}
		else if (somethingChanged)
		{
			doUpdate();
		}

		if (changedComments)
		{
			commentHandler.setParentId(this.id);
			commentHandler.save();
		}

// mark cache as dirty
		if (!newInstance &&(somethingChanged || changedComments))
		{
			MySQLCache.cacheProductBrand.remove(this.id);
		}
// populate the cache
		new ProductBrandLister().id(this.id).fetch();
		IProductBrand newValue = MySQLCache.cacheProductBrand.get(this.id);
		if (oldInstance != null)
		{
			oldInstance.flagReplaced(newValue);
		}

	//	tell any parent beans the child list has mutated
	//	only additions and deletions matter, other changes will be reflected through the child bean
		if (newInstance)
		{
			if (changedOwnBrandRetailerId && (ownBrandRetailerId != null) )
			{
				MySQLCache.cacheRetailer.get(ownBrandRetailerId).flagChildAdded(newValue);
			}
		}
		else
		{	//	updated
			if (changedOwnBrandRetailerId)
			{
				if (oldInstance != null)
				{
					oldInstance.getRetailer().ifPresent(item -> MySQLCache.cacheRetailer.get(item.getKey()).flagChildDeleted(oldInstance) );
				}
				newValue.getRetailer().ifPresent(item -> MySQLCache.cacheRetailer.get(item.getKey()).flagChildAdded(newValue) );
			}
		}

		//	stop multiple saves!
		oldInstance = null;

		somethingChanged = false;
		changedComments = false;
		changedAncestor = false;
		changedOwnBrandRetailerId = false;
		changedName = false;
		changedDescription = false;

		LOGGER.traceExit(log4jEntryMsg);
		return newValue;
	}	//	save()

	@Override
	public boolean needSave()
	{
		return somethingChanged || changedComments || changedAncestor;
	}	// needSave()

	@Override
	public boolean canSave()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("canSave(): newInstance={}", newInstance);

		if (!needSave())
		{//	save() will be a null-op but that's OK
			return true;
		}
		if (this.name == null ||
			this.name.isEmpty() )
		{
			LOGGER.debug("name not set");
			return false;
		}
        try {
            if (isNameDuplicate(this.name)) 
            {
                LOGGER.debug("ProductBrandBuilder: canSave(): {} already exists", this.name);
                return false;
            }
        } catch (GNDBException ex) {
			throw LOGGER.throwing(Level.DEBUG, new IllegalArgumentException("PlantSpeciesBuilder: commonName: " +this.name+" already exists"));
        }
		return true;
	}	// canSave()

	@Override
	public boolean canDelete() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("canDelete(): newInstance={}", newInstance);

		if (newInstance) return LOGGER.traceExit(log4jEntryMsg, false);

		boolean  readValue = false;
		try (	Connection conn = DBConnection.getConnection();
				Statement stmt = conn.createStatement();	)
		{
            String query;
            ResultSet rs;
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from product where productBrandId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from product where productBrandId = " + this.id + ") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from product where productBrandId = " + this.id + ") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("ProductBrandBuilder: canDelete(): no known RDBMS"));
				}
			}
LOGGER.debug("canDelete(): query: {}", query);
			rs = stmt.executeQuery(query);
            rs.next();
            readValue = readValue || rs.getBoolean("fred");
            rs.close();
LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("canDelete(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
		return LOGGER.traceExit(log4jEntryMsg, !readValue);
	}	// canDelete()

	@Override
	public void delete() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("delete(): newInstance={}", newInstance);

		if (newInstance) return;
		if (!canDelete()) return;

		int res = 0;
		String query = "delete from productbrand where productBrandId = " + this.id;
LOGGER.debug("delete(): query: {}", query);
		try (	Connection conn = DBConnection.getConnection();
				Statement stmt = conn.createStatement();	)
		{
			res = stmt.executeUpdate(query);
LOGGER.debug("delete(): result: {}", res);
			// tidy up dependencies with nullable keys
			if (res > 0) {
				query = "delete from comment where ownerId = " + this.id + " and ownerType = 'BR'";
				int res2 = stmt.executeUpdate(query);
LOGGER.debug("delete() comments: result: {}", res2);
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("delete(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
		if (res > 0)
		{
			oldInstance.flagDeleted();
			MySQLCache.cacheProductBrand.remove(this.id);
	//	tell any parent beans the child list has mutated
	//	only additions and deletions matter, other changes will be reflected through the child bean
			oldInstance.getRetailer().ifPresent(item -> MySQLCache.cacheRetailer.get(item.getKey()).flagChildDeleted(oldInstance) );
		}
		oldInstance = null;
LOGGER.traceExit(log4jEntryMsg);
	}	// delete()

	private void doUpdate() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("doUpdate(): newInstance={}, somethingChanged={}", newInstance, somethingChanged);

		if (newInstance) return;
		if (!somethingChanged) return;
		StringBuilder query = new StringBuilder("update productbrand set ");
		if (changedOwnBrandRetailerId)
		{
			query.append("ownBrandRetailerId = ?, ");
		}

		if (changedName)
		{
			query.append("name = ?, ");
		}

		if (changedDescription)
		{
			query.append("description = ?, ");
		}

		query.delete(query.length()-2, query.length());
		query.append(" where productBrandId = ").append(this.id);
LOGGER.debug("doUpdate(): query={} ", query.toString());
		try (	Connection conn = DBConnection.getConnection();
				PreparedStatement stmt = conn.prepareStatement(query.toString());	)
		{
			int paramIx = 1;
			if (changedOwnBrandRetailerId)
			{
				if (this.ownBrandRetailerId == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.INTEGER);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.ownBrandRetailerId);
					stmt.setInt(paramIx++, this.ownBrandRetailerId);
				}
			}

			if (changedName)
			{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.name);
				stmt.setString(paramIx++, this.name);
			}

			if (changedDescription)
			{
				if (this.description == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.description);
					stmt.setString(paramIx++, this.description);
				}
			}

			stmt.executeUpdate();

		}catch (SQLException ex) {
			LOGGER.error("doUpdate(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
LOGGER.traceExit(log4jEntryMsg);
	}	// doUpdate

	private void doInsert() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("doInsert(): newInstance={}", newInstance);

		if (!newInstance) return;
		if (!canSave())
		{
			throw LOGGER.throwing(Level.ERROR, new IllegalStateException("doInsert(): save criteria not met"));
		}
		if (!this.changedName)
		{
			throw LOGGER.throwing(Level.ERROR, new IllegalStateException("ProductBrandBuilder: doInsert(): name unspecified"));
		}

		StringBuilder query = new StringBuilder("insert into productbrand (");
		query.append("name, ");
		if (changedOwnBrandRetailerId)
		{
			query.append("ownBrandRetailerId, ");
		}

		if (changedDescription)
		{
			query.append("description, ");
		}

		query.replace(query.length()-2, query.length(), ") values (");
		query.append("?, ");
		if (changedOwnBrandRetailerId)
		{
			query.append("?, ");
		}

		if (changedDescription)
		{
			query.append("?, ");
		}

		query.replace(query.length()-2, query.length(), ")");
LOGGER.debug("doInsert(): query={}", query.toString());

		try (	Connection conn = DBConnection.getConnection();
				PreparedStatement stmt = conn.prepareStatement(query.toString(), Statement.RETURN_GENERATED_KEYS); )
		{
			int paramIx = 1;
LOGGER.debug("doInsert(): param {}={}", paramIx, this.name);
			stmt.setString(paramIx++, this.name);
			if (changedOwnBrandRetailerId) {
				if (this.ownBrandRetailerId == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.INTEGER);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.ownBrandRetailerId);
					stmt.setInt(paramIx++, this.ownBrandRetailerId);
				}
			}

			if (changedDescription) {
				if (this.description == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.description);
					stmt.setString(paramIx++, this.description);
				}
			}

			stmt.executeUpdate();

			ResultSet rs = stmt.getGeneratedKeys();
			rs.next();
			int newId = rs.getInt(1);
LOGGER.debug("doInsert(): newId: {}", newId);
			this.id = newId;
		}catch (SQLException ex) {
			LOGGER.error("doInsert(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

LOGGER.traceExit(log4jEntryMsg);
	}	// doInsert

	/**
	*	Used to restore a table from a JSON dump.
	*	The ProductBrandBuilder(oldVal) constructor MUST be used with a ProductBrand object created using its JSON constructor.
	*	All fields (including timestamps) are written to the database, no checks are made.
	*
	*	@param	newVal	A ProductBrand object created from a JSON object written as a DUMP.
*
*	@throws	GNDBException	If the underlying MySQL database throws SQLException it is translated to this.
*				The causal SQLException can be retrieved by <code>getCause()</code>
	*/
	void doJsonInsert(ProductBrand newVal) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("doJsonInsert(): newVal={}", newVal);

		int currId = newVal.getId();
		StringBuilder query = new StringBuilder("insert into productbrand (");
		if (newVal.getId() > 0)
		{//this forces the value of the id field.  The >0 test is a bodge.
			query.append("productBrandId, ");
		}
		query.append("ownBrandRetailerId, ");
		query.append("name, ");
		query.append("description, ");
		query.append("lastUpdated, ");
		query.append("created, ");
//		query.replace(query.length()-2, query.length(), ") values (");
		query.replace(query.length()-2, query.length(), ")");
        if (newVal.getId() > 0 && DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.hsqldb)
        {
            query.append(" overriding system value ");
        }
        query.append(" values (");
		if (newVal.getId() > 0)
		{//this forces the value of the id field.  The >0 test is a bodge.
			query.append(newVal.getId()).append(", ");
		}
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.replace(query.length()-2, query.length(), ")");
LOGGER.debug("doJsonInsert(): query={}", query.toString());

		try (	Connection conn = DBConnection.getConnection();
				PreparedStatement stmt = conn.prepareStatement(query.toString(), Statement.RETURN_GENERATED_KEYS); )
		{
			int paramIx = 1;
			if (newVal.getOwnBrandRetailerId() == null)
			{
LOGGER.debug("doJsonInsert(): param {} null", paramIx);
				stmt.setNull(paramIx++, java.sql.Types.INTEGER);
			}
			else
			{
LOGGER.debug("doJsonInsert(): param {}={}", paramIx, newVal.getOwnBrandRetailerId());
				stmt.setInt(paramIx++, newVal.getOwnBrandRetailerId());
			}

LOGGER.debug("doJsonInsert(): param {}={}", paramIx, newVal.getName());
			stmt.setString(paramIx++, newVal.getName());
			if (!newVal.getDescription().isPresent())
			{
LOGGER.debug("doJsonInsert(): param {} null", paramIx);
				stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
			}
			else
			{
LOGGER.debug("doJsonInsert(): param {}={}", paramIx, newVal.getDescription().get());
				stmt.setString(paramIx++, newVal.getDescription().get());
			}

LOGGER.debug("doJsonInsert(): param {}={}", paramIx, newVal.getLastUpdated());
			stmt.setTimestamp(paramIx++, Timestamp.valueOf(newVal.getLastUpdated()));
LOGGER.debug("doJsonInsert(): param {}={}", paramIx, newVal.getCreated());
			stmt.setTimestamp(paramIx++, Timestamp.valueOf(newVal.getCreated()));

            if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
            {
                conn.createStatement().execute("SET IDENTITY_INSERT productbrand ON");
            }
            
			stmt.executeUpdate();

			if (currId <= 0)
			{
                ResultSet rs = stmt.getGeneratedKeys();
				rs.next();
				currId = rs.getInt(1);
LOGGER.debug("doJsonInsert(): currId: {}", currId);
			}

            if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
            {
                conn.createStatement().execute("SET IDENTITY_INSERT productbrand OFF");
            }

            if (!newVal.getComments().isEmpty())
            {
                CommentBuilder cb = new CommentBuilder(NotebookEntryType.PRODUCTBRAND, currId);
                cb.doJsonInsert(newVal.getComments(), conn);
            }
		}catch (SQLException ex) {
			LOGGER.error("doJsonInsert(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

LOGGER.traceExit(log4jEntryMsg);
	}	// doJsonInsert

	@Override
	public boolean hasAncestor() throws GNDBException
	{
		//	this class does not participate in story lines
		return false;
	}	//	hasAncestor()

	@Override
	public StoryLineTree<? extends INotebookEntry> getAncestors() throws GNDBException
	{
		//	this class does not participate in story lines
		return StoryLineTree.emptyTree();
	}	//	getAncestors()

	@Override
	public IProductBrandBuilder ancestor(IPurchaseItem ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(PurchaseItem)

	@Override
	public IProductBrandBuilder ancestor(IGroundwork ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(Groundwork)

	@Override
	public IProductBrandBuilder ancestor(IAfflictionEvent ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(AfflictionEvent)

	@Override
	public IProductBrandBuilder ancestor(IHusbandry ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(Husbandry)

	@Override
	public IProductBrandBuilder ancestor(ISaleItem ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(SaleItem)

	@Override
	public boolean hasDescendant() throws GNDBException
	{
		//	this class does not participate in story lines
		return false;
	}	//	hasDescendant()

	@Override
	public StoryLineTree<? extends INotebookEntry> getDescendants() throws GNDBException
	{
		//	this class does not participate in story lines
		return StoryLineTree.emptyTree();
	}	//	getDescendants()

	@Override
	public boolean addDescendant(IPurchaseItem descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(PurchaseItem)

	@Override
	public boolean addDescendant(IGroundwork descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(Groundwork)

	@Override
	public boolean addDescendant(IAfflictionEvent descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(AfflictionEvent)

	@Override
	public boolean addDescendant(IHusbandry descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(Husbandry)

	@Override
	public boolean addDescendant(ISaleItem descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(SaleItem)

}

