/*
 * Copyright (C) 2018, 2019, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.time.LocalDateTime;
import java.util.*;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;
import jakarta.json.JsonObjectBuilder;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class PlantNoteLister implements IPlantNoteLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private final DBKeyHandler<IPlantNote> usePlantNote = new DBKeyHandler<>("plantNoteId");

	private final DBKeyHandler<IPlantSpecies> usePlantSpecies = new DBKeyHandler<>("plantSpeciesId");

	private final DBKeyHandler<IPlantVariety> usePlantVariety = new DBKeyHandler<>("plantVarietyId");

	private boolean useWhere = false;

	PlantNoteLister() {}

	@Override
	public List<IPlantNote> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		List<IPlantNote> vals = new ArrayList<>();
		boolean fetchAll = !useWhere;
        String query = "";
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = buildSQL_MySQL();
			case hsqldb -> query = buildSQL_hsqldb();
			case MSSQLServer -> query = buildSQL_MSSQLServer();
			default -> {
				LOGGER.debug("fetch(): no known rdbms");
				throw new GNDBException(new IllegalStateException("no known RDBMS"));
			}
		}
LOGGER.debug("fetch(): query: {}", query);

		try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
		{
			ResultSet rs = stmt.executeQuery(query);
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> vals = processResults_MySQL(rs);
				case hsqldb -> vals = processResults_hsqldb(rs);
				case MSSQLServer -> vals = processResults_MSSQLServer(rs);
				default -> {
					LOGGER.debug("fetch(): no known rdbms");
					throw new GNDBException(new IllegalStateException("no known RDBMS"));
				}
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("fetch(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

		if (vals.isEmpty()) return Collections.emptyList();

		for (IPlantNote vx : vals)
		{
			MySQLCache.cachePlantNote.putIfAbsent(vx.getKey(), vx);
		}
		if (fetchAll) { MySQLCache.completePlantNote = true; }

		populatePlantSpecies(vals, fetchAll);

		populatePlantVariety(vals, fetchAll);

LOGGER.traceExit(log4jEntryMsg);
		return vals;
	}	// fetch()

    private String buildSQL_MySQL()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from plantnote as d ");
        query.append("left join (select * from comment where ownerType = 'PN') as c ");
        query.append("on d.plantNoteId = c.ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.plantSpeciesId, LOWER(d.title), d.plantNoteId, c.date");
        return query.toString();
    }   //  buildSQL_MySQL()

    private String buildSQL_hsqldb()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from plantnote as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'PN') as c ");
        query.append("on d.plantNoteId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.plantSpeciesId, LOWER(d.title), d.plantNoteId, c_date");
        return query.toString();
    }   //  buildSQL_hsqldb()

    private String buildSQL_MSSQLServer()
    {
        StringBuilder query = new StringBuilder("select d.plantNoteId as d_plantNoteId, " +
                                                        "d.plantSpeciesId as d_plantSpeciesId, " +
                                                        "d.plantVarietyId as d_plantVarietyId, " +
                                                        "d.title as d_title, " +
                                                        "d.note as d_note, " +
                                                        "d.lastUpdated as d_lastUpdated, " +
                                                        "d.created as d_created," +
                                                        " c.* from plantnote as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'PN') as c ");
        query.append("on d.plantNoteId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.plantSpeciesId, LOWER(d.title), d.plantNoteId, c_date");
        return query.toString();
    }   //  buildSQL_MSSQLServer()

    private void buildCommonSQL(StringBuilder query)
    {
        boolean first = true;
		if (usePlantNote.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(usePlantNote.getSQLClause("d"));
			first = false;
			usePlantNote.clear();
		}

		if (this.usePlantSpecies.isUsed() || this.usePlantVariety.isUsed())
        {
            if (first)
                query.append(" where ");
            else
                query.append(" and ");

			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" ( ");

			if (this.usePlantSpecies.isUsed())
			{
                query.append(" ( ");
				query.append(usePlantSpecies.getSQLClause("d"));
                query.append(" and d.plantVarietyId is null )");
				first = false;
			}

			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" or ");
			if (this.usePlantVariety.isUsed())
			{
				query.append(usePlantVariety.getSQLClause("d"));
				first = false;
			}

			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" ) ");

			usePlantSpecies.clear();
			usePlantVariety.clear();
        }
    }   //  buildCommonSQL()

	private List<IPlantNote> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<IPlantNote> tempList = new ArrayList<>();

		PlantNote item = null;

		while (rs.next())
		{
			int plantNoteId = rs.getInt("d.plantNoteId");
			int plantSpeciesId = rs.getInt("d.plantSpeciesId");
			int tmp_plantVarietyId = rs.getInt("d.plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			String title = rs.getString("d.title");
			String note = rs.getString("d.note");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("plantNoteId: {}, plantSpeciesId: {}, plantVarietyId: {}, title: {}, lastUpdated: {}, created: {}", plantNoteId, plantSpeciesId, plantVarietyId, title, lastUpdated, created);
LOGGER.debug("note: {}", note);
			if (item != null && plantNoteId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c.commentId"),
					rs.getInt("c.ownerId"),
					"PN",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults(): extra comment is: {}", comm);
				item = new PlantNote(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
					item = new PlantNote(plantNoteId, plantSpeciesId, plantVarietyId, title, note, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						plantNoteId,
						"PN",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults(): first comment is: {}", comm);
					item = new PlantNote(plantNoteId, plantSpeciesId, plantVarietyId, title, note, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<IPlantNote> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<IPlantNote> tempList = new ArrayList<>();

		PlantNote item = null;

		while (rs.next())
		{
			int plantNoteId = rs.getInt("plantnote.plantNoteId");
			int plantSpeciesId = rs.getInt("plantnote.plantSpeciesId");
			int tmp_plantVarietyId = rs.getInt("plantnote.plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			String title = rs.getString("plantnote.title");
			String note = rs.getString("plantnote.note");
			LocalDateTime lastUpdated = rs.getTimestamp("plantnote.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("plantnote.created").toLocalDateTime();
LOGGER.debug("plantNoteId: {}, plantSpeciesId: {}, plantVarietyId: {}, title: {}, lastUpdated: {}, created: {}", plantNoteId, plantSpeciesId, plantVarietyId, title, lastUpdated, created);
LOGGER.debug("note: {}", note);
			if (item != null && plantNoteId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"PN",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults(): extra comment is: {}", comm);
				item = new PlantNote(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new PlantNote(plantNoteId, plantSpeciesId, plantVarietyId, title, note, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						plantNoteId,
						"PN",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults(): first comment is: {}", comm);
					item = new PlantNote(plantNoteId, plantSpeciesId, plantVarietyId, title, note, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<IPlantNote> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<IPlantNote> tempList = new ArrayList<>();

		PlantNote item = null;

		while (rs.next())
		{
			int plantNoteId = rs.getInt("d_plantNoteId");
			int plantSpeciesId = rs.getInt("d_plantSpeciesId");
			int tmp_plantVarietyId = rs.getInt("d_plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			String title = rs.getString("d_title");
			String note = rs.getString("d_note");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("plantNoteId: {}, plantSpeciesId: {}, plantVarietyId: {}, title: {}, lastUpdated: {}, created: {}", plantNoteId, plantSpeciesId, plantVarietyId, title, lastUpdated, created);
LOGGER.debug("note: {}", note);
			if (item != null && plantNoteId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"PN",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults(): extra comment is: {}", comm);
				item = new PlantNote(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new PlantNote(plantNoteId, plantSpeciesId, plantVarietyId, title, note, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						plantNoteId,
						"PN",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults(): first comment is: {}", comm);
					item = new PlantNote(plantNoteId, plantSpeciesId, plantVarietyId, title, note, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	/*
	*	Populate the parents slot(s)
	*/
	private void populatePlantSpecies(List<IPlantNote> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: TD
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populatePlantSpecies()");

		if (fetchAll)
		{
			new PlantSpeciesLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			mapToInt(c -> ((PlantNote)c).getPlantSpeciesId()).filter(c -> c>0).distinct().
			filter(c -> !MySQLCache.cachePlantSpecies.containsKey(c)).
			toArray();

		if (keys.length > 0)
		{
			new PlantSpeciesLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	private void populatePlantVariety(List<IPlantNote> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: NL
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populatePlantVariety()");

		if (fetchAll)
		{
			new PlantVarietyLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			map(c -> ((PlantNote)c).getPlantVarietyId()).filter(Objects::nonNull).
			mapToInt(Integer::intValue).filter(c -> c>0).distinct().
			toArray();

		if (keys.length > 0)
		{
			new PlantVarietyLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	void clear()
	{
		MySQLCache.cachePlantNote.clear();
		MySQLCache.completePlantNote = false;
	}

	/**
	*
	*Select only the PlantNote entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	PlantNoteLister id(int... vals)
	{
		usePlantNote.id(vals);
		useWhere = useWhere || usePlantNote.isUsed();
		return this;
	}

	@Override
	public IPlantNoteLister plantNote(IPlantNote... items)
	{
		usePlantNote.item(items);
		useWhere = useWhere || usePlantNote.isUsed();
		return this;
	}

	@Override
	public IPlantNoteLister plantNote(List<IPlantNote> items)
	{
		usePlantNote.item(items);
		useWhere = useWhere || usePlantNote.isUsed();
		return this;
	}

	@Override
	public IPlantNoteLister plantSpecies(IPlantSpecies... items)
	{
		usePlantSpecies.item(items);
		useWhere = useWhere || usePlantSpecies.isUsed();
		return this;
	}

	@Override
	public IPlantNoteLister plantSpecies(List<IPlantSpecies> items)
	{
		usePlantSpecies.item(items);
		useWhere = useWhere || usePlantSpecies.isUsed();
		return this;
	}

	@Override
	public IPlantNoteLister plantVariety(IPlantVariety... items)
	{
		usePlantVariety.item(items);
		useWhere = useWhere || usePlantVariety.isUsed();
		return this;
	}

	@Override
	public IPlantNoteLister plantVariety(List<IPlantVariety> items)
	{
		usePlantVariety.item(items);
		useWhere = useWhere || usePlantVariety.isUsed();
		return this;
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		useWhere = false;
		fetch();

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (IPlantNote ihc : MySQLCache.cachePlantNote.values())
		{
			PlantNote hc = (PlantNote)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "PlantNote");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "PlantNote.json"), false)))
		{
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJson(): IOException", ex);
		}
	}	// toJson

}
