/*
 * Copyright (C) 2018, 2019, 2021, 2022, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
	3.0.0	Support Locations and Quantity field
	3.0.1	Bug fix - used 'groundworkActivityId' but s.b. 'activityId' - which is an anomalous name!
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;
import jakarta.json.JsonObjectBuilder;

//import java.time.*;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class GroundworkLister implements IGroundworkLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private final DBKeyHandler<IGroundwork> useGroundwork = new DBKeyHandler<>("groundworkId");

	private final DBKeyHandler<IGroundworkActivity> useActivity = new DBKeyHandler<>("activityId");

	private final DBKeyHandler<IPlantSpecies> usePlantSpecies = new DBKeyHandler<>("plantSpeciesId");

	private final DBKeyHandler<IPlantVariety> usePlantVariety = new DBKeyHandler<>("plantVarietyId");

	private boolean forceNoPlant = false;

	private final DBKeyHandler<ILocation> useLocation = new DBKeyHandler<>("locationId");

	private boolean useFromDate = false;
	private LocalDate fromDate;
	private boolean useToDate = false;
	private LocalDate toDate;

	private boolean useWhere = false;

	GroundworkLister() {}

	@Override
	public List<IGroundwork> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		List<IGroundwork> vals = new ArrayList<>();
		boolean fetchAll = !useWhere;
        String query = "";
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = buildSQL_MySQL();
			case hsqldb -> query = buildSQL_hsqldb();
			case MSSQLServer -> query = buildSQL_MSSQLServer();
			default -> {
				LOGGER.debug("fetch(): no known rdbms");
				throw new GNDBException(new IllegalStateException("no known RDBMS"));
			}
		}
LOGGER.debug("fetch(): query: {}", query);

		try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
		{
			ResultSet rs = stmt.executeQuery(query);
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> vals = processResults_MySQL(rs);
				case hsqldb -> vals = processResults_hsqldb(rs);
				case MSSQLServer -> vals = processResults_MSSQLServer(rs);
				default -> {
					LOGGER.debug("fetch(): no known rdbms");
					throw new GNDBException(new IllegalStateException("no known RDBMS"));
				}
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("fetch(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

		if (vals.isEmpty()) return Collections.emptyList();

		for (IGroundwork vx : vals)
		{
			MySQLCache.cacheGroundwork.putIfAbsent(vx.getKey(), vx);
		}
		if (fetchAll) { MySQLCache.completeGroundwork = true; }

		populatePlantSpecies(vals, fetchAll);

		populateGroundworkActivity();

		populatePlantVariety(vals, fetchAll);

LOGGER.traceExit(log4jEntryMsg);
		return vals;
	}	// fetch()

    private String buildSQL_MySQL() 
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from groundwork as d ");
        query.append("left join (select * from comment where ownerType = 'GW') as c ");
        query.append("on d.groundWorkId = c.ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.activityId, d.groundWorkId, c.date");
        return query.toString();
    }   //  buildSQL_MySQL()

    private String buildSQL_hsqldb() 
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from groundwork as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'GW') as c ");
        query.append("on d.groundWorkId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.activityId, d.groundWorkId, c_date");
        return query.toString();
    }   //  buildSQL_hsqldb()

    private String buildSQL_MSSQLServer() 
    {
        StringBuilder query = new StringBuilder("select d.groundWorkId as d_groundWorkId, " +
                                                        "d.activityId as d_activityId, " +
                                                        "d.plantSpeciesId as d_plantSpeciesId, " +
                                                        "d.plantVarietyId as d_plantVarietyId, " +
														"d.locationId as d_locationId, " +
                                                        "d.date as d_date, " +
														"d.quantity as d_quantity, " +
                                                        "d.lastUpdated as d_lastUpdated, " +
                                                        "d.created as d_created," +
                                                        " c.* from groundwork as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'GW') as c ");
        query.append("on d.groundWorkId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.activityId, d.groundWorkId, c_date");
        return query.toString();
    }   //  buildSQL_MSSQLServer()

    private void buildCommonSQL(StringBuilder query)
    {
        boolean first = true;
		if (useGroundwork.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and ");
			query.append(useGroundwork.getSQLClause("d"));
			first = false;
			useGroundwork.clear();
		}

        if (forceNoPlant)
        {
            if (first)
                query.append(" where ");
            else
                query.append(" and ");

            query.append(" d.plantSpeciesId is null and d.plantVarietyId is null ");
            first = false;
        }
//        else if (this.usePlantSpeciesId || this.usePlantVarietyId)
 		else if (this.usePlantSpecies.isUsed() || this.usePlantVariety.isUsed())
		{
            if (first)
                query.append(" where ");
            else
                query.append(" and ");

			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" ( ");
			if (this.usePlantSpecies.isUsed())
			{
				query.append(usePlantSpecies.getSQLClause("d"));
				first = false;
			}
			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" or ");
			if (this.usePlantVariety.isUsed())
			{
				query.append(usePlantVariety.getSQLClause("d"));
				first = false;
			}
			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" ) ");
			usePlantSpecies.clear();
			usePlantVariety.clear();
		}

		if (useActivity.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and ");
			query.append(useActivity.getSQLClause("d"));
			first = false;
			useActivity.clear();
		}

		if (useLocation.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and ");
			query.append(useLocation.getSQLClause("d"));
			first = false;
			useLocation.clear();
		}

        if (this.useFromDate)
        {
            if (first) query.append(" where ");
            else query.append(" and ");

            query.append(" d.date >= '").append(this.fromDate).append("'");
            first = false;
            this.useFromDate = false;
        }
        if (this.useToDate)
        {
            if (first) query.append(" where ");
            else query.append(" and ");

            query.append(" d.date <= '").append(this.toDate).append("'");
        //                first = false;
        this.useToDate = false;
        }
    }   //  buildCommonSQL

	private List<IGroundwork> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<IGroundwork> tempList = new ArrayList<>();

		Groundwork item = null;

		while (rs.next())
		{
			int groundWorkId = rs.getInt("d.groundWorkId");
			int activityId = rs.getInt("d.activityId");
			int tmp_plantSpeciesId = rs.getInt("d.plantSpeciesId");
			Integer plantSpeciesId = rs.wasNull() ? null : tmp_plantSpeciesId;
			int tmp_plantVarietyId = rs.getInt("d.plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			int tmp_locationId = rs.getInt("d.locationId");
			Integer locationId = rs.wasNull() ? null : tmp_locationId;
			LocalDate date = rs.getDate("date").toLocalDate();
			String quantity = rs.getString("d.quantity");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("groundWorkId: {}, activityId: {}, plantSpeciesId: {}, plantVarietyId: {}, locationId: {}, date: {}, quantity: {}, lastUpdated: {}, created: {}",
                        groundWorkId, activityId, plantSpeciesId, plantVarietyId, locationId, date, quantity, lastUpdated, created);
			if (item != null && groundWorkId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_MySQL(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("commentId"),
					rs.getInt("ownerId"),
					"GW",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): extra comment is: {}", comm);
				item = new Groundwork(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
					item = new Groundwork(groundWorkId, activityId, plantSpeciesId, plantVarietyId, locationId, date, quantity, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						groundWorkId,
						"GW",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): first comment is: {}", comm);
					item = new Groundwork(groundWorkId, activityId, plantSpeciesId, plantVarietyId, locationId, date, quantity, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<IGroundwork> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<IGroundwork> tempList = new ArrayList<>();

		Groundwork item = null;

		while (rs.next())
		{
			int groundWorkId = rs.getInt("groundwork.groundWorkId");
			int activityId = rs.getInt("groundwork.activityId");
			int tmp_plantSpeciesId = rs.getInt("groundwork.plantSpeciesId");
			Integer plantSpeciesId = rs.wasNull() ? null : tmp_plantSpeciesId;
			int tmp_plantVarietyId = rs.getInt("groundwork.plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			int tmp_locationId = rs.getInt("groundwork.locationId");
			Integer locationId = rs.wasNull() ? null : tmp_locationId;
			LocalDate date = rs.getDate("date").toLocalDate();
			String quantity = rs.getString("groundwork.quantity");
			LocalDateTime lastUpdated = rs.getTimestamp("groundwork.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("groundwork.created").toLocalDateTime();
LOGGER.debug("groundWorkId: {}, activityId: {}, plantSpeciesId: {}, plantVarietyId: {}, locationId: {}, date: {}, quantity: {}, lastUpdated: {}, created: {}",
                        groundWorkId, activityId, plantSpeciesId, plantVarietyId, locationId, date, quantity, lastUpdated, created);
			if (item != null && groundWorkId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"GW",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new Groundwork(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new Groundwork(groundWorkId, activityId, plantSpeciesId, plantVarietyId, locationId, date, quantity, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						groundWorkId,
						"GW",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new Groundwork(groundWorkId, activityId, plantSpeciesId, plantVarietyId, locationId, date, quantity, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<IGroundwork> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<IGroundwork> tempList = new ArrayList<>();

		Groundwork item = null;

		while (rs.next())
		{
			int groundWorkId = rs.getInt("d_groundWorkId");
			int activityId = rs.getInt("d_activityId");
			int tmp_plantSpeciesId = rs.getInt("d_plantSpeciesId");
			Integer plantSpeciesId = rs.wasNull() ? null : tmp_plantSpeciesId;
			int tmp_plantVarietyId = rs.getInt("d_plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			int tmp_locationId = rs.getInt("d_locationId");
			Integer locationId = rs.wasNull() ? null : tmp_locationId;
			LocalDate date = rs.getDate("d_date").toLocalDate();
			String quantity = rs.getString("d_quantity");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("groundWorkId: {}, activityId: {}, plantSpeciesId: {}, plantVarietyId: {}, date: {}, lastUpdated: {}, created: {}",
                        groundWorkId, activityId, plantSpeciesId, plantVarietyId, date, lastUpdated, created);
			if (item != null && groundWorkId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"GW",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new Groundwork(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new Groundwork(groundWorkId, activityId, plantSpeciesId, plantVarietyId, locationId, date, quantity, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						groundWorkId,
						"GW",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new Groundwork(groundWorkId, activityId, plantSpeciesId, plantVarietyId, locationId, date, quantity, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	/*
	*	Populate the parents slot(s)
	*/
	private void populatePlantSpecies(List<IGroundwork> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: TD
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populatePlantSpecies()");

		if (fetchAll)
		{
			new PlantSpeciesLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			map(c -> ((Groundwork)c).getPlantSpeciesId()).filter(Objects::nonNull).
			mapToInt(Integer::intValue).filter(c -> c>0).distinct().
			toArray();

		if (keys.length > 0)
		{
			new PlantSpeciesLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	private void populateGroundworkActivity() throws GNDBException
	{
// parent table type: TF
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populateGroundworkActivity()");

// just make sure everything is cached
		new GroundworkActivityLister().load();
LOGGER.traceExit(log4jEntryMsg);
	}

	private void populatePlantVariety(List<IGroundwork> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: NL
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populatePlantVariety()");

		if (fetchAll)
		{
			new PlantVarietyLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			map(c -> ((Groundwork)c).getPlantVarietyId()).filter(Objects::nonNull).
			mapToInt(Integer::intValue).filter(c -> c>0).distinct().
			toArray();

		if (keys.length > 0)
		{
			new PlantVarietyLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	void clear()
	{
		MySQLCache.cacheGroundwork.clear();
		MySQLCache.completeGroundwork = false;
	}

	/**
	*
	*Select only the Groundwork entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	GroundworkLister id(int... vals)
	{
		useGroundwork.id(vals);
		useWhere = useWhere || useGroundwork.isUsed();
		return this;
	}

	@Override
	public IGroundworkLister groundwork(IGroundwork... items)
	{
		useGroundwork.item(items);
		useWhere = useWhere || useGroundwork.isUsed();
		return this;
	}

	@Override
	public IGroundworkLister groundwork(List<IGroundwork> items)
	{
		useGroundwork.item(items);
		useWhere = useWhere || useGroundwork.isUsed();
		return this;
	}

	@Override
	public IGroundworkLister plantSpecies(IPlantSpecies... items)
	{
		usePlantSpecies.item(items);
		useWhere = useWhere || usePlantSpecies.isUsed();
		return this;
	}

	@Override
	public IGroundworkLister plantSpecies(List<IPlantSpecies> items)
	{
		usePlantSpecies.item(items);
		useWhere = useWhere || usePlantSpecies.isUsed();
		return this;
	}

	@Override
	public IGroundworkLister groundworkActivity(IGroundworkActivity... items)
	{
		useActivity.item(items);
		useWhere = useWhere || useActivity.isUsed();
		return this;
	}

	@Override
	public IGroundworkLister groundworkActivity(List<IGroundworkActivity> items)
	{
		useActivity.item(items);
		useWhere = useWhere || useActivity.isUsed();
		return this;
	}

	@Override
	public IGroundworkLister plantVariety(IPlantVariety... items)
	{
		usePlantVariety.item(items);
		useWhere = useWhere || usePlantVariety.isUsed();
		return this;
	}

	@Override
	public IGroundworkLister plantVariety(List<IPlantVariety> items)
	{
		usePlantVariety.item(items);
		useWhere = useWhere || usePlantVariety.isUsed();
		return this;
	}

	@Override
	public IGroundworkLister requireNoPlant()
	{
		forceNoPlant = true;
		useWhere = true;
		return this;
	}

	@Override
	public IGroundworkLister location(ILocation... items)
	{
		useLocation.item(items);
		useWhere = useWhere || useLocation.isUsed();
		return this;
	}

	@Override
	public IGroundworkLister location(List<ILocation> items)
	{
		useLocation.item(items);
		useWhere = useWhere || useLocation.isUsed();
		return this;
	}

	@Override
	public IGroundworkLister fromDate(LocalDate item)
	{
		if (item == null) return this;
		this.fromDate = item;
		this.useFromDate = true;
		this.useWhere = true;
		return this;
	}

	@Override
	public IGroundworkLister toDate(LocalDate item)
	{
		if (item == null) return this;
		this.toDate = item;
		this.useToDate = true;
		this.useWhere = true;
		return this;
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		useWhere = false;
		fetch();

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (IGroundwork ihc : MySQLCache.cacheGroundwork.values())
		{
			Groundwork hc = (Groundwork)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "Groundwork");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "Groundwork.json"), false)))
		{
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJson(): IOException", ex);
		}
	}	// toJson

}
