/*
 *
 *  Copyright (C) 2021, 2023 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

/*
    Change log
	3.1.0	Use jakarta implementation of JSON
*/

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import jakarta.json.*;
import java.beans.PropertyChangeListener;
import java.time.LocalDateTime;
import java.util.*;

/**
 *{@inheritDoc}
 *
 *	@author	Andy Gegg
 *	@version	3.1.0
 *	@since	3.0.0
 */

final class CroppingActual implements ICroppingActual
{
    private final FlagHandler<ICroppingActual> flagHandler;
    {
        flagHandler = new FlagHandler<>(this);
    }

    private final int id;
    private final int croppingPlanId;
    private final int plantSpeciesId;
    private final Integer plantVarietyId;
    private final LocalDateTime lastUpdated;
    private final LocalDateTime created;
    private final List<Comment> commentsList;

    /**
     *	Build an immutable CroppingActual entry one field at a time
     */
    public CroppingActual(int id,
                          int croppingPlanId,
                          int plantSpeciesId,
                          Integer plantVarietyId,
                          LocalDateTime lastUpdated,
                          LocalDateTime created,
                          Comment... comments)
    {
        this.id = id;
        this.croppingPlanId = croppingPlanId;
        this.plantSpeciesId = plantSpeciesId;
        this.plantVarietyId = plantVarietyId;
        this.lastUpdated = lastUpdated;
        this.created = created;

        if (comments != null && comments.length>0)
        {
            this.commentsList = new ArrayList<>(Arrays.asList(comments));
        }
        else
        {
            this.commentsList = Collections.emptyList();
        }
    }

    /**
     *	Build an immutable CroppingActual entry cloning the given CroppingActual entry but adding the comments list
     */
    CroppingActual(
            final CroppingActual toCopy,
            final Comment... comments)
    {
        this(toCopy, Arrays.asList(comments));
    }

    /**
     *	Build an immutable CroppingActual entry cloning the given CroppingActual entry but adding the comments list
     */
    CroppingActual(
            final CroppingActual toCopy,
            final List<Comment> comments)
    {
        this.id = toCopy.id;
        this.croppingPlanId = toCopy.croppingPlanId;
        this.plantSpeciesId = toCopy.plantSpeciesId;
        this.plantVarietyId = toCopy.plantVarietyId;
        this.lastUpdated = toCopy.lastUpdated;
        this.created = toCopy.created;

        if (comments != null && comments.size()>0)
        {
            if (toCopy.commentsList.size()>0)
            {
                // append new comments to previous list
                this.commentsList = new ArrayList<>(toCopy.commentsList);
                this.commentsList.addAll(comments);
            }
            else
            {	// no comments on original item
                this.commentsList = new ArrayList<>(comments);
            }
        }
        else
        {	// no new comments to add
            this.commentsList = toCopy.commentsList;
        }
    }

    /**
     *	Build an immutable CroppingActual entry from a JSON dump.
     *	The dumped object must be complete (all non-nullable fields must have values) except
     *	the id field can be null or absent to indicate that this is a new item to be inserted.
     *
     *	@param	json	a JsonObject holding all the fields for a full initialisation.
     */
    CroppingActual(JsonObject json)
    {
        this.id = json.getInt("id", -1);
        this.croppingPlanId = json.getInt("croppingPlanId");
        this.plantSpeciesId = json.getInt("plantSpeciesId");
        if (json.containsKey("plantVarietyId") && !json.isNull("plantVarietyId"))
        {
            this.plantVarietyId = json.getInt("plantVarietyId");
        }
        else
        {
            this.plantVarietyId = null;
        }

        this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
        this.created = LocalDateTime.parse(json.getString("created"));
        JsonArray jsonComments = json.getJsonArray("comments");
        if (jsonComments != null && !jsonComments.isEmpty())
        {// there is probably only one comment
            this.commentsList = new ArrayList<>(jsonComments.size());
            for (JsonObject ct : jsonComments.getValuesAs(JsonObject.class))
            {
                this.commentsList.add(new Comment(ct));
            }
        }
        else
        {
            this.commentsList = Collections.emptyList();
        }
    }	//constructor from JSON

    int getId()
    {
        return id;
    }
    @Override
    public Integer getKey()
    {
        return id;
    }

    @Override
    public NotebookEntryType getType()
    {
        return NotebookEntryType.CROPPINGACTUAL;
    }

    int getCroppingPlanId()
    {
        return croppingPlanId;
    }
    @Override
    public ICroppingPlan getCroppingPlan()
    {
        return MySQLCache.cacheCroppingPlan.get(croppingPlanId);
    }

    int getPlantSpeciesId()
    {
        return plantSpeciesId;
    }
    @Override
    public IPlantSpecies getPlantSpecies()
    {
        return MySQLCache.cachePlantSpecies.get(plantSpeciesId);
    }

    Integer getPlantVarietyId()
    {
        return plantVarietyId;
    }
    @Override
    public Optional<IPlantVariety> getPlantVariety()
    {
        return Optional.ofNullable(MySQLCache.cachePlantVariety.get(plantVarietyId));
    }

    @Override
    public LocalDateTime getLastUpdated()
    {
        return lastUpdated;
    }

    @Override
    public LocalDateTime getCreated()
    {
        return created;
    }

    @Override
    public List<IComment> getComments()
    {
        return new ArrayList<>(this.commentsList);
    }

    JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
    {
        JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
        jsonBuilder.add("id", id);
        jsonBuilder.add("croppingPlanId", croppingPlanId);
        jsonBuilder.add("plantSpeciesId", plantSpeciesId);
        if (plantVarietyId != null)
        {
            jsonBuilder.add("plantVarietyId", plantVarietyId);
        }
        else
        {
            jsonBuilder.addNull("plantVarietyId");
        }
        jsonBuilder.add("lastUpdated", lastUpdated.toString());
        jsonBuilder.add("created", created.toString());
        if (commentsList != null && !commentsList.isEmpty())
        {// no point writing an empty comments array (the loaders handle this)
            JsonArrayBuilder jsonComments = jsonFactory.createArrayBuilder();
            for (Comment ct : commentsList)
            {
                jsonComments.add(ct.toJson(jsonFactory));
            }
            jsonBuilder.add("comments", jsonComments);
        }
        jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "CroppingActual");
        return jsonBuilder;
    }	//	toJson

    @Override
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener)
    {
        flagHandler.addPropertyChangeListener(propertyName, listener);
    }

    @Override
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener)
    {
        flagHandler.removePropertyChangeListener(propertyName, listener);
    }

    @Override
    public void flagDeleted()
    {
        flagHandler.flagDeleted();
    }

    @Override
    public void flagReplaced(ICroppingActual newValue)
    {
        flagHandler.flagReplaced(newValue, newValue::addPropertyChangeListener);
    }

    @Override
    public String toString() {
        return "CroppingActual: " + "id: " + id + ", " +
                "croppingPlanId: " + croppingPlanId + ", " +
                "plantSpeciesId: " + plantSpeciesId + ", " +
                "plantVarietyId: " + plantVarietyId + ", " +
                "lastUpdated: " + lastUpdated + ", " +
                "created: " + created;
    }

}
