/*
 * Copyright (C) 2018, 2019, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.time.LocalDateTime;

import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;
import jakarta.json.JsonObjectBuilder;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class AfflictionLister implements IAfflictionLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private boolean useName = false;
	private String[] nameList = new String[10];
	private int nameNext = 0;	// next free slot in list

	private boolean useId = false;
	private int[] idList = new int[10];
	private int idNext = 0;	// next free slot in list

	private boolean useAfflictionClassId = false;
	private int[] afflictionClassIdList = new int[10];
	private int afflictionClassIdNext = 0;

	private boolean useWhere = false;

	AfflictionLister() {}

	@Override
	public List<IAffliction> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		List<IAffliction> vals = new ArrayList<>();
		boolean fetchAll = !useWhere;
        String query = "";
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = buildSQL_MySQL();
			case hsqldb -> query = buildSQL_hsqldb();
			case MSSQLServer -> query = buildSQL_MSSQLServer();
			default -> {
				LOGGER.debug("fetch(): no known rdbms");
				throw new GNDBException(new IllegalStateException("no known RDBMS"));
			}
		}
LOGGER.debug("fetch(): query: {}", query);

		try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
		{
			ResultSet rs = null;
			if (this.useName)
			{
				PreparedStatement pstmt = conn.prepareStatement(query);
				int pstmtIx = 1;
				if (this.useName)
                {
					for(int ix =0; ix < nameNext; ix++) {
LOGGER.debug("fetch(): pstmtIx: {} = {}", pstmtIx, nameList[ix].toLowerCase());
						pstmt.setString(pstmtIx++, nameList[ix].toLowerCase());}
				}
				rs = pstmt.executeQuery();
			}
			else {
				rs = stmt.executeQuery(query);
			}
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> vals = processResults_MySQL(rs);
				case hsqldb -> vals = processResults_hsqldb(rs);
				case MSSQLServer -> vals = processResults_MSSQLServer(rs);
				default -> {
					LOGGER.debug("fetch(): no known rdbms");
					throw new GNDBException(new IllegalStateException("no known RDBMS"));
				}
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("fetch(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

		if (vals.isEmpty()) return Collections.emptyList();

		for (IAffliction vx : vals)
		{
			MySQLCache.cacheAffliction.putIfAbsent(vx.getKey(), vx);
		}
		if (fetchAll) { MySQLCache.completeAffliction = true; }

		populateAfflictionClass();

LOGGER.traceExit(log4jEntryMsg);
		return vals;
	}	// fetch()

    private String buildSQL_MySQL()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from affliction as d ");
        query.append("left join (select * from comment where ownerType = 'AF') as c ");
        query.append("on d.afflictionId = c.ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by LOWER(d.name), d.afflictionId, c.date");
        return query.toString();
    }   //  buildSQL_MySQL()

    private String buildSQL_hsqldb()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from affliction as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'AF') as c ");
        query.append("on d.afflictionId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query); //				first = false;
        }
        query.append(" order by LOWER(d.name), d.afflictionId, c_date");
        return query.toString();
    }   //  buildSQL_hsqldb()

    private String buildSQL_MSSQLServer()
    {
        StringBuilder query = new StringBuilder("select d.afflictionId as d_afflictionId, " +
                                                        "d.afflictionClassId as d_afflictionClassId, " +
                                                        "d.name as d_name, " +
                                                        "d.description as d_description, " +
                                                        "d.lastUpdated as d_lastUpdated, " +
                                                        "d.created as d_created, " +
                                                        " c.* from affliction as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'AF') as c ");
        query.append("on d.afflictionId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query); //				first = false;
        }
        query.append(" order by LOWER(d.name), d.afflictionId, c_date");
        return query.toString();
    }   //  buildSQL_MSSQLServer()

    private void buildCommonSQL(StringBuilder query)
    {
        boolean first = true;
        if (this.useId)
        {
            if (first) query.append(" where ");
            else query.append(" and");
            if (idNext > 1) {
                query.append(" d.afflictionId in (");
                for(int ix = 0; ix < idNext; ix++) { query.append(idList[ix]).append(", "); }
                query.replace(query.length()-2, query.length(), ")");
            }
            else
                query.append(" d.afflictionId = ").append(idList[0]);
            first = false;
            this.useId = false;
            this.idNext = 0;
        }
        if (this.useAfflictionClassId)
        {
            if (first) query.append(" where ");
            else query.append(" and");
            if (afflictionClassIdNext > 1) {
                query.append(" d.afflictionClassId in (");
                for(int ix =0; ix < afflictionClassIdNext; ix++) {query.append(afflictionClassIdList[ix]).append(", ");}
                query.replace(query.length()-2, query.length(), ")");
            }
            else
                query.append(" d.afflictionClassId = ").append(afflictionClassIdList[0]);
            first = false;
            this.useAfflictionClassId = false;
            this.afflictionClassIdNext = 0;
        }
        if (this.useName)
        {
            if (first) query.append(" where ");
            else query.append(" and");
            if (nameNext > 1) {
                query.append(" LOWER(d.name) in (");
				query.append("?, ".repeat(nameNext));
                query.replace(query.length()-2, query.length(), ")");
            }
            else
                query.append("LOWER(d.name) = ?");
            this.useName = false;
            this.nameNext = 0;
        }
    }   //  buildCommonSQL()

	private List<IAffliction> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<IAffliction> tempList = new ArrayList<>();

		Affliction item = null;

		while (rs.next())
		{
			int afflictionId = rs.getInt("d.afflictionId");
			int tmp_afflictionClassId = rs.getInt("d.afflictionClassId");
			Integer afflictionClassId = rs.wasNull() ? null : tmp_afflictionClassId;
			String name = rs.getString("d.name");
			String description = rs.getString("d.description");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("afflictionId: {}, afflictionClassId: {}, name: {}, description: {}, lastUpdated: {}, created: {}",
                    afflictionId, afflictionClassId, name, description, lastUpdated, created);
			if (item != null && afflictionId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_MySQL(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("commentId"),
					rs.getInt("ownerId"),
					"AF",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): extra comment is: {}", comm);
				item = new Affliction(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
					item = new Affliction(afflictionId, afflictionClassId, name, description, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						afflictionId,
						"AF",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): first comment is: {}", comm);
					item = new Affliction(afflictionId, afflictionClassId, name, description, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<IAffliction> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<IAffliction> tempList = new ArrayList<>();

		Affliction item = null;

		while (rs.next())
		{
			int afflictionId = rs.getInt("affliction.afflictionId");
			int tmp_afflictionClassId = rs.getInt("affliction.afflictionClassId");
			Integer afflictionClassId = rs.wasNull() ? null : tmp_afflictionClassId;
			String name = rs.getString("affliction.name");
			String description = rs.getString("affliction.description");
			LocalDateTime lastUpdated = rs.getTimestamp("affliction.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("affliction.created").toLocalDateTime();
LOGGER.debug("afflictionId: {}, afflictionClassId: {}, name: {}, description: {}, lastUpdated: {}, created: {}",
                    afflictionId, afflictionClassId, name, description, lastUpdated, created);
			if (item != null && afflictionId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"AF",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new Affliction(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new Affliction(afflictionId, afflictionClassId, name, description, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						afflictionId,
						"AF",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new Affliction(afflictionId, afflictionClassId, name, description, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<IAffliction> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<IAffliction> tempList = new ArrayList<>();

		Affliction item = null;

		while (rs.next())
		{
			int afflictionId = rs.getInt("d_afflictionId");
			int tmp_afflictionClassId = rs.getInt("d_afflictionClassId");
			Integer afflictionClassId = rs.wasNull() ? null : tmp_afflictionClassId;
			String name = rs.getString("d_name");
			String description = rs.getString("d_description");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("afflictionId: {}, afflictionClassId: {}, name: {}, description: {}, lastUpdated: {}, created: {}", 
                    afflictionId, afflictionClassId, name, description, lastUpdated, created);
			if (item != null && afflictionId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"AF",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new Affliction(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new Affliction(afflictionId, afflictionClassId, name, description, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						afflictionId,
						"AF",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new Affliction(afflictionId, afflictionClassId, name, description, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	/*
	*	Populate the parents slot(s)
	*/
	private void populateAfflictionClass() throws GNDBException
	{
// parent table type: TF
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populateAfflictionClass()");

// just make sure everything is cached
		new AfflictionClassLister().load();
LOGGER.traceExit(log4jEntryMsg);
	}

	void clear()
	{
		MySQLCache.cacheAffliction.clear();
		MySQLCache.completeAffliction = false;
	}

	/**
	*
	*Select only the Affliction entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	AfflictionLister id(int... vals)
	{
		useId = true;
		if (idNext + vals.length >= idList.length)
		{
			idList = Arrays.copyOf(idList, idList.length+vals.length+10);
		}
		for (int val : vals)
		{
			idList[idNext++] = val;
		}
		this.useWhere = true;
		return this;
	}

	@Override
	public IAfflictionLister affliction(IAffliction... items)
	{
		if (items == null) return this;
		if (items.length == 0) return this;
		int[] keys = new int[items.length];
		int keyCount = 0;
		for (IAffliction item : items)
		{
			if (item == null) continue;
			Object ky = item.getKey();
			if (ky == null) continue;
			if (ky instanceof Integer) keys[keyCount++] = (Integer)ky;
		}
		if (keyCount == 0) return this;
		keys = Arrays.copyOf(keys, keyCount);	// trim array to actual size - should be a null-op
		return this.id(keys);
	}

	@Override
	public IAfflictionLister affliction(List<IAffliction> items)
	{
		if (items == null) return this;
		if (items.isEmpty()) return this;
		return this.affliction(items.toArray(new IAffliction[0]));
	}

	@Override
	public IAfflictionLister afflictionClass(IAfflictionClass... items)
	{
		if (items == null) return this;
		if (items.length == 0) return this;
		useAfflictionClassId = true;
		if (afflictionClassIdNext + items.length >= afflictionClassIdList.length)
		{
			afflictionClassIdList = Arrays.copyOf(afflictionClassIdList, afflictionClassIdList.length+items.length+10);
		}
		for (IAfflictionClass item : items) {
			if (item == null) continue;
			Object ky = item.getKey();
			if (ky == null) continue;
			if (ky instanceof Integer) {afflictionClassIdList[afflictionClassIdNext++] = (Integer)ky;}
		}
		useWhere = true;
		return this;
	}

	@Override
	public IAfflictionLister afflictionClass(List<IAfflictionClass> items)
	{
		if (items == null) return this;
		if (items.isEmpty()) return this;
		return this.afflictionClass(items.toArray(new IAfflictionClass[0]));
	}

	@Override
	public IAfflictionLister name(String... items)
	{
		if (items == null) return this;
		if (items.length == 0) return this;
		useName = true;
		if (nameNext + items.length >= nameList.length)
		{
			nameList = Arrays.copyOf(nameList, nameList.length + items.length + 10);
		}
		for (String item : items)
		{
			if (item != null) nameList[nameNext++] = item;
		}
		this.useWhere = true;
		return this;
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		useWhere = false;
		fetch();

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (IAffliction ihc : MySQLCache.cacheAffliction.values())
		{
			Affliction hc = (Affliction)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "Affliction");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "Affliction.json"), false)))
		{
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJson(): IOException", ex);
		}
	}	// toJson

}
