/*
 * Copyright (C) 2018, 2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
*	Change log
*	2.9.5	entries sorted into alphabetical order (by IntelliJ - very handy) except UNKNOWN
* 	3.0.0	Extra entries added for new table types
 */

package uk.co.gardennotebook.spi;

import java.util.Map;
import java.util.HashMap;

/**
 *	enum of the various types of Notebook entry used.
 *
 *    @author Andy Gegg
 *    @version 3.0.0
 *    @since 1.0
 */

public enum NotebookEntryType
{
	/**
	 *	These are the various afflictions observed.
	 */
	AFFLICTION("AF"),
	/**
	 *	Groups the types of affliction - pest, disease, death
	 */
	AFFLICTIONCLASS("AC"),
	/**
	 *	Records an observation of an affliction - for instance, an outbreak of red spider mite
	 */
	AFFLICTIONEVENT("AE"),
	/**
	 *	A comment on any item type
	 */
	COMMENT("CT"),
	/**
	 *	CROPGROUPSEQUENCE - the sequence of CropRotationGroups through any bed
	 */
	CROPGROUPSEQUENCE("RS"),
	/**
	 *	CROPPINGACTUAL - what was actually grown in the location shown in CroppingPlan
	 */
	CROPPINGACTUAL("CA"),
	/**
	 *	CROPPINGPLAN - what is grown where in a rotation cycle
	 */
	CROPPINGPLAN("CP"),
	/**
	 * Crop Rotation Group - how various plant species should be rotated
	 */
	CROPROTATIONGROUP("RG"),
	/**
	 *	An instance of groundwork being done.
	 */
	GROUNDWORK("GW"),
	/**
	 *	A type of groundwork activity.
	 */
	GROUNDWORKACTIVITY("GA"),
	/**
	 *	Diary entries for events in the life of a crop.
	 */
	HUSBANDRY("HU"),
	/**
	 *	The various types of diary entry for the husbandry lifecycle.
	 */
	HUSBANDRYCLASS("HC"),
	/**
	 *	Journal - a free form Diart entry
	 */
	JOURNAL("JN"),
	/**
	 *	Location, physical or logical locations of various activities
	 */
	LOCATION("LN"),
	/**
	 *	Various types of notes, such as cultivation notes, too long for or not best organised as comments.
	 */
	PLANTNOTE("PN"),
	/**
	 *	A species of plant grown in the garden, for instance, 'tomato'.
	 *	A species will have zero or more varieties.
	 *	Note the use of a Trigger to reflect changes to commonName into Product.
	 */
	PLANTSPECIES("PS"),
	/**
	 *	A particular variety of a plant species.
	 *	Note that the common name is NOT a unique value as different species may have varieties with the same name - e.g. Money Maker tomato AND aubergine.
	 *	Note the use of a Trigger to reflect changes to commonName into Product.
	 */
	PLANTVARIETY("PV"),
	/**
	 *	A purchasable product.
	 */
	PRODUCT("PR"),
	/**
	 *	A brand for one or more Products - such as Verve ({@literal B&Q} own brand)
	 */
	PRODUCTBRAND("BR"),
	/**
	 *	Categories of purchased products.
	 */
	PRODUCTCATEGORY("PC"),
	/**
	 *	Description of a purchase from a retailer.  The items purchased are recorded as PurchaseItems.
	 */
	PURCHASE("PU"),
	/**
	 *	An item in a Purchase.
	 */
	PURCHASEITEM("PI"),
	/**
	 *	A list of reminders of things to do.
	 *	Reminders can be 'one-off' or repeated at various intervals (e.g. 'sow lettuce in March' shown every year.).
	 */
	REMINDER("RM"),
	/**
	 *	Review - a Free form text entry reviewing some aspect of the gardening year
	 */
	REVIEW("RW"),
	/**
	 *	ReviewReferences - the items referenced in a Review
	 */
	REVIEWREFERENCES("RR"),
	/**
	 *	Anywhere that items are purchased, includes garden centres, supermarkets, online shopping, etc, etc.
	 *	(A more obvious name would be Supplier but that causes a potential name clash in Java)
	 */
	RETAILER("SU"),
	/**
	 *	Implements the many-many link between retailers and products.
	 *	Many products, particularly own-brand products will only have a single supplier, others may be available from many retailers.
	 */
	RETAILERHASPRODUCT("RP"),
	/**
	 *	A sale of produce.
	 */
	SALE("SA"),
	/**
	 *	An item in a sale of produce.
	 */
	SALEITEM("SI"),
	/**
	 *	A list of items which need to be acquired.
	 */
	SHOPPINGLIST("SL"),
	/**
	 *	A list of tasks to be done.
	 *	Reminders are scheduled, possibly repeating and will create items here when they are due.
	 @apiNote This list is not user updated - users create reminders which generate todo items
	 */
	TODOLIST("TD"),
	/**
	 *	A noteworthy weather event.
	 */
	WEATHER("WE"),
	/**
	 *	The types of weather observed as noteworthy.  For daily observations, use the Weather Details table.
	 */
	WEATHERCONDITION("WC"),
	/**
	 *	A noteworthy observation of wildlife - an interesting event, unusual species, etc
	 */
	WILDLIFE("WI"),
	/**
	 *	Grouping for wildlife observations.  Completely informal, e.g. 'insect', 'bee', 'bird'.
	 */
	WILDLIFESPECIES("WS"),
	/**
	 *	Unknown item type - ERROR CONDITION.
	 */
	UNKNOWN("??");

	private final String typeString;

	private static final Map<String, NotebookEntryType> stringToEnum = new HashMap<>();

	static
	{
		for (NotebookEntryType t : values()) stringToEnum.put(t.type(), t);
	}

	private NotebookEntryType(String type)
	{
		this.typeString = type;
	}

	/**
	 *	Translate a string into the appropriate enum.
	 *
	 *    @param    type    the string representation of the enum
	 *
	 *    @return the enum represented by the string
	 */
	public static NotebookEntryType fromString(String type)
	{
		return stringToEnum.get(type);
	}

	/**
	 *	Get the string value corresponding to an enum.
	 *
	 *    @return the string representation of the enum
	 */
	public String type()
	{
		return this.typeString;
	}

}
