/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook.spi;

import java.time.LocalDate;
import java.time.Year;
import java.util.List;

/**
 *	Review article for a gardening year.
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	1.0
 */
public interface IReviewLister
{
    /**
     *	Returns the Review entries matching the given criteria.
     *	If no criteria have been specified all Journal entries will be returned.
     *	Criteria are applied additively (i.e. criterion AND criterion).
     *
     *	@return	A list of Journal entries matching previously specified criteria.
     *
     *	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
     *			The causal Exception can be retrieved via <code>getCause()</code>
     */
    public List<? extends IReview> fetch() throws GNDBException;

    /**
     *	Select only these Review entries.
     *	May be called repeatedly to add to the list of Review entries to be returned.
     *
     *	@param items	the list of Review entries.  A null value or empty list is a null-op
     *	@return	this Lister
     */
    public IReviewLister review(IReview... items);

    /**
     *	Select only these Review entries.
     *	May be called repeatedly to add to the list of Review entries to be returned.
     *
     *	@param items	the list of Review entries.  A null value or empty list is a null-op
     *	@return	this Lister
     */
    public IReviewLister review(List<IReview> items);

    /**
     *	Select only those Review entries which cover this year.  Only the year part of the date will be used.
     *	A null value will be ignored.
     *	May be called repeatedly to add to the list of Journal entries to be returned.
     *
     *	@param	date	the earliest date to match
     *	@return	this Lister
     */
    public IReviewLister yearInReview(Year... date);

    /**
     *	Select only those Journal entries which cover this year.  Only the year part of the date will be used.
     *	A null value will be ignored.
     *	May be called repeatedly to add to the list of Journal entries to be returned.
     *
     *	@param	date	the earliest date to match
     *	@return	this Lister
     */
    public IReviewLister yearInReview(List<Year> date);

    /**
     *	Select only those Review entries which cover a period starting on or after this date.
     *	A null value will be ignored.
     *	If called repeatedly only the last value given will be used.
     *	If this method is not called, no earliest date will be used in selection.
     *
     *	@param	date	the earliest date to match
     *	@return	this Lister
     */
    public IReviewLister coverFrom(LocalDate date);
    /**
     *	Select only those Review entries which cover a period ending on or before this date.
     *	A null value will be ignored.
     *	If called repeatedly only the last value given will be used.
     *	If this method is not called, no latest date will be used in selection.
     *
     *	@param	date	the latest date to match
     *	@return	this Lister
     */
    public IReviewLister coverTo(LocalDate date);

    /**
     *	Select only those Review entries which are dated on or after this date.
     *	A null value will be ignored.
     *	If called repeatedly only the last value given will be used.
     *	If this method is not called, no earliest date will be used in selection.
     *
     *	@param	date	the earliest date to match
     *	@return	this Lister
     */
    public IReviewLister fromDate(LocalDate date);
    /**
     *	Select only those Review entries which are dated on or before this date.
     *	A null value will be ignored.
     *	If called repeatedly only the last value given will be used.
     *	If this method is not called, no latest date will be used in selection.
     *
     *	@param	date	the latest date to match
     *	@return	this Lister
     */
    public IReviewLister toDate(LocalDate date);

}
