/*
 * Copyright (C) 2018, 2019 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
    2.3.0   Selection by Product
 */

package uk.co.gardennotebook.spi;

import java.util.List;

import java.time.*;

import java.math.BigDecimal;

/**
Description of a purchase from a retailer.  The items purchased are recorded as PurchaseItems.
*/
public interface IPurchaseLister
{
	/**
	*	Returns the Purchase entries matching the given criteria.
	*	If no criteria have been specified all Purchase entries will be returned.
	*	Criteria are applied additively (i.e. criterion AND criterion).
	*
	*	@return	A list of Purchase entries matching previously specified criteria.
	*
	*	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
	*			The causal Exception can be retrieved via <code>getCause()</code>
	*/
	public List<IPurchase> fetch() throws GNDBException;

	/**
	*	Select only these Purchase entries
	*	May be called repeatedly to add to the list of Purchase entries to be returned
	*
	*	@param items	the list of Purchase entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IPurchaseLister purchase(IPurchase... items);

	/**
	*	Select only these Purchase entries
	*	May be called repeatedly to add to the list of Purchase entries to be returned
	*
	*	@param items	the list of Purchase entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IPurchaseLister purchase(List<IPurchase> items);

	/**
	*	Select only those Purchase entries which are for these Retailer entries.
	*	May be called repeatedly to add to the list of Retailer entries to be selected.
	*
	*	@param items	the list of Retailer items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IPurchaseLister retailer(IRetailer... items);

	/**
	*	Select only those Purchase entries which are for these Products.
	*	May be called repeatedly to add to the list of Products to be selected
	*
	*	@param items	the list of Products.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IPurchaseLister product(List<IProduct> items);

	/**
	*	Select only those Purchase entries which are for these Products.
	*	May be called repeatedly to add to the list of Products to be selected.
	*
	*	@param items	the list of Products.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IPurchaseLister product(IProduct... items);

	/**
	*	Select only those Purchase entries which are for these Retailer entries.
	*	May be called repeatedly to add to the list of Retailer entries to be selected
	*
	*	@param items	the list of Retailer items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IPurchaseLister retailer(List<IRetailer> items);

	/**
	*	Select only those Purchase entries which are on or after this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no earliest date will be used in selection
	*
	*	@param	date	the earliest date to match
	*	@return	this Lister
	*/
	public IPurchaseLister fromDate(LocalDate date);
	/**
	*	Select only those Purchase entries which are on or before this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no latest date will be used in selection
	*
	*	@param	date	the latest date to match
	*	@return	this Lister
	*/
	public IPurchaseLister toDate(LocalDate date);

	/**
	*	Select only those Purchase entries which are on or after this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no earliest date will be used in selection
	*
	*	@param	date	the earliest date to match
	*	@return	this Lister
	*/
	public IPurchaseLister fromDeliveryDate(LocalDate date);
	/**
	*	Select only those Purchase entries which are on or before this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no latest date will be used in selection
	*
	*	@param	date	the latest date to match
	*	@return	this Lister
	*/
	public IPurchaseLister toDeliveryDate(LocalDate date);

}
