/*
 * Copyright (C) 2018, 2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Support 'watch for' reminders and ToDoList entries
 */

package uk.co.gardennotebook.spi;

import uk.co.gardennotebook.util.SimpleMoney;
import java.beans.PropertyChangeListener;
import java.util.List;
import java.time.*;

import java.math.BigDecimal;

import java.util.Optional;

/**
*	An item in a Purchase.
*
*	@author	Andy Gegg
*	@version	3.0.0
*	@since	1.0
*/
public interface IPurchaseItem extends INotebookEntry
{
	/**
	*	The owning Purchase.
	*
	*	@return	the Purchase
	*/
	public IPurchase getPurchase();

	/**
	*	The owning Product.
	*
	*	@return	the Product
	*/
	public IProduct getProduct();

	/**
	*	Get the value of quantity (if present).
	*
	*	@return	the value of quantity
	*/
	public Optional<BigDecimal> getQuantity();

	/**
	*	Get the value of unit (if present).
	*
	*	@return	the value of unit
	*/
	public Optional<String> getUnit();

	/**
	*	Get the value of itemCost (if present).
	*
	*	@return	the value of itemCost
	*/
	public Optional<BigDecimal> getItemCost();

	/**
	*	Get the value of currency (if present).
	*
	*	@return	the value of currency
	*/
	public Optional<String> getCurrency();

	/**
	*	Get the value of lastUpdated.
	*
	*	@return	the value of lastUpdated
	*/
	public LocalDateTime getLastUpdated();

	/**
	*	Get the value of created.
	*
	*	@return	the value of created
	*/
	public LocalDateTime getCreated();

	/**
	*	Get the value of itemCost as a Money value.
	*
	*	@return	the value of itemCost
	*/
	public SimpleMoney getItemPrice();

	/**
	 *	Get the ToDoList entries for this Husbandry.
	 *
	 *	@return	 a lister for the ToDoList entries
	 */
	public IToDoListLister getToDoList();

	/**
	 *	Get the Reminder entries for this Husbandry.
	 *
	 *	@return	 a lister for the Reminder entries
	 */
	public IReminderLister getReminder();

	/**
	*	Add a change listener to the PurchaseItem.
	*
	*	@param	propertyName	identifies the change being flagged.  One of:<UL>
	*			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
	*			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
	*			<LI>Comment	-	this comment has been added to or deleted from this PurchaseItem, newValue is an added comment, oldValue is a deleted comment
	</UL>
	*	@param	listener	a PropertyChangeListener (may be a Lambda expression)
	*/
	public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener);

	/**
	*	Remove a change listener from the PurchaseItem.
	*
	*	@param	propertyName	identifies the change being flagged.  One of:<UL>
	*			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
	*			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
	*			<LI>Comment	-	this comment has been added to or deleted from this PurchaseItem, newValue is an added comment, oldValue is a deleted comment
	</UL>
	*	@param	listener	the listener to remove
	*/
	public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener);

	/**
	*	Signal to all listeners that this PurchaseItem has been deleted.
	*
	*/
	public void flagDeleted();

	/**
	*	Signal to all listeners that this PurchaseItem has been replaced.
	*
	*	@param	newValue	the replacement PurchaseItem
	*/
	public void flagReplaced(IPurchaseItem newValue);

	/**
	 *	Signal to all listeners that a child has been deleted.
	 *
	 *	@param	child	this ToDoList is no longer a child of this Husbandry
	 */
	public void flagChildDeleted(IToDoList child);

	/**
	 *	Signal to all listeners that a child has been added.
	 *
	 *	@param	child	this ToDoList has been added as a child of this Husbandry
	 */
	public void flagChildAdded(IToDoList child);

	/**
	 *	Signal to all listeners that a child has been deleted.
	 *
	 *	@param	child	this Reminder is no longer a child of this Husbandry
	 */
	public void flagChildDeleted(IReminder child);

	/**
	 *	Signal to all listeners that a child has been added.
	 *
	 *	@param	child	this Reminder has been added as a child of this Husbandry
	 */
	public void flagChildAdded(IReminder child);

}

