/*
 * Copyright (C) 2018-2020 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
    2.8.0   Allow PlantVariety selection to be forced null
                This is to support editing ShoppingList and PurchaseItem entries
 */

package uk.co.gardennotebook.spi;

import java.util.List;


/**
*   A purchasable product.
*
*	@author	Andy Gegg
*	@version	2.8.0
*	@since	1.0
*/
public interface IProductLister
{
	/**
	*	Returns the Product entries matching the given criteria.
	*	If no criteria have been specified all Product entries will be returned.
	*	Criteria are applied additively (i.e. criterion AND criterion).
	*
	*	@return	A list of Product entries matching previously specified criteria.
	*
	*	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
	*			The causal Exception can be retrieved via <code>getCause()</code>
	*/
	public List<IProduct> fetch() throws GNDBException;

	/**
	*	Select only these Product entries
	*	May be called repeatedly to add to the list of Product entries to be returned
	*
	*	@param items	the list of Product entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IProductLister product(IProduct... items);

	/**
	*	Select only these Product entries
	*	May be called repeatedly to add to the list of Product entries to be returned
	*
	*	@param items	the list of Product entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IProductLister product(List<IProduct> items);

	/**
	*	Select only those Product entries which are for these PlantSpecies entries.
	*	May be called repeatedly to add to the list of PlantSpecies entries to be selected.
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Product entries matching either will be selected.
	*
	*	@param items	the list of PlantSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IProductLister plantSpecies(IPlantSpecies... items);

	/**
	*	Select only those Product entries which are for these PlantSpecies entries.
	*	May be called repeatedly to add to the list of PlantSpecies entries to be selected
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Product entries matching either will be selected.
	*
	*	@param items	the list of PlantSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IProductLister plantSpecies(List<IPlantSpecies> items);

	/**
	*	Select only those Product entries which are for these ProductBrand entries.
	*	May be called repeatedly to add to the list of ProductBrand entries to be selected.
	*
	*	@param items	the list of ProductBrand items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IProductLister productBrand(IProductBrand... items);

	/**
	*	Select only those Product entries which are for these ProductBrand entries.
	*	May be called repeatedly to add to the list of ProductBrand entries to be selected
	*
	*	@param items	the list of ProductBrand items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IProductLister productBrand(List<IProductBrand> items);

	/**
	*	Select only those Product entries which are for these PlantVariety entries.
	*	May be called repeatedly to add to the list of PlantVariety entries to be selected.
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Product entries matching either will be selected.
	*
	*	@param items	the list of PlantVariety items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IProductLister plantVariety(IPlantVariety... items);

	/**
	*	Select only those Product entries which are for these PlantVariety entries.
	*	May be called repeatedly to add to the list of PlantVariety entries to be selected
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Product entries matching either will be selected.
	*
	*	@param items	the list of PlantVariety items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IProductLister plantVariety(List<IPlantVariety> items);

	/**
	*	Select only those Product entries which are for these ProductCategory entries.
	*	May be called repeatedly to add to the list of ProductCategory entries to be selected.
	*
	*	@param items	the list of ProductCategory items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IProductLister productCategory(IProductCategory... items);

	/**
	*	Select only those Product entries which are for these ProductCategory entries.
	*	May be called repeatedly to add to the list of ProductCategory entries to be selected
	*
	*	@param items	the list of ProductCategory items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IProductLister productCategory(List<IProductCategory> items);

	/**
	*	Select only the Product entries with these names
	*	If called repeatedly, only the last name(s) will be used
	*
	*	@param name	the main name of the Product.  If null, <B>all</B> name checks are cleared
	*	@param nameDetail_1	the secondary name of the Product.  If null only Products with a NULL value will be returned
	*	@param nameDetail_2	the tertiary name of the Product.  If null only Products with a NULL value will be returned
	*	@return	this Lister
	*/
	public IProductLister name(String name, String nameDetail_1, String nameDetail_2);

	/**
	*	Select only the Product entries with a null value for ProductBrand
	*	@param requireNull	true if only Products with an explicitly null value for Productbrand are required
	*	@return	this Lister
	*/
	public IProductLister productBrandRequireNull(boolean requireNull);

	/**
	*	Select only the Product entries with a null value for PlantSpecies
	*	@param requireNull	true if only Products with an explicitly null value for PlantSpecies are required
	*	@return	this Lister
	*/
	public IProductLister plantSpeciesRequireNull(boolean requireNull);

	/**
	*	Select only the Product entries with a null value for PlantSpecies
	*	@param requireNull	true if only Products with an explicitly null value for PlantSpecies are required
	*	@return	this Lister
    * 
    *   @since  2.8.0
	*/
	public IProductLister plantVarietyRequireNull(boolean requireNull);

}
