/*
 *
 *  Copyright (C) 2023 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

 /*
    Change log
    3.2.1   Support selection by Location and allow multiple selections.
 */

package uk.co.gardennotebook.spi;

import java.time.LocalDate;
import java.util.List;

/**
 *   Lifecycle events analysed into time sequence(s).
 *
 *	@author	Andy Gegg
 *	@version	3.2.1
 *	@since	3.2.0
 */
public interface ILifecycleAnalysisLister
{

    List<ILifecycleAnalysis> fetch()  throws GNDBException;

    /**
     * Perform life cycle analysis for this plant species.
     *
     * @param plantSpecies  the PlantSpecies for analysis.
     *
     * @return  this Lister.
     */
    ILifecycleAnalysisLister plantSpecies(IPlantSpecies... plantSpecies);
    /**
     * Perform life cycle analysis for this plant species.
     *
     * @param plantSpecies  the PlantSpecies for analysis.
     *
     * @return  this Lister.
     */
    ILifecycleAnalysisLister plantSpecies(List<IPlantSpecies> plantSpecies);

    /**
     * Perform life cycle analysis for these plant varieties.
     * Repeated calls will append to the current list.
     * If no plant varieties are explicitly given, all varieties for the species will be analysed.
     *
     * @param plantVarieties  the PlantVarieties for analysis.
     *
     * @return  this Lister.
     */
    ILifecycleAnalysisLister plantVariety(IPlantVariety... plantVarieties);
    /**
     * Perform life cycle analysis for these plant varieties.
     * Repeated calls will append to the current list.
     * If no plant varieties are explicitly given, all varieties for the species will be analysed.
     *
     * @param plantVarieties  the PlantVarieties for analysis.
     *
     * @return  this Lister.
     */
    ILifecycleAnalysisLister plantVariety(List<IPlantVariety> plantVarieties);

    /**
     * Perform life cycle analysis only for events on this location.
     *
     * @param locations the location to use.
     *
     * @return this lister
     */
    ILifecycleAnalysisLister location(ILocation... locations);
    /**
     * Perform life cycle analysis only for events on this location.
     *
     * @param locations the location to use.
     *
     * @return this lister
     */
    ILifecycleAnalysisLister location(List<ILocation> locations);

    /**
     *	Select only those Lifecycle events which are on or after this date.
     *	A null value will be ignored.
     *	If called repeatedly only the last value given will be used.
     *	If this method is not called, no earliest date will be used in selection.
     *
     *	@param	date	the earliest date to match
     *	@return	this Lister
     */
    ILifecycleAnalysisLister fromDate(LocalDate date);
    /**
     *	Select only those Lifecycle events which are on or before this date.
     *	A null value will be ignored.
     *	If called repeatedly only the last value given will be used.
     *	If this method is not called, no latest date will be used in selection.
     *
     *	@param	date	the latest date to match
     *	@return	this Lister
     */
    ILifecycleAnalysisLister toDate(LocalDate date);

}
