/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook.spi;

import java.time.LocalDate;
import java.util.List;

/**
 *	Free text diary entries not connected to any specific plant, pest, etc.
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	1.0
 */
public interface IJournalLister
{
    /**
     *	Returns the Journal entries matching the given criteria.
     *	If no criteria have been specified all Journal entries will be returned.
     *	Criteria are applied additively (i.e. criterion AND criterion).
     *
     *	@return	A list of Journal entries matching previously specified criteria.
     *
     *	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
     *			The causal Exception can be retrieved via <code>getCause()</code>
     */
    public List<? extends IJournal> fetch() throws GNDBException;

    /**
     *	Select only these Journal entries
     *	May be called repeatedly to add to the list of Journal entries to be returned
     *
     *	@param items	the list of Journal entries.  A null value or empty list is a null-op
     *	@return	this Lister
     */
    public IJournalLister journal(IJournal... items);

    /**
     *	Select only these Journal entries
     *	May be called repeatedly to add to the list of Journal entries to be returned
     *
     *	@param items	the list of Journal entries.  A null value or empty list is a null-op
     *	@return	this Lister
     */
    public IJournalLister journal(List<IJournal> items);

    /**
     *	Select only those Journal entries which are on or after this date
     *	A null value will be ignored
     *	If called repeatedly only the last value given will be used
     *	If this method is not called no earliest date will be used in selection
     *
     *	@param	date	the earliest date to match
     *	@return	this Lister
     */
    public IJournalLister fromDate(LocalDate date);
    /**
     *	Select only those Journal entries which are on or before this date
     *	A null value will be ignored
     *	If called repeatedly only the last value given will be used
     *	If this method is not called no latest date will be used in selection
     *
     *	@param	date	the latest date to match
     *	@return	this Lister
     */
    public IJournalLister toDate(LocalDate date);

}
