/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook.spi;

import java.time.Year;
import java.util.List;

/**
 * This records the type of crop grown in a given location in a given planting year.  For instance, 'Alliums' in 'Bed 1'
 * in 2021.  Details of the actual plant species and, possibly, varieties can be added through the CroppingActual table.
 * Entries for past years are records of what was grown, entries for future years are to record the intended rotation of
 * crops through the various beds; the intention is that more detail is added (in CroppingActual) as plants are
 * ordered and sown.
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	3.0.0
 */

public interface ICroppingPlanLister
{
    /**
     *	Returns the CroppingPlan entries matching the given criteria.
     *	If no criteria have been specified all CroppingPlan entries will be returned.
     *	Criteria are applied additively (i.e. criterion AND criterion).
     *
     *	@return	A list of CroppingPlan entries matching previously specified criteria.
     *
     *	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
     *			The causal Exception can be retrieved via <code>getCause()</code>
     */
    public List<? extends ICroppingPlan> fetch() throws GNDBException;

    /**
     *	Select only these CroppingPlan entries
     *	May be called repeatedly to add to the list of CroppingPlan entries to be returned
     *
     *	@param items	the list of CroppingPlan entries.  A null value or empty list is a null-op
     *	@return	this Lister
     */
    public ICroppingPlanLister croppingPlan(ICroppingPlan... items);

    /**
     *	Select only these CroppingPlan entries
     *	May be called repeatedly to add to the list of CroppingPlan entries to be returned
     *
     *	@param items	the list of CroppingPlan entries.  A null value or empty list is a null-op
     *	@return	this Lister
     */
    public ICroppingPlanLister croppingPlan(List<ICroppingPlan> items);

    /**
     *	Select only those CroppingPlan entries which are for these CropRotationGroup entries.
     *	May be called repeatedly to add to the list of CropRotationGroup entries to be selected.
     *
     *	@param items	the list of CropRotationGroup items.  A null value or empty list is a null-op
     *	@return	this Lister
     */
    public ICroppingPlanLister cropRotationGroup(ICropRotationGroup... items);

    /**
     *	Select only those CroppingPlan entries which are for these CropRotationGroup entries.
     *	May be called repeatedly to add to the list of CropRotationGroup entries to be selected
     *
     *	@param items	the list of HusbandryClass items.  A null value or empty list is a null-op
     *	@return	this Lister
     */
    public ICroppingPlanLister cropRotationGroup(List<ICropRotationGroup> items);

    /**
     *	Select only those CroppingPlan entries which are for these Locations.
     *	May be called repeatedly to add to the list of Locations to be selected.
     *
     *	@param items	the list of Locations.  A null value or empty list is a null-op
     *	@return	this Lister
     */
    public ICroppingPlanLister location(ILocation... items);

    /**
     *	Select only those CroppingPlan entries which are for these Locations.
     *	May be called repeatedly to add to the list of Locations to be selected
     *
     *	@param items	the list of Locations.  A null value or empty list is a null-op
     *	@return	this Lister
     */
    public ICroppingPlanLister location(List<ILocation> items);

    /**
     *	Select only those CroppingPlan entries for this year.  Only the year part of the date will be used.
     *	A null value will be ignored.
     *	May be called repeatedly to add to the list of CroppingPlan entries to be returned.
     *
     *	@param	yearOfPlan	the earliest date to match
     *	@return	this Lister
     */
    public ICroppingPlanLister yearOfPlan(Year... yearOfPlan);

    /**
     *	Select only those CroppingPlan entries for this year.  Only the year part of the date will be used.
     *	A null value will be ignored.
     *	May be called repeatedly to add to the list of CroppingPlan entries to be returned.
     *
     *	@param	yearOfPlan	the earliest date to match
     *	@return	this Lister
     */
    public ICroppingPlanLister yearOfPlan(List<Year> yearOfPlan);

}
