/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook.spi;

import java.beans.PropertyChangeListener;
import java.time.LocalDateTime;
import java.util.Optional;

/**
 *	The crop rotation groups as used to avoid disease build up, etc.
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	3.0.0
 */
public interface ICropRotationGroup extends INotebookEntry
{
    /*
	*	Get the value of name.
	*
    *	@return	the value of name
	*/
    public String getName();

    /**
     *	Get the value of description (if present).
     *
     *	@return	the value of description
     */
    public Optional<String> getDescription();

    /**
     * Get the CropRotationGroup which follows this one
     *
     * @return  the following CropRotationGroup
     */
    public Optional<ICropRotationGroup> getSubsequentCropGroup();

    /**
     *	Get the value of lastUpdated.
     *
     *	@return	the value of lastUpdated
     */
    public LocalDateTime getLastUpdated();

    /**
     *	Get the value of created.
     *
     *	@return	the value of created
     */
    public LocalDateTime getCreated();

    /**
     *	Get the PlantSpecies entries for this CropRotationGroup.
     *
     *	@return	 a lister for the PlantSpecies entries
     */
    public IPlantSpeciesLister getPlantSpecies();

    /**
     *	Add a change listener to the CropRotationGroup.
     *
     *	@param	propertyName	identifies the change being flagged.  One of:<UL>
     *			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
     *			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
     *			<LI>Comment	-	this comment has been added to or deleted from this AfflictionClass, newValue is an added comment, oldValue is a deleted comment
    </UL>
     *	@param	listener	a PropertyChangeListener (may be a Lambda expression)
     */
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener);

    /**
     *	Remove a change listener from the CropRotationGroup.
     *
     *	@param	propertyName	identifies the change being flagged.  One of:<UL>
     *			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
     *			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
     *			<LI>Comment	-	this comment has been added to or deleted from this AfflictionClass, newValue is an added comment, oldValue is a deleted comment
    </UL>
     *	@param	listener	the listener to remove
     */
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener);

    /**
     *	Signal to all listeners that this CropRotationGroup has been deleted.
     *
     */
    public void flagDeleted();

    /**
     *	Signal to all listeners that this CropRotationGroup has been replaced.
     *
     *	@param	newValue	the replacement CropRotationGroup
     */
    public void flagReplaced(ICropRotationGroup newValue);

    /**
     *	Signal to all listeners that a child has been deleted.
     *
     *	@param	child	this PlantSpecies is no longer a child of this AfflictionClass
     */
    public void flagChildDeleted(IPlantSpecies child);

    /**
     *	Signal to all listeners that a child has been added.
     *
     *	@param	child	this PlantSpecies has been added as a child of this AfflictionClass
     */
    public void flagChildAdded(ICroppingPlan child);

    /**
     *	Signal to all listeners that a child has been deleted.
     *
     *	@param	child	this PlantSpecies is no longer a child of this AfflictionClass
     */
    public void flagChildDeleted(ICroppingPlan child);

    /**
     *	Signal to all listeners that a child has been added.
     *
     *	@param	child	this PlantSpecies has been added as a child of this AfflictionClass
     */
    public void flagChildAdded(IPlantSpecies child);

}
