/*
 * Copyright (C) 2018, 2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Location, Quantity and Severity added
 */

package uk.co.gardennotebook.spi;

import java.util.List;

import java.time.*;

/**
Records an observation of an affliction - for instance, an outbreak of red spider mite
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	1.0
*/
public interface IAfflictionEventLister
{
	/**
	*	Returns the AfflictionEvent entries matching the given criteria.
	*	If no criteria have been specified all AfflictionEvent entries will be returned.
	*	Criteria are applied additively (i.e. criterion AND criterion).
	*
	*	@return	A list of AfflictionEvent entries matching previously specified criteria.
	*
	*	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
	*			The causal Exception can be retrieved via <code>getCause()</code>
	*/
	public List<IAfflictionEvent> fetch() throws GNDBException;

	/**
	*	Select only these AfflictionEvent entries
	*	May be called repeatedly to add to the list of AfflictionEvent entries to be returned
	*
	*	@param items	the list of AfflictionEvent entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IAfflictionEventLister afflictionEvent(IAfflictionEvent... items);

	/**
	*	Select only these AfflictionEvent entries
	*	May be called repeatedly to add to the list of AfflictionEvent entries to be returned
	*
	*	@param items	the list of AfflictionEvent entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IAfflictionEventLister afflictionEvent(List<IAfflictionEvent> items);

	/**
	*	Select only those AfflictionEvent entries which are for these PlantSpecies entries.
	*	May be called repeatedly to add to the list of PlantSpecies entries to be selected.
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, AfflictionEvent entries matching either will be selected.
	*
	*	@param items	the list of PlantSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IAfflictionEventLister plantSpecies(IPlantSpecies... items);

	/**
	*	Select only those AfflictionEvent entries which are for these PlantSpecies entries.
	*	May be called repeatedly to add to the list of PlantSpecies entries to be selected
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, AfflictionEvent entries matching either will be selected.
	*
	*	@param items	the list of PlantSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IAfflictionEventLister plantSpecies(List<IPlantSpecies> items);

	/**
	*	Select only those AfflictionEvent entries which are for these Affliction entries.
	*	May be called repeatedly to add to the list of Affliction entries to be selected.
	*
	*	@param items	the list of Affliction items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IAfflictionEventLister affliction(IAffliction... items);

	/**
	*	Select only those AfflictionEvent entries which are for these Affliction entries.
	*	May be called repeatedly to add to the list of Affliction entries to be selected
	*
	*	@param items	the list of Affliction items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IAfflictionEventLister affliction(List<IAffliction> items);

	/**
	*	Select only those AfflictionEvent entries which are for these PlantVariety entries.
	*	May be called repeatedly to add to the list of PlantVariety entries to be selected.
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, AfflictionEvent entries matching either will be selected.
	*
	*	@param items	the list of PlantVariety items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IAfflictionEventLister plantVariety(IPlantVariety... items);

	/**
	*	Select only those AfflictionEvent entries which are for these PlantVariety entries.
	*	May be called repeatedly to add to the list of PlantVariety entries to be selected
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, AfflictionEvent entries matching either will be selected.
	*
	*	@param items	the list of PlantVariety items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IAfflictionEventLister plantVariety(List<IPlantVariety> items);

	/**
	*	Select only those AfflictionEvent entries which do NOT specify a plant species or variety.
	*
	*	@return	this Lister
	*/
	public IAfflictionEventLister requireNoPlant();

	/**
	 *	Select only those AfflictionEvent entries which are for these Locations.
	 *	May be called repeatedly to add to the list of Locations to be selected.
	 *
	 *	@param items	the list of Locations.  A null value or empty list is a null-op
	 *	@return	this Lister
	 */
	public IAfflictionEventLister location(ILocation... items);

	/**
	 *	Select only those AfflictionEvent entries which are for these Locations.
	 *	May be called repeatedly to add to the list of Locations to be selected
	 *
	 *	@param items	the list of Locations.  A null value or empty list is a null-op
	 *	@return	this Lister
	 */
	public IAfflictionEventLister location(List<ILocation> items);

	/**
	*	Select only those AfflictionEvent entries which are on or after this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no earliest date will be used in selection
	*
	*	@param	date	the earliest date to match
	*	@return	this Lister
	*/
	public IAfflictionEventLister fromDate(LocalDate date);
	/**
	*	Select only those AfflictionEvent entries which are on or before this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no latest date will be used in selection
	*
	*	@param	date	the latest date to match
	*	@return	this Lister
	*/
	public IAfflictionEventLister toDate(LocalDate date);

}
