/*
 *
 *  Copyright (C) 2021, 2023 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

/*
	Change log
    3.0.0   First version.
    3.1.0   Fixing the bug in CommentCellImpl meant that the comments tried to bind to the non-existent ReviewBean for
            a Year node.  So, added a pretend bean and made sure the cell is not editable.
 */

package uk.co.gardennotebook;

import javafx.beans.property.ReadOnlyStringProperty;
import javafx.beans.property.ReadOnlyStringWrapper;
import javafx.beans.property.SimpleStringProperty;
import javafx.beans.property.StringProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.CommentBean;
import uk.co.gardennotebook.fxbean.ReviewBean;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.NotebookEntryType;

import java.time.Year;
import java.util.List;

/**
 *	An aggregate and facade on the ReviewBean and a placeholder for the Year In Review.
 *
 *	@author	Andy Gegg
 *	@version	3.1.0
 *	@since	3.0.0
 */
final class ReviewTreeBean
{

    private static final Logger LOGGER = LogManager.getLogger();

    private ReviewBean reviewBean = null;
//    private Year yearMarker;

    private final NotebookEntryType nodeType;    // RW for Review, ?? for year marker (this is a bodge!)

    private StringProperty yearProperty = new SimpleStringProperty(this, "yearInReview", null);
    private StringProperty titleProperty = new SimpleStringProperty(this, "title", "");
    private StringProperty descriptionProperty = new SimpleStringProperty(this, "description", "");
    private final ReadOnlyStringWrapper summaryProperty = new ReadOnlyStringWrapper(this, "summary", "");

//    private ObservableList<ReviewTreeBean> yearReviews = FXCollections.emptyObservableList();

    /**
     * Add a new Year node for the tree.
     *
     * @param yearInReview  the year to add
     */
    ReviewTreeBean(Year yearInReview)
    {
        reviewBean = new ReviewBean();
        yearProperty.setValue(yearInReview.toString());
        nodeType = NotebookEntryType.UNKNOWN;
//        yearReviews = FXCollections.observableArrayList();

    }

    /**
     * Add a Review node to the tree.
     *
     * @param review    the Review to add
     */
    ReviewTreeBean(ReviewBean review)
    {
        reviewBean = review;
        titleProperty = review.titleProperty();
        descriptionProperty.setValue(review.getDescription());
        summaryProperty.set(review.getSummary());
        nodeType = NotebookEntryType.REVIEW;
    }

    NotebookEntryType getNodeType()
    {
        return nodeType;
    }

    ReviewBean getItem()
    {
        return reviewBean;
    }

    ObservableList<CommentBean> getComments()
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("getComments()");
        LOGGER.debug("Node type: {}, summary: {}", nodeType, summaryProperty.getReadOnlyProperty().get());

        if (nodeType == NotebookEntryType.REVIEW)
        {
            return reviewBean.getComments();
        }
        else
        {
           return FXCollections.emptyObservableList();
        }

    }

    boolean canDelete()
    {
        try {
            if (nodeType == NotebookEntryType.REVIEW)
            {
                return reviewBean.canDelete();
            }
            else
            {
                return false;
            }
        } catch (GNDBException ex) {
            PanicHandler.panic(ex);
        }
        return false;
    }

    void delete()
    {
        if (nodeType == NotebookEntryType.REVIEW)
        {
            try {
                    reviewBean.delete();
            } catch (GNDBException ex) {
                PanicHandler.panic(ex);
            }
        }
    }

    /**
     *	Wraps the YearInReview value of the underlying item
     *
     *	@return	a writable property wrapping the yearInReview attribute
     */
    public final StringProperty yearInReviewProperty()
    {
        return yearProperty;
    }

    /**
     *	Wraps the Title value of the underlying item
     *
     *	@return	a writable property wrapping the title attribute
     */
    public final StringProperty titleProperty()
    {
        return titleProperty;
    }

    /**
     *	Wraps the Description value of the underlying item
     *
     *	@return	a writable property wrapping the description attribute
     */
    public final StringProperty descriptionProperty()
    {
        return descriptionProperty;
    }

    /**
     *	Wraps the Description value of the underlying item
     *
     *	@return	a writable property wrapping the description attribute
     */
    public final ReadOnlyStringProperty summaryProperty()
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("summaryProperty()");
        LOGGER.debug("Node type: {}, summary: {}", nodeType, summaryProperty.getReadOnlyProperty().get());
        return summaryProperty;
    }

}
