/*
 * Copyright (C) 2019 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
	Change log
    2.3.0   First version
 */

package uk.co.gardennotebook;

import javafx.beans.property.SimpleStringProperty;
import javafx.beans.property.StringProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import uk.co.gardennotebook.fxbean.ProductBean;
import uk.co.gardennotebook.fxbean.ProductCategoryBean;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.NotebookEntryType;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.CommentBean;
import uk.co.gardennotebook.fxbean.INotebookBean;
//import uk.co.gardennotebook.fxbean.ProductBrandBean;

/**
 *	An aggregate and facade on the various product classes.
 *	Used to show the Product catalogue and also in ShoppingList and Purchase editors
 *
	*	@author	Andy Gegg
	*	@version	2.3.0
	*	@since	2.3.0
 */
final class ProductTreeBean {
    
	private static final Logger LOGGER = LogManager.getLogger();
    
    private ProductCategoryBean productCategoryBean = null;
    private ProductBean productBean = null;
    
	private final NotebookEntryType nodeType;	// PR for Product, PC for ProductCategory
    
    private ObservableList<ProductTreeBean> categoryChildren = null;

    private StringProperty nameProperty = new SimpleStringProperty(this, "productName", "");
	private StringProperty nameDetail_1Property = null;
    private StringProperty descriptionProperty = null;
//	private BooleanProperty plantLikeProperty = null;
	private StringProperty parentProductBrandProperty = new SimpleStringProperty(this, "productBrand", null);

    ProductTreeBean(ProductCategoryBean newVal)
    {
        productCategoryBean = newVal;
        nameProperty = newVal.nameProperty();
        descriptionProperty = newVal.descriptionProperty();
        nameDetail_1Property = new SimpleStringProperty(this, "nameDetail_1", "");
        nodeType = NotebookEntryType.PRODUCTCATEGORY;
    }
    
    ProductTreeBean(ProductBean newVal)
    {
        productBean = newVal;
        nameProperty = newVal.nameProperty();
        descriptionProperty = newVal.descriptionProperty();
        nameDetail_1Property = newVal.nameDetail_1Property();
        if (newVal.hasProductBrand())
        {
            parentProductBrandProperty = newVal.productBrandProperty().get().nameProperty();
        }
        nodeType = NotebookEntryType.PRODUCT;
        categoryChildren = FXCollections.emptyObservableList();
    }

    ObservableList<ProductTreeBean> getChildren()
    {
        switch (nodeType)
        {
            case PRODUCTCATEGORY:
                if (categoryChildren == null)
                {
                    ObservableList<ProductBean> kids = FXCollections.emptyObservableList();
                    try {
                        kids = productCategoryBean.getProduct();
                    } catch (GNDBException ex) {
                        PanicHandler.panic(ex);
                    }
                    categoryChildren = FXCollections.observableArrayList();
                    kids.forEach(x -> categoryChildren.add(new ProductTreeBean(x)));
                }
                break;
            case PRODUCT:
                break;
            default:
                LOGGER.debug("Unexpected node type: {}", nodeType);
        }
        return categoryChildren;
    }
    
    NotebookEntryType getNodeType()
    {
        return nodeType;
    }
    
    ProductCategoryBean getProductCategoryBean()
    {
        return productCategoryBean;
    }

    ProductBean getProductBean()
    {
        return productBean;
    }
    
	INotebookBean getItem()
	{
		switch (nodeType)
		{
			case PRODUCTCATEGORY: return productCategoryBean;
			case PRODUCT: return productBean;
			default:
				LOGGER.debug("getItem(): Unknown node type: {}", nodeType);
				return null;
		}
	}
	
    ObservableList<CommentBean> getComments()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("getComments()");

        switch (nodeType)
        {
            case PRODUCTCATEGORY:
                return productCategoryBean.getComments();
            case PRODUCT:
                return productBean.getComments();
            default:
                LOGGER.debug("Unexpected node type: {}", nodeType);
                return FXCollections.emptyObservableList();
        }
    }
    
    boolean canDelete()
    {
        try {
            switch (nodeType)
            {
                case PRODUCTCATEGORY:
                {
                    return productCategoryBean.canDelete();
                }

                case PRODUCT:
                    return productBean.canDelete();
                default:
                    LOGGER.debug("Unexpected node type: {}", nodeType);
            }
        } catch (GNDBException ex) {
            PanicHandler.panic(ex);
        }
        return false;        
    }

    void delete()
    {
        try {
            switch (nodeType)
            {
                case PRODUCTCATEGORY -> productCategoryBean.delete();
                case PRODUCT -> productBean.delete();
                default -> LOGGER.debug("Unexpected node type: {}", nodeType);
            }
        } catch (GNDBException ex) {
            PanicHandler.panic(ex);
        }
    }

	/**
	*	Wraps the Name value of the underlying item
	*
	*	@return	a writable property wrapping the name attribute
	*/
	public final StringProperty nameProperty()
	{
		return nameProperty;
	}

	/**
	*	Wraps the Description value of the underlying item
	*
	*	@return	a writable property wrapping the description attribute
	*/
	public final StringProperty descriptionProperty()
	{
		return descriptionProperty;
	}

	/**
	*	Wraps the NameDetail_1 value of the underlying item
	*
	*	@return	a writable property wrapping the nameDetail_1 attribute
	*/
	public final StringProperty nameDetail_1Property()
	{
		return nameDetail_1Property;
	}

	/**
	*	Wraps the NameDetail_1 value of the underlying item
	*
	*	@return	a writable property wrapping the nameDetail_1 attribute
	*/
	public final StringProperty productBrandProperty()
	{
		return parentProductBrandProperty;
	}

}
