/*
 * Copyright (C) 2018, 2019, 2021, 2022 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
	Change log
	2.2.0   Better checking for duplicated name
    2.4.0   Support tab-out on comments and created varieties
    2.9.6	When a Diary entry is added/changed, make sure updated comments are shown
    3.0.1	CommentAdder handling moved to separate class
    3.0.2	Check before Cancel if changes have been made
    3.0.4	Use new convenience class NotebookBeanDeleter for deletion.
    		Use new convenience class NotebookBeanCanceller to handle edit cancellation.
    		Use new convenience class EditorCommentTableHandler to handle Comment table construction.
    		Set focus on first field.
 */

package uk.co.gardennotebook;

import java.io.IOException;
import java.util.ResourceBundle;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import javafx.application.Platform;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.event.ActionEvent;
import javafx.event.Event;
import javafx.fxml.FXML;
import javafx.fxml.FXMLLoader;
import javafx.scene.Node;
import javafx.scene.control.Button;
import javafx.scene.control.Tab;
import javafx.scene.control.TabPane;
import javafx.scene.control.TableColumn;
import javafx.scene.control.TableView;
import javafx.scene.control.TextField;
import javafx.scene.layout.GridPane;
import uk.co.gardennotebook.fxbean.*;
import uk.co.gardennotebook.spi.GNDBException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;

/**
	*	Controller class for create/update of PlantSpecies items
	*
	*	@author Andy Gegg
	*	@version	3.0.4
	*	@since	1.0
 */
final class PlantSpeciesEditor extends GridPane implements INotebookLoadable
{
	private static final Logger LOGGER = LogManager.getLogger();
    
	private final PlantSpeciesBean thisValueBean;

	private Consumer<Node> loadSplit;
	private Consumer<Node> clearSplit;
	private BiConsumer<String, Node> loadTab;
	private Consumer<Node> clearTab;

	//	an observable property that the caller can monitor when a new item is being created
	private SimpleObjectProperty<PlantSpeciesBean> newBean;
	private final boolean addingItem;

	//	an observable property that the caller can monitor when an item is deleted
	private SimpleObjectProperty<Object> deletedBean;

	@FXML
	private ResourceBundle resources;

	@FXML
	private Button btnSave;
	@FXML
	private Button btnDelete;
	@FXML
	private Button btnDeleteNote;
	@FXML
	private TextField fldCommonName;
	@FXML
	private TextField fldLatinName;
	@FXML
	private TextField fldDescription;

	@FXML
	private TextField fldUtility;
	@FXML
	private TextField fldHardiness;
	@FXML
	private TextField fldLifeType;
	@FXML
	private TextField fldPlantType;

	@FXML
	private CropRotationGroupCombo parentCropRotationGroup;
	
	@FXML
	private TableView<CommentBean> tbvComment;
	@FXML
	private TableColumn<CommentBean, CommentBean> tbvCommentDate;
	@FXML
	private TableColumn<CommentBean, String> tbvCommentText;

	@FXML
	private TabPane plantNotes;
	
	@FXML
	private TableView<PlantVarietyBean> tbvVarieties;
	@FXML
	private TableColumn<PlantVarietyBean, String> tbvPVCommonName;
	@FXML
	private TableColumn<PlantVarietyBean, String> tbvPVLatinName;
	@FXML
	private TableColumn<PlantVarietyBean, String> tbvPVDesc;
	@FXML
	private TableColumn<PlantVarietyBean, PlantVarietyBean> tbvPVComment;

	//	handles addition of multiple plant varieties
	private final ChangeListener<String> varietyAdder = new ChangeListener<>() {
		@Override
		public void changed(ObservableValue<? extends String> obj, String oldVal, String newVal) {
			if (newVal != null) 
			{
				PlantVarietyBean pib_new = new PlantVarietyBean();
				if (!addingItem)
				{
					pib_new.setPlantSpecies(thisValueBean);
				}
				pib_new.commonNameProperty().addListener(this);
				tbvVarieties.getItems().add(pib_new);
				obj.removeListener(varietyAdder);
			}
		}
	};

	PlantSpeciesEditor()
	{
		this(null);
	}

	PlantSpeciesEditor(PlantSpeciesBean initialVal)
	{
		this.thisValueBean = (initialVal != null ? initialVal : new PlantSpeciesBean());
		this.addingItem = (initialVal == null);
		FXMLLoader fxmlLoader = new FXMLLoader(getClass().getResource("/fxml/PlantSpeciesEditor.fxml"),
			ResourceBundle.getBundle("notebook") );
		fxmlLoader.setRoot(this);
		fxmlLoader.setController(this);
		try {
			fxmlLoader.load();	// NB initialize is called from in here
		} catch (IOException exception) {
			throw new RuntimeException(exception);
		}
	}

	/**
	* Initializes the controller class.
	*/
	@FXML
	private void initialize()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("initialize()");
		
		fldCommonName.textProperty().bindBidirectional(thisValueBean.commonNameProperty());
		fldCommonName.setTextFormatter(MaxLengthTextFormatter.getFormatter(255));
        //  2.2.0
        fldCommonName.focusedProperty().addListener( new ChangeListener<>() {
            @Override
            public void changed(ObservableValue obs, Boolean oldVal, Boolean newVal)
            {
                if (oldVal && !newVal)
                {
                    LOGGER.debug("fldCommonName lost focus");
                    thisValueBean.checkForDuplicateCommonName(fldCommonName.getText());
                }
            }
        });
		fldLatinName.textProperty().bindBidirectional(thisValueBean.latinNameProperty());
		fldDescription.textProperty().bindBidirectional(thisValueBean.descriptionProperty());
		fldUtility.textProperty().bindBidirectional(thisValueBean.utilityProperty());
		fldUtility.setTextFormatter(MaxLengthTextFormatter.getFormatter(45));
		fldHardiness.textProperty().bindBidirectional(thisValueBean.hardinessProperty());
		fldHardiness.setTextFormatter(MaxLengthTextFormatter.getFormatter(45));
		fldLifeType.textProperty().bindBidirectional(thisValueBean.lifeTypeProperty());
		fldLifeType.setTextFormatter(MaxLengthTextFormatter.getFormatter(45));
		fldPlantType.textProperty().bindBidirectional(thisValueBean.plantTypeProperty());
		fldPlantType.setTextFormatter(MaxLengthTextFormatter.getFormatter(45));

		parentCropRotationGroup.valueProperty().bindBidirectional(thisValueBean.cropRotationGroupProperty());

		final EditorCommentTableHandler<PlantSpeciesBean> cth = new EditorCommentTableHandler<>(resources);
		cth.constructCommentTable(tbvComment, tbvCommentDate, tbvCommentText, thisValueBean);

		tbvPVCommonName.setCellValueFactory(cdf -> cdf.getValue().commonNameProperty());
		tbvPVCommonName.setCellFactory( x -> new TextFieldTableCellTabOut<>() );
		tbvPVLatinName.setCellValueFactory(cdf -> cdf.getValue().latinNameProperty());
		tbvPVLatinName.setCellFactory( x -> new TextFieldTableCellTabOut<>() );
		tbvPVDesc.setCellValueFactory(cdf -> cdf.getValue().descriptionProperty());
		tbvPVDesc.setCellFactory( x -> new TextFieldTableCellTabOut<>() );
		tbvPVComment.setCellValueFactory(cdf -> new SimpleObjectProperty<>(cdf.getValue()));
		tbvPVComment.setCellFactory(x -> new EditorCommentTableCell<>(resources));
		try {
			tbvVarieties.getItems().setAll(thisValueBean.getPlantVariety());
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}
		tbvVarieties.setColumnResizePolicy(NotebookResizer.using(tbvVarieties));
		ObservableList<PlantVarietyBean> pvBeans = FXCollections.emptyObservableList();
		try {
			pvBeans = thisValueBean.getPlantVariety();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}

		for (PlantVarietyBean pvb : pvBeans)
		{
			pvb.setSaveRequired(true);
		}
		pvBeans = null;
		PlantVarietyBean pvb = new PlantVarietyBean();
		if (!this.addingItem)
		{
			pvb.setPlantSpecies(thisValueBean);
		}
		pvb.commonNameProperty().addListener(varietyAdder);
		tbvVarieties.getItems().add(pvb);
		
		ObservableList<PlantNoteBean> pnbList = FXCollections.emptyObservableList();
		try {
			pnbList = thisValueBean.getPlantNote();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}
		for (PlantNoteBean pnb : pnbList)
		{
			PlantNoteTab pnt = new PlantNoteTab(pnb);
			Tab tab = new Tab("", pnt);
			plantNotes.getTabs().add(tab);
			tab.textProperty().bind(pnt.titleProperty());
		}

		thisValueBean.setSaveRequired(true);
		
		btnSave.disableProperty().bind(fldCommonName.textProperty().isNull().or(fldCommonName.textProperty().isEmpty()).or(thisValueBean.duplicateCommonNameProperty()));   //  2.2.0
		try {
			btnDelete.setDisable(addingItem || !(this.thisValueBean.canDelete()));
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}

		Platform.runLater(() -> {fldCommonName.requestFocus(); fldCommonName.deselect();});

		LOGGER.traceExit(log4jEntryMsg);
	}

	@Override
	public void setLoadSplit(Consumer<Node> code)
	{
		loadSplit = code;
	}

	@Override
	public void setClearSplit(Consumer<Node> code)
	{
		clearSplit = code;
	}

	@Override
	public void setLoadTab(BiConsumer<String, Node> code)
	{
		loadTab = code;
	}

	@Override
	public void setClearTab(Consumer<Node> code)
	{
		clearTab = code;
	}

	@Override
	public void clearUpOnClose(Event e)
	{
		if (isCancelDenied())
		{
			e.consume();
			return;
		}

		thisValueBean.cancelEdit();
		ObservableList<PlantVarietyBean> pvBeans = FXCollections.emptyObservableList();
		try {
			pvBeans = thisValueBean.getPlantVariety();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}

		for (PlantVarietyBean pvb : pvBeans)
		{
			pvb.cancelEdit();
		}
	}

	@FXML
	private void btnCancelOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnCancelOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("Bean is null!");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

		if (isCancelDenied())
		{
			return;
		}

		thisValueBean.cancelEdit();
		ObservableList<PlantVarietyBean> pvBeans = FXCollections.emptyObservableList();
		try {
			pvBeans = thisValueBean.getPlantVariety();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}

		for (PlantVarietyBean pvb : pvBeans)
		{
			pvb.cancelEdit();
		}
		
		ObservableList<PlantNoteBean> pvNotes = FXCollections.emptyObservableList();
		try {
			pvNotes = thisValueBean.getPlantNote();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}

		for (PlantNoteBean pvn : pvNotes)
		{
			pvn.cancelEdit();
		}
				
		clearTab.accept(this);
		LOGGER.traceExit(log4jEntryMsg);
	}

	/**
	 * Check if user really wants to quit if changes have been made
	 *
	 * @return	true	user does NOT want to quit
	 */
	private boolean isCancelDenied()
	{
		if (thisValueBean.needSave())
		{
			final NotebookEditorCanceller<PlantSpeciesBean> cancelChecker = new NotebookEditorCanceller<>(resources);
			return cancelChecker.isCancelDenied(thisValueBean);
		}
		return false;
	}

	@FXML
	private void btnDeleteOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnDeleteOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("Bean is null!");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}
		final NotebookBeanDeleter<PlantSpeciesBean> deleterImpl = new NotebookBeanDeleter<>(resources);
		if (deleterImpl.deleteItemImpl(thisValueBean))
		{
			deletedBeanProperty().setValue(new Object());
			clearTab.accept(this);
		}

		LOGGER.traceExit(log4jEntryMsg);
	}

	@FXML
	private void btnSaveOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnSaveOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

		// check everything CAN be saved!
		if (!thisValueBean.canSave())
			return;
		try {
			thisValueBean.save();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}

		for (PlantVarietyBean pvb : tbvVarieties.getItems())
		{
			LOGGER.debug("PlantVariety loop: for: {}", pvb.toString());
            if (addingItem)
            {
                pvb.setPlantSpecies(thisValueBean);
            }
			if (!pvb.canSave())
			{
                LOGGER.debug("PlantVariety loop: cannot save");
				continue;
			}
			if (!pvb.needSave())
			{
                LOGGER.debug("PlantVariety loop: save not needed");
				continue;
			}
			pvb.setPlantSpecies(thisValueBean);
			try {
				pvb.save();
			} catch (GNDBException ex) {
				PanicHandler.panic(ex);
			}
		}

		for (Tab tab : plantNotes.getTabs())
		{
			LOGGER.debug("PlantNote loop: for: {}", tab.getText());
			PlantNoteBean pnb = ((PlantNoteTab)(tab.getContent())).getPlantNote();
			LOGGER.debug("PlantNote loop: plant note: {}", pnb);
            if (addingItem)
            {
    			LOGGER.debug("PlantNote loop: setting PlantSpecies");
                pnb.setPlantSpecies(thisValueBean);
            }
			if (!pnb.canSave())
			{
                LOGGER.debug("PlantNote loop: cannot save");
				continue;
			}
			if (!pnb.needSave())
			{
                LOGGER.debug("PlantNote loop: save not needed");
				continue;
			}
			try {
				pnb.save();
			} catch (GNDBException ex) {
				PanicHandler.panic(ex);
			}
		}

		newBeanProperty().setValue(thisValueBean);
		clearTab.accept(this);
		LOGGER.traceExit(log4jEntryMsg);
	}


	@FXML
	private void btnAddNoteOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnAddNoteOnAction()");
		PlantNoteTab pnt = new PlantNoteTab(thisValueBean);
		Tab tab = new Tab("", pnt);
		plantNotes.getTabs().add(tab);
		tab.textProperty().bind(pnt.titleProperty());
		plantNotes.getSelectionModel().select(tab);		
	}
	
	@FXML
	private void btnDeleteNoteOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnDeleteNoteOnAction()");
		
		Tab tab = plantNotes.getSelectionModel().getSelectedItem();
		if (tab == null)
			return;
		PlantNoteBean pnb = ((PlantNoteTab)(tab.getContent())).getPlantNote();
		
		boolean canDelete = true;
		try {
			canDelete = pnb.canDelete();
		} catch (GNDBException ex) {
				PanicHandler.panic(ex);
		}
		if (!canDelete)
			return;
		
		try {
			pnb.delete();
		} catch (GNDBException ex) {
				PanicHandler.panic(ex);
		}
		plantNotes.getTabs().remove(tab);
	}
	
	SimpleObjectProperty<PlantSpeciesBean> newBeanProperty()
	{
		if (newBean == null)
		{
			newBean = new SimpleObjectProperty<>();
		}
		return newBean;
	}

	SimpleObjectProperty<Object> deletedBeanProperty()
	{
		if (deletedBean == null)
		{
			deletedBean = new SimpleObjectProperty<>();
		}
		return deletedBean;
	}


}
