/*
 * Copyright (C) 2018 Andrew Gegg
 
 *
 *   This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software:  you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */
package uk.co.gardennotebook;

import java.text.NumberFormat;
import java.text.ParsePosition;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.scene.control.TextField;
import javafx.scene.control.TextFormatter;
import javafx.scene.layout.VBox;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import uk.co.gardennotebook.fxbean.INotebookBean;

/**
 *	Utility class to simplify entering integer values into TableCells
 * 
 * @author andy
 */
class NotebookIntegerCellType<S extends INotebookBean> extends NotebookEditCellType<S, Integer>
{
	private static final Logger LOGGER = LogManager.getLogger();
	
	private static final NumberFormat intFmt = NumberFormat.getIntegerInstance();

	private final int minValue;
	private final boolean isMandatory;
	
	/**
	 */	
	NotebookIntegerCellType()
	{
		this(0, false);
	}
	
	/**
	 */	
	NotebookIntegerCellType(final int minValue, final boolean isMandatory)
	{
		this.minValue = minValue;
		this.isMandatory = isMandatory;
	}
	
	@Override
	void updateViewMode(S rowItem, Integer cellItem)
	{
		LOGGER.debug("NotebookIntegerCellType: updateViewMode(): item: {}", cellItem);
		LOGGER.debug("NotebookIntegerCellType: updateViewMode(): rowItem: {}", rowItem);
		setGraphic(null);
		setText(null);
		
		if (isEditing() && mayEdit(rowItem, cellItem))
		{
			VBox vBox = new VBox();
			TextField dp = new TextField(cellItem.toString());
			dp.setEditable(true);
			dp.setTextFormatter(new TextFormatter<Integer>(c -> {
				if ( c.getControlNewText().isEmpty() )
				{
					return c;
				}

			    ParsePosition parsePosition = new ParsePosition( 0 );
				Number parsedVal = intFmt.parse(c.getControlNewText(), parsePosition);

				if ( parsedVal == null || parsePosition.getIndex() < c.getControlNewText().length() )
				{
					return null;	// reject change
				}
				else
				{
					return c;
				}

			}));
			dp.setOnAction(ev -> {
				LOGGER.debug("NotebookIntegerCellType: setOnAction: selection: {}", dp.getText());
				if (dp.getText().isEmpty())
				{
					if (isMandatory) return;
					commitEdit(null);
					return;
				}
				ParsePosition parsePosition = new ParsePosition(0);
				Number setVal = intFmt.parse(dp.getText(), parsePosition);
				if ( setVal == null || parsePosition.getIndex() < dp.getText().length() )
				{
					return;
				}
				if (setVal.intValue() < minValue)
				{
					return;
				}
				commitEdit(setVal.intValue());
			});
			vBox.getChildren().add(dp);
			setGraphic(vBox);
		}
		else
		{
			setText(intFmt.format(cellItem));
		}
		
	}
	
}
