/*
 * Copyright (C) 2018-2020, 2023 Andrew Gegg
 
 *
 *   This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software:  you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
    2.8.0   Add convenience overridable method in the onAction handler.
    3.1.2	Improve implementation of onAction hooks
 */

package uk.co.gardennotebook;

import javafx.scene.control.TableCell;
import uk.co.gardennotebook.fxbean.INotebookBean;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 *	Simplifies provision of custom editors for notebook entities.
 *
 * @param <S> the type of the TableView
 * @param <T> the type of the TableColumn (i.e. the type of item in the cell)
 * 
 * @author Andy Gegg
 * @version 3.1.2
 * @since 1.1.0
 */
abstract class NotebookEditCellType<S extends INotebookBean, T> extends TableCell<S, T>
{
	private static final Logger LOGGER = LogManager.getLogger();


/**
 * 
 */	
	NotebookEditCellType()
	{
	}
	
	@Override
	protected void updateItem(T item, boolean empty)
	{
		super.updateItem(item, empty);
		if (empty || item == null)
		{
			setGraphic(null);
			setText(null);
			return;
		}
		updateViewMode(getRowItem(), getItem());
	}			

	@Override
	public void startEdit() {
		super.startEdit();
		updateViewMode(getRowItem(), getItem());
	}

	@Override
	public void cancelEdit() {
		super.cancelEdit();
		updateViewMode(getRowItem(), getItem());
	}

	/**
	 * Returns the item represented by the current <em>row</em> of the table.
	 * This facilitates edit modes which require access to multiple fields within the current item.
	 * 
	 * @return the item represented by the current table row
	 */	
	private S getRowItem()
	{
		return getTableView().getSelectionModel().getSelectedItem();
	}

	/**
	 * Implement this function to provide the actual editing experience
	 */
	abstract void updateViewMode(S rowItem, T cellItem);
	
	/**
	 * Override this to specify if the cell may be edited.
	 *
	 * @return true if the cell may be edited, false otherwise
	 */
	protected boolean mayEdit(S rowItem, T cellItem)
	{
		return true;
	}

	/**
	 * A hook which is called just before the edit is committed in the onAction event handler.
	 * The value of cellItem will be the old value.
	 *
	 * @implNote
	 * It is the responsibility of the implementing class to call this hook in updateViewMode().
	 *
	 * @param rowItem The current row in the table being edited.
	 * @param cellItem	The value of the cellItem after committing the edit.
	 * @return Return false if the commit is to be cancelled, else true.
	 */
	protected boolean onActionPreCommit(S rowItem, T cellItem)
    {
		return true;
	}

	/**
	 * A hook which is called just after the edit is committed in the onAction event handler.
	 * The value of cellItem will be the new, committed value.
	 *
	 * @implNote
	 * It is the responsibility of the implementing class to call this hook in updateViewMode().
	 *
	 * @param rowItem The current row in the table being edited.
	 * @param cellItem	The val;ue of the cellItem after committing the edit.
	 */
	protected void onActionPostCommit(S rowItem, T cellItem)
	{}
}
