/*
 * Copyright (C) 2018-2020, 2023 Andrew Gegg
 
 *
 *   This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software:  you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
    2.8.0   Include overridable convenience method in the onAction handler.
    3.1.2	Implement changed onAction hooks.
    			(this is for the PurchaseEditor setting watch-for dates).
 */

package uk.co.gardennotebook;

import java.util.function.BiFunction;
import java.util.function.Function;
import javafx.scene.layout.VBox;
import uk.co.gardennotebook.fxbean.INotebookBean;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 *	Simplifies table cell editing where a notebook entity combo box is required.
 *
 * @param <S> the type of the TableView
 * @param <T> the type of the TableColumn (i.e. the type of item in the cell)
 * @param <C> the type of ComboBox to be used
 * 
 * @author Andy Gegg
 * @version 3.1.2
 * @since 1.1.0
 */
class NotebookDropDownCellType<S extends INotebookBean, 
									T extends INotebookBean, 
									C extends DiaryEntryCombo<T>> extends NotebookEditCellType<S, T>
{
	private static final Logger LOGGER = LogManager.getLogger();

	private final BiFunction<S, T, C> comboSupplier;
	private final Function<T, String> getName;

/**
 * @param comboSupplier	return a suitable ComboBox for the cell type.
 *		The value passed to the call is the current value of the cell, to initialise the combo box
 * @param getName	return a string representation of the cell content.
 *		Used when the cell is not being edited.
 */	
	NotebookDropDownCellType(BiFunction<S, T, C> comboSupplier, Function<T, String> getName)
	{
		this.comboSupplier = comboSupplier;
		this.getName = getName;
	}
	
	@Override
	void updateViewMode(S rowItem, T cellItem)
	{
		LOGGER.debug("DropDownCellType: updateViewMode(): item: {}", cellItem);
		LOGGER.debug("DropDownCellType: updateViewMode(): rowItem: {}", rowItem);
		setGraphic(null);
		setText(null);
		
		if (isEditing() && mayEdit(rowItem, cellItem) && comboSupplier != null)
		{
			VBox vBox = new VBox();
			C cb = comboSupplier.apply(rowItem, cellItem);
			cb.setEditable(true);
			cb.setOnAction(ev -> {
				LOGGER.debug("DropDownCellType: setOnAction: selection: {}", cb.getSelectionModel().getSelectedItem());
//                onActionExtra(rowItem, cellItem);
				if (!onActionPreCommit(rowItem, cellItem))
				{
					cancelEdit();
					return;
				}
				commitEdit(cb.getValue());
				onActionPostCommit(rowItem, cellItem);
			});
			vBox.getChildren().add(cb);
			setGraphic(vBox);
		}
		else
		{
			setText(cellItem == null? "" : getName.apply(cellItem));
		}
		
	}
    
}
