/*
 *
 *  Copyright (C) 2022 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook;

import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.scene.control.TableColumn;
import javafx.scene.control.TableView;
import javafx.scene.text.Text;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import uk.co.gardennotebook.fxbean.CommentBean;
import uk.co.gardennotebook.fxbean.INotebookBean;

import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.ResourceBundle;

/**
 * A convenience class to handle the common task of setting up a table of comments in an editor.
 *
 * @param resources the resource bundle containing the text of the confirmation question for the user
 *
 * @param <T>   type of object handled by the calling editor.
 *
 *	@author Andy Gegg
 *	@version	3.0.4
 *	@since	3.0.4
 */
record EditorCommentTableHandler<T extends INotebookBean>(ResourceBundle resources)
{
//    private static final Logger LOGGER = LogManager.getLogger();

    void constructCommentTable( TableView<CommentBean> tbvComment,
                                TableColumn<CommentBean, CommentBean> tbvCommentDate,
                                TableColumn<CommentBean, String> tbvCommentText,
                                T thisValueBean)
    {
        tbvCommentDate.setCellValueFactory(e-> new SimpleObjectProperty<>( e.getValue()));
        tbvCommentDate.setCellFactory(x -> new EditorCommentDateTableCell(resources));
        tbvCommentText.setCellValueFactory(e->e.getValue().commentProperty());
        tbvCommentText.setCellFactory(x -> new EditorCommentTextTableCell(resources));

        //  set column resize behaviour
        tbvComment.setColumnResizePolicy(NotebookResizer.using(tbvComment));

        //  load comments into table
        tbvComment.getItems().setAll(thisValueBean.getComments());

        //  add empty row to enter new comment
        CommentBean cb_new = new CommentBean(thisValueBean);

        //  handle adding new comment
        ChangeListener<String> commentAdder = new EditorCommentAdder<>(thisValueBean, tbvComment);
        cb_new.commentProperty().addListener(commentAdder);
        tbvComment.getItems().add(cb_new);

        //  calculate optimal size for date column
        CommentBean cb_one = tbvComment.getItems().get(0);
        Text t = new Text(cb_one.getDate().format(DateTimeFormatter.ofLocalizedDate(FormatStyle.SHORT)));
        double wid = t.getLayoutBounds().getWidth();
        tbvCommentDate.setPrefWidth(wid+10);

        //  select first comment row when user tabs into table
        tbvComment.focusedProperty().addListener((obj, wasFocused, isFocused)->{
            if (!wasFocused && isFocused)
            {
                tbvComment.getSelectionModel().selectFirst();
            }
            else if (wasFocused && !isFocused)
            {
                tbvComment.getSelectionModel().clearSelection();
            }
        });
    }

}
