/*
 *
 *  Copyright (C) 2022 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook;

import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.Property;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.property.StringProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.CommentBean;
import uk.co.gardennotebook.fxbean.INotebookBean;
import uk.co.gardennotebook.fxbean.WeatherBean;
import uk.co.gardennotebook.fxbean.WeatherConditionBean;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.INotebookEntry;
import uk.co.gardennotebook.spi.NotebookEntryType;

import java.time.LocalDate;
import java.util.Objects;

/**
 *	A specific implementation of DiaryBean for Weather diary entries.
 *
 * @author Andy Gegg
 *	@version	3.0.5
 *	@since	3.0.5
 */
final class DiaryBeanWeather extends DiaryBean
{
    private static final Logger LOGGER = LogManager.getLogger();

    private final WeatherBean weather;

    private Integer itemKey = 0;

    final ObjectProperty<LocalDate> dateProperty;

    private final ObjectProperty<WeatherConditionBean> mainTitleProperty;	//	the title - the Weather Condition
    private final StringBinding mainTitleText;
    private final StringProperty subTitleProperty;	//	the Severity of the Weather
    private final StringBinding subTitleText;
    private final Property<?> varietyProperty = null;	// null
    private final StringBinding varietyText;
    private final Property<?> detailProperty = null;    //  null
    private final StringBinding detailText;

    private final SimpleObjectProperty<ObservableList<CommentBean>> comments =
            new SimpleObjectProperty<>(this, "comments", FXCollections.emptyObservableList());

    DiaryBeanWeather(WeatherBean weatherBean)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("DiaryBeanWeather: constructor(): WeatherBean:{}", weatherBean);

        weather = weatherBean;
        itemKey = weather.getKey();

        dateProperty = weather.dateProperty();

        mainTitleProperty = weather.weatherConditionProperty();
        mainTitleText = new StringBinding()
        {
            {
                super.bind(mainTitleProperty);
            }
            @Override
            protected String computeValue()
            {
                    return weather.getWeatherCondition().getName();
            }

            @Override
            public void dispose()
            {
                super.unbind(mainTitleProperty);
            }
        };
        subTitleProperty = weather.severityProperty();
        subTitleText = new StringBinding()
        {
            {
                super.bind(subTitleProperty);
            }
            @Override
            protected String computeValue()
            {
                return weather.getSeverity();
            }

            @Override
            public void dispose()
            {
                super.unbind(subTitleProperty);
            }
        };

//        varietyProperty = null;
        varietyText = new StringBinding()
        {
            @Override
            protected String computeValue()
            {
                return "";
            }

        };

//        detailProperty = null;
        detailText = new StringBinding()
        {
           @Override
            protected String computeValue()
            {
                return "";
            }

        };

        LOGGER.traceExit(log4jEntryMsg);
    }

    @Override
    INotebookBean getItem()
    {
        return weather;
    }

    @Override
    INotebookEntry getBaseItem()
    {
        return weather.get().orElse(null);
    }

    @Override
    Integer getKey()
    {
        return itemKey;
    }

    @Override
    NotebookEntryType getItemType()
    {
        return NotebookEntryType.WEATHER;
    }

    @Override
    boolean canDelete() throws GNDBException
    {
        return weather.canDelete();
    }

    /**
     * The date of this diary entry
     *
     * @return	the (value of) the dateProperty
     */
    @Override
    LocalDate getDate()
    {
        return dateProperty.getValue();
    }

    @Override
    void setDate(LocalDate date)
    {
        dateProperty().setValue(date);
    }

    @Override
    ObjectProperty<LocalDate> dateProperty()
    {
        return dateProperty;
    }

    /**
     * The title for this Diary entry - here, the Weather Condition
     *
     * @return	the (value of) the mainTitleProperty
     */
    @Override
    StringBinding getMainTitleText()
    {
        return mainTitleText;
    }

    @Override
    ObjectProperty<WeatherConditionBean> mainTitleProperty()
    {
        return mainTitleProperty;
    }

    /**
     * The first detail for this Diary entry - here, the Severity
     *
     * @return	the (value of) the subTitleProperty
     */
    @Override
    StringBinding getSubTitleText()
    {
        return subTitleText;
    }

    @Override
    StringProperty subTitleProperty()
    {
        return subTitleProperty;
    }

    /**
     * The second detail for this Diary entry - here, null
     *
     * @return	the (value of) the varietyProperty
     */
    @Override
    StringBinding getVarietyText()
    {
        return varietyText;
    }

    @Override
    Property<?> varietyProperty()
    {
        return varietyProperty;
    }

    /**
     * The third detail for this Diary entry - here, null
     *
     * @return	the (value of) the detailProperty
     */
    @Override
    StringBinding getDetailText()
    {
        return detailText;
    }

    @Override
    Property<?> detailProperty()
    {
        return detailProperty;
    }

    private ObservableList<CommentBean> getCommentList()
    {
        return weather.getComments();
    }

    @Override
    ObservableList<CommentBean> getComments()
    {
        LOGGER.debug("getComments(): getValue(): {}", commentProperty().getValue());
        return commentProperty().getValue();
    }

    @Override
    SimpleObjectProperty<ObservableList<CommentBean>> commentProperty()
    {
        LOGGER.debug("commentProperty(): getCommentList(): {}", getCommentList());
        comments.set(getCommentList());	//	2.9.6
        return comments;
    }

    @Override
    /**
     * Implement the comparison for sorting.
     * Items are first sorted in date order.
     * if same date, then by item type
     *      -   weather then husbandry, then affliction,
     * 		-	then groundwork, then wildlife,
     * 		-	then purchases and items within purchases,
     * 		-	then sales and sale items within sales
     * 		-	and journal entries
     * 	Items of same type on same day are sorted by key to give a stable ordering
     * 	    (this will usually correspond to sorting by order of creation).
     */
    public int compareTo(DiaryBean o)
    {
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("compareTo(): this: {}. other: {}", this.toString(), o.toString());

//        LOGGER.debug("this.dateProperty: {}", this.dateProperty.get());
//        LOGGER.debug("other.dateProperty: {}", o.dateProperty().get());

        if (!this.dateProperty.get().equals(o.dateProperty().get()))
        {//sort on date as primary key
            return this.dateProperty.get().compareTo(o.dateProperty().get());
        }

        return switch (o.getItemType())
                {
                    case WEATHER -> this.itemKey.compareTo(o.getKey()); //  stable ordering for entries of same type
                    default -> -1;
                };
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        DiaryBeanWeather that = (DiaryBeanWeather) o;
        return weather.equals(that.weather) && itemKey.equals(that.itemKey) && dateProperty.equals(that.dateProperty) && mainTitleProperty.equals(that.mainTitleProperty) && mainTitleText.equals(that.mainTitleText) && subTitleProperty.equals(that.subTitleProperty) && subTitleText.equals(that.subTitleText) && Objects.equals(varietyProperty, that.varietyProperty) && Objects.equals(varietyText, that.varietyText) && Objects.equals(detailProperty, that.detailProperty) && Objects.equals(detailText, that.detailText) && Objects.equals(comments, that.comments);
    }

    @Override
    public int hashCode()
    {
        return Objects.hash(super.hashCode(), weather, itemKey, dateProperty, mainTitleProperty, mainTitleText, subTitleProperty, subTitleText, varietyProperty, varietyText, detailProperty, detailText, comments);
    }

    @Override
    public  String toString()
    {
        return ("DiaryBeanWeather of: " + weather.toString());
    }

}
