/*
 * Copyright (C) 2018-2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
    2.2.5   Guard against occasional NPE on item delete removing base listeners
    2.9.6	When a Diary entry is added/changed, make sure updated comments are shown
    3.0.0	Support 'watch for' entries on PurchaseItems
    3.0.1	Fix bug whereby 'watch for' items got a duplicate Comment
    3.0.2	Convenience features on ToDoList handling in the UI
    3.0.4	Comment handling.  (There is no current mechanism for adding Comments to ToDOList items - but it keeps everything in step)
*/

package uk.co.gardennotebook.fxbean;

import javafx.beans.property.*;
import uk.co.gardennotebook.spi.*;
import uk.co.gardennotebook.util.StoryLineTree;
import java.util.Optional;
import java.util.List;
import java.util.ArrayList;
import java.beans.PropertyChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.beans.value.ChangeListener;

import javafx.collections.FXCollections;
import javafx.collections.ObservableList;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
//import org.apache.logging.log4j.Level;

import java.time.*;
import java.util.function.Consumer;

import javafx.beans.property.SimpleObjectProperty;

/**
	*	A list of tasks to be done.  Reminders are scheduled, possibly repeating and will create items here when they are due.<BR>
@apiNote
This list is not user updated - users create Reminders which generate ToDo items
	*
	*	@author	Andy Gegg
	*	@version	3.0.4
	*	@since	1.0
*/
final public class ToDoListBean implements INotebookBean
{
	private static final Logger LOGGER = LogManager.getLogger();

	private IToDoList baseItem = null;

	private Integer itemKey = 0;
	private boolean newItem = false;
	private boolean explicitSave = false;
	private final SimpleBooleanProperty saveRequiredProperty = new SimpleBooleanProperty(this, "saveRequired", explicitSave);
	private IToDoListBuilder explicitBuilder = null;

		// handle changes to the base item itself
	private PropertyChangeListener baseItemDeleted;
	private PropertyChangeListener baseItemReplaced;

	private final SimpleObjectProperty<PlantSpeciesBean> parentPlantSpeciesProperty = new SimpleObjectProperty<>(this, "plantSpecies", null);
	private final ChangeListener<PlantSpeciesBean> plantSpeciesIdListener = this::onPlantSpeciesIdChange;
	private final ReadOnlyBooleanWrapper hasParentPlantSpeciesProperty = new ReadOnlyBooleanWrapper(this, "hasPlantSpecies", false);
	private final SimpleObjectProperty<PlantVarietyBean> parentPlantVarietyProperty = new SimpleObjectProperty<>(this, "plantVariety", null);
	private final ChangeListener<PlantVarietyBean> plantVarietyIdListener = this::onPlantVarietyIdChange;
	private final ReadOnlyBooleanWrapper hasParentPlantVarietyProperty = new ReadOnlyBooleanWrapper(this, "hasPlantVariety", false);
	private final SimpleObjectProperty<HusbandryClassBean> parentHusbandryClassProperty = new SimpleObjectProperty<>(this, "husbandryClass", null);
	private final ChangeListener<HusbandryClassBean> husbandryClassIdListener = this::onHusbandryClassIdChange;
	private final ReadOnlyBooleanWrapper hasParentHusbandryClassProperty = new ReadOnlyBooleanWrapper(this, "hasHusbandryClass", false);
	private final SimpleObjectProperty<GroundworkActivityBean> parentGroundworkActivityProperty = new SimpleObjectProperty<>(this, "groundworkActivity", null);
	private final ChangeListener<GroundworkActivityBean> groundWorkActivityIdListener = this::onGroundWorkActivityIdChange;
	private final ReadOnlyBooleanWrapper hasParentGroundworkActivityProperty = new ReadOnlyBooleanWrapper(this, "hasGroundworkActivity", false);

	/*
	*	The Husbandry editor can set up a ToDoList entry intended as a 'watch for this to happen' (e.g. on sowing, add a 'watch for germination').<BR>
This field is the id of that parent/ancestor Husbandry event so when this ToDoList entry is processed the appropriate Storyline links can be set up.
	*/
	private final SimpleObjectProperty<HusbandryBean> parentHusbandryProperty = new SimpleObjectProperty<>(this, "husbandry", null);
	private final ChangeListener<HusbandryBean> husbandryIdListener = this::onHusbandryIdChange;
	private final ReadOnlyBooleanWrapper hasParentHusbandryProperty = new ReadOnlyBooleanWrapper(this, "hasHusbandry", false);

	/*
	*	The Purchase editor can set up a ToDoList entry intended as a 'watch for this to happen' (e.g. for a seed purchase, watch for sowing).<BR>
This field is the id of that parent/ancestor PurchaseItem event so when this ToDoList entry is processed the appropriate Storyline links can be set up.
	*/
	private final SimpleObjectProperty<PurchaseItemBean> parentPurchaseItemProperty = new SimpleObjectProperty<>(this, "purchaseItem", null);
	private final ChangeListener<PurchaseItemBean> purchaseItemIdListener = this::onPurchaseItemIdChange;
	private final ReadOnlyBooleanWrapper hasParentPurchaseItemProperty = new ReadOnlyBooleanWrapper(this, "purchaseItem", false);

	private final SimpleStringProperty descriptionProperty = new SimpleStringProperty(this, "description", "");
	private final ChangeListener<String> descriptionListener = this::onDescriptionChange;
	private final ReadOnlyObjectWrapper<LocalDateTime> lastUpdatedProperty = new ReadOnlyObjectWrapper<>(this, "lastUpdated", LocalDateTime.now());
	private final ReadOnlyObjectWrapper<LocalDateTime> createdProperty = new ReadOnlyObjectWrapper<>(this, "created", LocalDateTime.now());
	private ReadOnlyBooleanWrapper canDeleteProperty = null;
	private final ReadOnlyBooleanWrapper hasAncestorProperty = new ReadOnlyBooleanWrapper(this, "hasAncestor", false);
	private final ReadOnlyBooleanWrapper hasDescendantProperty = new ReadOnlyBooleanWrapper(this, "hasDescendant", false);

	private ReadOnlyBooleanWrapper isNewProperty = new ReadOnlyBooleanWrapper(this, "isNew", newItem);	//	2.9.6

	private BeanCommentHandler<IToDoList> beanCommentHandler;	//	2.9.6
	private final ReadOnlyStringWrapper commentTextProperty = new ReadOnlyStringWrapper(this, "commentText", "");

	/**
	*	Construct an 'empty' Bean.  Set the various property values then call save() to create the new ToDoListBean
	*/
	public ToDoListBean()
	{
		this(null);
	}
	/**
	*	Construct a Bean wrapping the given ToDoList
	*	If the parameter is null a new 'empty' Bean will be constructed
	*
	*	@param	initialValue	the ToDoList to wrap.  If null an 'empty' bean will be constructed
	*
	*/
	public ToDoListBean(final IToDoList initialValue)
	{
		ChangeListener<Boolean> saveRequiredListener = (obs, old, nval) -> {
			if (nval && !explicitSave)
			{
				explicitSave = true;
				ITrug server = TrugServer.getTrugServer().getTrug();
				explicitBuilder = server.getToDoListBuilder(baseItem);
			}
			if (!nval && explicitSave && (baseItem != null))
			{
				explicitSave = false;
				explicitBuilder = null;
			}
		};

		saveRequiredProperty.addListener(saveRequiredListener);

		if(initialValue == null)
		{
			newItem = true;
			//	add the listeners BEFORE setting values, or default values never get sent to the builder!
			addListeners();
			setDefaults();
			saveRequiredProperty.set(true);
			return;
		}

		baseItem = initialValue;

		itemKey = baseItem.getKey();

		newItem = false;
		setValues();

		addListeners();
		declareBaseListeners();
		addBaseListeners();
	}

	/**
	*	Returns all ToDoList items wrapped as ToDoListBean.
	*
	*	@return	a collection of ToDoListBean beans
	*
	*	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	public static ObservableList<ToDoListBean> fetchAll() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetchAll()");
		ITrug server = TrugServer.getTrugServer().getTrug();
		IToDoListLister gal = server.getToDoListLister();
		List<ToDoListBean> ll = gal.fetch().stream()
					.collect(ArrayList::new, (c, e) -> c.add(new ToDoListBean(e)), ArrayList::addAll);
		LOGGER.traceExit();
		return FXCollections.observableArrayList(ll);
	}

	/**
	*	Returns the underlying ToDoList, if present
	*
	*	@return	the underlying ToDoList, if present
	*/
	public Optional<IToDoList> get()
	{
		return getValue();
	}

	/**
	*	Returns the underlying ToDoList if present
	*
	*	@return	the underlying ToDoList, if present
	*/
	public Optional<IToDoList> getValue()
	{
		return Optional.ofNullable(baseItem);
	}

	@Override
	public NotebookEntryType getType()
	{
		return NotebookEntryType.TODOLIST;
	}

	@Override
	public Integer getKey()
	{
		return itemKey;
	}

	@Override
	public boolean sameAs(final INotebookBean other)
	{
		if (other == null || ((ToDoListBean)other).baseItem == null || baseItem == null)
		{
			return false;
		}
		if (other.getType() != NotebookEntryType.TODOLIST)
		{
			return false;
		}
		return baseItem.sameAs(((ToDoListBean)other).baseItem);
	}

	@Override
	public boolean isNew()
	{
		return isNewProperty().get();
	}

	@Override
	public ReadOnlyBooleanProperty isNewProperty()
	{
		if (isNewProperty == null)
		{
			isNewProperty = new ReadOnlyBooleanWrapper(this, "isNew", newItem);
		}
		return isNewProperty.getReadOnlyProperty();
	}

	@Override
	public boolean canDelete() throws GNDBException
	{
		return canDeleteProperty().get();
	}

	@Override
	public ReadOnlyBooleanProperty canDeleteProperty() throws GNDBException
	{
		if (canDeleteProperty == null)
		{
			ITrug server = TrugServer.getTrugServer().getTrug();
			boolean canDel = server.getToDoListBuilder(baseItem).canDelete();
			canDeleteProperty = new ReadOnlyBooleanWrapper(this, "canDelete", canDel);
		}
		return canDeleteProperty.getReadOnlyProperty();
	}

	@Override
	public boolean hasAncestor() throws GNDBException
	{
		//	ToDoList items do not participate in story lines
		return false;
	}

	@Override
	public ReadOnlyBooleanProperty hasAncestorProperty() throws GNDBException
	{
		//	ToDoList items do not participate in story lines
		return hasAncestorProperty.getReadOnlyProperty();
	}	//	hasAncestorProperty()

	@Override
	public StoryLineTree<? extends INotebookBean> getAncestors() throws GNDBException
	{
		//	ToDoList items do not participate in story lines
			return StoryLineTree.emptyTree();
	}	//	getAncestors()

	@Override
	public boolean hasDescendant() throws GNDBException
	{
		//	ToDoList items do not participate in story lines
		return false;
	}

	@Override
	public ReadOnlyBooleanProperty hasDescendantProperty() throws GNDBException
	{
		//	ToDoList items do not participate in story lines
		return hasDescendantProperty.getReadOnlyProperty();
	}	//	hasDescendantProperty()

	@Override
	public StoryLineTree<? extends INotebookBean> getDescendants() throws GNDBException
	{
		//	ToDoList items do not participate in story lines
			return StoryLineTree.emptyTree();
	}	//	getDescendants()

	public boolean hasPlantSpecies()
	{
		return hasPlantSpeciesProperty().getValue();
	}
	/**
	*	Use this to check if the PlantSpecies parent of the ToDoList this Bean wraps is present
	*
	*	@return	true if this ToDoList is linked to a PlantSpecies
	*/
	public ReadOnlyBooleanProperty hasPlantSpeciesProperty()
	{
		return hasParentPlantSpeciesProperty.getReadOnlyProperty();
	}
	public PlantSpeciesBean getPlantSpecies()
	{
		return plantSpeciesProperty().getValue();
	}
	public void setPlantSpecies(final PlantSpeciesBean bean)
	{
		plantSpeciesProperty().setValue(bean);
	}
	public void setPlantSpecies(final IPlantSpecies item)
	{
		plantSpeciesProperty().setValue(new PlantSpeciesBean(item));
	}
	/**
	*	Returns the PlantSpecies parent of the ToDoList this Bean wraps
	*	Call hasPlantSpecies() first to check if this value is set
	*
	*	@return	the PlantSpecies parent of the ToDoList this Bean wraps
	*/
	public ObjectProperty<PlantSpeciesBean> plantSpeciesProperty()
	{
		return parentPlantSpeciesProperty;
	}

	/**
	*	Handle changes to the PlantSpeciesId value
	*
	*	@throws	GNDBRuntimeException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	private void onPlantSpeciesIdChange(ObservableValue<? extends PlantSpeciesBean> obs, PlantSpeciesBean old, PlantSpeciesBean nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onPlantSpeciesIdChange(): old={}, new={}", old, nval);
		if (nval != null && nval.sameAs(old))
		{
LOGGER.debug("onPlantSpeciesIdChange(): nval is sameAs old");
			return;
		}
		hasParentPlantSpeciesProperty.set(nval != null);

		if ((nval != null) && !nval.isNew())
		{
			if (explicitSave)
			{
LOGGER.debug("onPlantSpeciesIdChange(): explicitSave");
				explicitBuilder.plantSpecies(nval.get().get());
			}
			else
			{
LOGGER.debug("onPlantSpeciesIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).plantSpecies(nval.get().get()).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}
		else if (nval == null)
		{
			if (explicitSave)
			{
				explicitBuilder.plantSpecies(null);
			}
			else
			{
LOGGER.debug("onPlantSpeciesIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).plantSpecies(null).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}

		LOGGER.traceExit(log4jEntryMsg);
	}


	public boolean hasPlantVariety()
	{
		return hasPlantVarietyProperty().getValue();
	}
	/**
	*	Use this to check if the PlantVariety parent of the ToDoList this Bean wraps is present
	*
	*	@return	true if this ToDoList is linked to a PlantVariety
	*/
	public ReadOnlyBooleanProperty hasPlantVarietyProperty()
	{
		return hasParentPlantVarietyProperty.getReadOnlyProperty();
	}
	public PlantVarietyBean getPlantVariety()
	{
		return plantVarietyProperty().getValue();
	}
	public void setPlantVariety(final PlantVarietyBean bean)
	{
		plantVarietyProperty().setValue(bean);
	}
	public void setPlantVariety(final IPlantVariety item)
	{
		plantVarietyProperty().setValue(new PlantVarietyBean(item));
	}
	/**
	*	Returns the PlantVariety parent of the ToDoList this Bean wraps
	*	Call hasPlantVariety() first to check if this value is set
	*
	*	@return	the PlantVariety parent of the ToDoList this Bean wraps
	*/
	public ObjectProperty<PlantVarietyBean> plantVarietyProperty()
	{
		return parentPlantVarietyProperty;
	}

	/**
	*	Handle changes to the PlantVarietyId value
	*
	*	@throws	GNDBRuntimeException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	private void onPlantVarietyIdChange(ObservableValue<? extends PlantVarietyBean> obs, PlantVarietyBean old, PlantVarietyBean nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onPlantVarietyIdChange(): old={}, new={}", old, nval);
		if (nval != null && nval.sameAs(old))
		{
LOGGER.debug("onPlantVarietyIdChange(): nval is sameAs old");
			return;
		}
		hasParentPlantVarietyProperty.set(nval != null);

		if ((nval != null) && !nval.isNew())
		{
			if (explicitSave)
			{
LOGGER.debug("onPlantVarietyIdChange(): explicitSave");
				explicitBuilder.plantVariety(nval.get().get());
			}
			else
			{
LOGGER.debug("onPlantVarietyIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).plantVariety(nval.get().get()).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}
		else if (nval == null)
		{
			if (explicitSave)
			{
				explicitBuilder.plantVariety(null);
			}
			else
			{
LOGGER.debug("onPlantVarietyIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).plantVariety(null).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}

		LOGGER.traceExit(log4jEntryMsg);
	}


	public boolean hasHusbandryClass()
	{
		return hasHusbandryClassProperty().getValue();
	}
	/**
	*	Use this to check if the HusbandryClass parent of the ToDoList this Bean wraps is present
	*
	*	@return	true if this ToDoList is linked to a HusbandryClass
	*/
	public ReadOnlyBooleanProperty hasHusbandryClassProperty()
	{
		return hasParentHusbandryClassProperty.getReadOnlyProperty();
	}
	public HusbandryClassBean getHusbandryClass()
	{
		return husbandryClassProperty().getValue();
	}
	public void setHusbandryClass(final HusbandryClassBean bean)
	{
		husbandryClassProperty().setValue(bean);
	}
	public void setHusbandryClass(final IHusbandryClass item)
	{
		husbandryClassProperty().setValue(new HusbandryClassBean(item));
	}
	/**
	*	Returns the HusbandryClass parent of the ToDoList this Bean wraps
	*	Call hasHusbandryClass() first to check if this value is set
	*
	*	@return	the HusbandryClass parent of the ToDoList this Bean wraps
	*/
	public ObjectProperty<HusbandryClassBean> husbandryClassProperty()
	{
		return parentHusbandryClassProperty;
	}

	/**
	*	Handle changes to the HusbandryClassId value
	*
	*	@throws	GNDBRuntimeException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	private void onHusbandryClassIdChange(ObservableValue<? extends HusbandryClassBean> obs, HusbandryClassBean old, HusbandryClassBean nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onHusbandryClassIdChange(): old={}, new={}", old, nval);
		if (nval != null && nval.sameAs(old))
		{
LOGGER.debug("onHusbandryClassIdChange(): nval is sameAs old");
			return;
		}
		hasParentHusbandryClassProperty.set(nval != null);

		if ((nval != null) && !nval.isNew())
		{
			if (explicitSave)
			{
LOGGER.debug("onHusbandryClassIdChange(): explicitSave");
				explicitBuilder.husbandryClass(nval.get().get());
			}
			else
			{
LOGGER.debug("onHusbandryClassIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).husbandryClass(nval.get().get()).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}
		else if (nval == null)
		{
			if (explicitSave)
			{
				explicitBuilder.husbandryClass(null);
			}
			else
			{
LOGGER.debug("onHusbandryClassIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).husbandryClass(null).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}

		LOGGER.traceExit(log4jEntryMsg);
	}	//	onHusbandryClassIdChange()


	public boolean hasGroundworkActivity()
	{
		return hasGroundworkActivityProperty().getValue();
	}
	/**
	*	Use this to check if the GroundworkActivity parent of the ToDoList this Bean wraps is present
	*
	*	@return	true if this ToDoList is linked to a GroundworkActivity
	*/
	public ReadOnlyBooleanProperty hasGroundworkActivityProperty()
	{
		return hasParentGroundworkActivityProperty.getReadOnlyProperty();
	}
	public GroundworkActivityBean getGroundworkActivity()
	{
		return groundworkActivityProperty().getValue();
	}
	public void setGroundworkActivity(final GroundworkActivityBean bean)
	{
		groundworkActivityProperty().setValue(bean);
	}
	public void setGroundworkActivity(final IGroundworkActivity item)
	{
		groundworkActivityProperty().setValue(new GroundworkActivityBean(item));
	}
	/**
	*	Returns the GroundworkActivity parent of the ToDoList this Bean wraps
	*	Call hasGroundworkActivity() first to check if this value is set
	*
	*	@return	the GroundworkActivity parent of the ToDoList this Bean wraps
	*/
	public ObjectProperty<GroundworkActivityBean> groundworkActivityProperty()
	{
		return parentGroundworkActivityProperty;
	}

	/**
	*	Handle changes to the GroundWorkActivityId value
	*
	*	@throws	GNDBRuntimeException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	private void onGroundWorkActivityIdChange(ObservableValue<? extends GroundworkActivityBean> obs, GroundworkActivityBean old, GroundworkActivityBean nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onGroundWorkActivityIdChange(): old={}, new={}", old, nval);
		if (nval != null && nval.sameAs(old))
		{
LOGGER.debug("onGroundWorkActivityIdChange(): nval is sameAs old");
			return;
		}
		hasParentGroundworkActivityProperty.set(nval != null);

		if ((nval != null) && !nval.isNew())
		{
			if (explicitSave)
			{
LOGGER.debug("onGroundWorkActivityIdChange(): explicitSave");
				explicitBuilder.groundworkActivity(nval.get().get());
			}
			else
			{
LOGGER.debug("onGroundWorkActivityIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).groundworkActivity(nval.get().get()).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}
		else if (nval == null)
		{
			if (explicitSave)
			{
				explicitBuilder.groundworkActivity(null);
			}
			else
			{
LOGGER.debug("onGroundWorkActivityIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).groundworkActivity(null).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}

		LOGGER.traceExit(log4jEntryMsg);
	}


	public boolean hasHusbandry()
	{
		return hasHusbandryProperty().getValue();
	}
	/**
	*	Use this to check if the Husbandry parent of the ToDoList this Bean wraps is present
	*
	*	@return	true if this ToDoList is linked to a Husbandry
	*/
	public ReadOnlyBooleanProperty hasHusbandryProperty()
	{
		return hasParentHusbandryProperty.getReadOnlyProperty();
	}
	public HusbandryBean getHusbandry()
	{
		return husbandryProperty().getValue();
	}
	public void setHusbandry(final HusbandryBean bean)
	{
		husbandryProperty().setValue(bean);
	}
	public void setHusbandry(final IHusbandry item)
	{
		husbandryProperty().setValue(new HusbandryBean(item));
	}
	/**
	*	Returns the Husbandry parent of the ToDoList this Bean wraps
	*	Call hasHusbandry() first to check if this value is set
	*
	*	@return	the Husbandry parent of the ToDoList this Bean wraps
	*/
	public ObjectProperty<HusbandryBean> husbandryProperty()
	{
		return parentHusbandryProperty;
	}

	/**
	*	Handle changes to the husbandryId value
	*
	*	@throws	GNDBRuntimeException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	private void onHusbandryIdChange(ObservableValue<? extends HusbandryBean> obs, HusbandryBean old, HusbandryBean nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onHusbandryIdChange(): old={}, new={}", old, nval);
		if (nval != null && nval.sameAs(old))
		{
LOGGER.debug("onHusbandryIdChange(): nval is sameAs old");
			return;
		}
		hasParentHusbandryProperty.set(nval != null);

		//	cannot have Husbandry AND PurchaseItem as parents!
		if (nval != null)
		{
			purchaseItemProperty().setValue(null);
		}

		if ((nval != null) && !nval.isNew())
		{
			if (explicitSave)
			{
LOGGER.debug("onHusbandryIdChange(): explicitSave");
				explicitBuilder.husbandry(nval.get().get());
			}
			else
			{
LOGGER.debug("onHusbandryIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).husbandry(nval.get().get()).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}
		else if (nval == null)
		{
			if (explicitSave)
			{
				explicitBuilder.husbandry(null);
			}
			else
			{
LOGGER.debug("onHusbandryIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).husbandry(null).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}

		LOGGER.traceExit(log4jEntryMsg);
	}

	public boolean hasPurchaseItem()
	{
		return hasPurchaseItemProperty().getValue();
	}
	/**
	 *	Use this to check if the PurchaseItem parent of the ToDoList this Bean wraps is present
	 *
	 *	@return	true if this ToDoList is linked to a PurchaseItem
	 */
	public ReadOnlyBooleanProperty hasPurchaseItemProperty()
	{
		return hasParentPurchaseItemProperty.getReadOnlyProperty();
	}
	public PurchaseItemBean getPurchaseItem()
	{
		return purchaseItemProperty().getValue();
	}
	public void setPurchaseItem(final PurchaseItemBean bean)
	{
		purchaseItemProperty().setValue(bean);
	}
	public void setPurchaseItem(final IPurchaseItem item)
	{
		purchaseItemProperty().setValue(new PurchaseItemBean(item));
	}
	/**
	 *	Returns the PurchaseItem parent of the ToDoList this Bean wraps
	 *	Call hasPurchaseItem() first to check if this value is set
	 *
	 *	@return	the PurchaseItem parent of the ToDoList this Bean wraps
	 */
	public ObjectProperty<PurchaseItemBean> purchaseItemProperty()
	{
		return parentPurchaseItemProperty;
	}

	/**
	 *	Handle changes to the purchaseItemId value
	 *
	 *	@throws	GNDBRuntimeException	if the underlying persisted storage engine (e.g. database server) throws an exception
	 *				The original error can be retrieved by <code>getCause()</code>
	 */
	private void onPurchaseItemIdChange(ObservableValue<? extends PurchaseItemBean> obs, PurchaseItemBean old, PurchaseItemBean nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onPurchaseItemIdChange(): old={}, new={}", old, nval);
		if (nval != null && nval.sameAs(old))
		{
			LOGGER.debug("onPurchaseItemIdChange(): nval is sameAs old");
			return;
		}
		hasParentPurchaseItemProperty.set(nval != null);

		//	cannot have Husbandry AND PurchaseItem as parents!
		if (nval != null)
		{
			husbandryProperty().setValue(null);
		}

		if ((nval != null) && !nval.isNew())
		{
			if (explicitSave)
			{
				LOGGER.debug("onPurchaseItemIdChange(): explicitSave");
				explicitBuilder.purchaseItem(nval.get().get());
			}
			else
			{
				LOGGER.debug("onPurchaseItemIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).purchaseItem(nval.get().get()).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}
		else if (nval == null)
		{
			if (explicitSave)
			{
				explicitBuilder.purchaseItem(null);
			}
			else
			{
				LOGGER.debug("onPurchaseItemIdChange(): NOT explicitSave");
				ITrug server = TrugServer.getTrugServer().getTrug();
				//	the Builder will send an event to the baseItem to say it's been replaced
				try
				{
					server.getToDoListBuilder(baseItem).purchaseItem(null).save();
				} catch (GNDBException ex) {
					throw new GNDBRuntimeException(ex);
				}
			}
		}

		LOGGER.traceExit(log4jEntryMsg);
	}

	public String getDescription()
	{
		return descriptionProperty.get();
	}
	public void setDescription(final String newVal)
	{
		descriptionProperty.set(newVal);
	}
	/**
	*	Wraps the Description value of the ToDoList
	*
	*	@return	a writable property wrapping the description attribute
	*/
	public StringProperty descriptionProperty()
	{
		return descriptionProperty;
	}

	private void onDescriptionChange(ObservableValue<? extends String> obs, String old, String nval)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("onDescriptionChange(): old={}, new={}", old, nval);
		if (explicitSave)
		{
LOGGER.debug("onDescriptionChange(): explicitSave");
			explicitBuilder.description(nval);
		}
		else
		{
LOGGER.debug("onDescriptionChange(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			try
			{
				baseItem = server.getToDoListBuilder(baseItem).description(nval).save();
			} catch (GNDBException ex) {
				throw new GNDBRuntimeException(ex);
			}
		}
		LOGGER.traceExit(log4jEntryMsg);
	}

	public LocalDateTime getLastUpdated()
	{
		return lastUpdatedProperty().get();
	}
	/**
	*	Wraps the LastUpdated value of the ToDoList
	*	Note that this value cannot be changed by the user
	*
	*	@return	a read-only property wrapping the lastUpdated attribute
	*/
	public ReadOnlyObjectProperty<LocalDateTime> lastUpdatedProperty()
	{
		return lastUpdatedProperty.getReadOnlyProperty();
	}

	public LocalDateTime getCreated()
	{
		return createdProperty().get();
	}
	/**
	*	Wraps the Created value of the ToDoList
	*	Note that this value cannot be changed by the user
	*
	*	@return	a read-only property wrapping the created attribute
	*/
	public ReadOnlyObjectProperty<LocalDateTime> createdProperty()
	{
		return createdProperty.getReadOnlyProperty();
	}

	/**
	*	The ToDoList action has been completed, make an entry in the diary
	*
	 * @param showHusbandry A callback to the calling object.  Typically, used to pop up a HusbandryEditor with the new Husbandry.
	 *                      May be null.
	 *
	 * @param showGroundwork A callback to the calling object.  Typically, used to pop up a GroundworkEditor with the new Groundwork.
	 *                      May be null.
	 *
	*	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
//	public void completeAction() throws GNDBException
	public void completeAction(Consumer<HusbandryBean> showHusbandry, Consumer<GroundworkBean> showGroundwork) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("completeAction()");

		if (hasHusbandryClass())
		{
LOGGER.debug("completeAction(): hasHusbandryClass");
			HusbandryBean hb = new HusbandryBean();
			hb.setHusbandryClass(getHusbandryClass());
			if (hasPlantSpecies())
			{
LOGGER.debug("completeAction(): hasPlantSpecies");
				hb.setPlantSpecies(getPlantSpecies());
				if (hasPlantVariety())
				{
LOGGER.debug("completeAction(): hasPlantVariety");
					hb.setPlantVariety(getPlantVariety());
				}
			}
			if (hasHusbandry())
			{
				HusbandryBean parentHusbandry = husbandryProperty().get();
				hb.setAncestor(parentHusbandry);
				if (parentHusbandry.hasLocation())
				{
					hb.setLocation(parentHusbandry.getLocation());
				}
			}
			else if (hasPurchaseItem())
			{
				hb.setAncestor(purchaseItemProperty().get());
			}
//			LOGGER.debug("completeAction(): about to add comment: {}", getDescription());
			CommentBean cb = new CommentBean(hb);
			cb.setComment(getDescription());
//			LOGGER.debug("completeAction(): after adding comment, HB has: {}", hb.getComments());
			hb.save();
			if (showHusbandry != null)
			{
				showHusbandry.accept(hb);
			}
		}

		if (hasGroundworkActivity())
		{
LOGGER.debug("completeAction(): hasGroundworkActivity");
			GroundworkBean gb = new GroundworkBean();
			gb.setGroundworkActivity(getGroundworkActivity());
			if (hasPlantSpecies())
			{
LOGGER.debug("completeAction(): hasPlantSpecies");
				gb.setPlantSpecies(getPlantSpecies());
				if (hasPlantVariety())
				{
LOGGER.debug("completeAction(): hasPlantVariety");
					gb.setPlantVariety(getPlantVariety());
				}
			}
			if (hasHusbandry())
			{
				HusbandryBean parentHusbandry = husbandryProperty().get();
				gb.setAncestor(parentHusbandry);
				if (parentHusbandry.hasLocation())
				{
					gb.setLocation(parentHusbandry.getLocation());
				}
			}
			else if (hasPurchaseItem())
			{
				gb.setAncestor(purchaseItemProperty().get());
			}
			LOGGER.debug("before adding comment: {}", getDescription());
			CommentBean cb = new CommentBean(gb);
			cb.setComment(getDescription());
			gb.save();
			if (showGroundwork != null)
			{
				showGroundwork.accept(gb);
			}
		}
		delete();
		LOGGER.traceExit(log4jEntryMsg);
	}	// completeAction

	/**
	 *	The ToDoList action is to be delayed.
	 *	Convert the ToDoList to a Reminder with a later date.
	 *
	 */
	public void delayAction() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("delayAction()");
		ReminderBean rmb = new ReminderBean(this);
		rmb.setShowFrom(LocalDate.now().plusDays(7));
		rmb.save();
		this.delete();
	}

	@Override
	public ObservableList<CommentBean> getComments()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("getComments()");

		return LOGGER.traceExit(log4jEntryMsg, beanCommentHandler.getComments());
	}	//	getComments()

	//	2.9.6
	@Override
	public ReadOnlyStringProperty commentTextProperty()
	{
//		return beanCommentHandler.commentTextProperty();
		commentTextProperty.set(beanCommentHandler.commentTextProperty().get());
		return commentTextProperty;
	}

	@Override
	public void addComment(final String text) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment({})", text);
		if (text == null || text.isBlank()) return;	//	2.9.6

		beanCommentHandler.addComment(text);	//	2.9.6

		if (explicitSave)
		{
LOGGER.debug("addComment(): explicitSave");
		}
		else
		{	//	this cannot be a new instance of the parent ToDoList
LOGGER.debug("addComment(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			baseItem = server.getToDoListBuilder(baseItem).addComment(text).save();
			setValues();	//	2.9.6
		}
		LOGGER.traceExit(log4jEntryMsg);
	}	//	addComment()

	//	2.9.6
	@Override
	public void addComment(CommentBean comment) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment(comment bean): comment: {}, text : {}", comment, comment==null ? "null" :comment.getComment());
		if (comment == null) return;
		if (comment.getParentType() != this.getType()) return;
		if (comment.getComment() == null || comment.getComment().isBlank()) return;

		beanCommentHandler.addComment(comment);

		if (explicitSave)
		{
			LOGGER.debug("addComment(): explicitSave");
		}
		else
		{	//	this cannot be a new instance of the parent Wildlife
			LOGGER.debug("addComment(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			baseItem = server.getToDoListBuilder(baseItem).addComment(comment.getComment()).save();
			setValues();	//	2.9.6
		}
		LOGGER.debug("addComment(comment bean): commentTextProperty: {}", commentTextProperty().get());

		LOGGER.traceExit(log4jEntryMsg);
	}

	@Override
	public void changeCommentText(final CommentBean comment, final String text) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeCommentText(): comment={}, text={}", comment, text);
		if (text == null || text.isBlank()) return;

		if (comment == null)	//	2.9.6
		{
			addComment(text);
			return;
		}

		if (comment.getParentType() != this.getType()) return;

		beanCommentHandler.changeCommentText(comment, text);

		if (explicitSave)
		{
LOGGER.debug("changeCommentText(): explicitSave");
		}
		else
		{	//	this cannot be a new instance of the parent ToDoList
LOGGER.debug("changeCommentText(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			baseItem = server.getToDoListBuilder(baseItem).changeComment(comment.get().get(), text).save();
			setValues();	//	2.9.6
		}
		LOGGER.traceExit(log4jEntryMsg);
	}	//	changeCommentText()

	@Override
	public void changeCommentDate(CommentBean comment, final LocalDate date) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeCommentDate(): comment={}, date={}", comment, date);
		if (date == null) return;

		//	2.9.6
		if (comment == null)
		{
			return;
		}

		if (comment.getParentType() != this.getType()) return;

		beanCommentHandler.changeCommentDate(comment, date);

		if (explicitSave)
		{
LOGGER.debug("changeCommentDate(): explicitSave");
		}
		else
		{	//	this cannot be a new instance of the parent ToDoList
LOGGER.debug("changeCommentDate(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			baseItem = server.getToDoListBuilder(baseItem).changeComment(comment.get().get(), date).save();
			setValues();	//	2.9.6
		}
		LOGGER.traceExit(log4jEntryMsg);
	}	//	changeCommentDate()

	@Override
	public void deleteComment(CommentBean comment) throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment(): comment={}", comment);
		if (comment == null) return;

		if (comment.getParentType() != this.getType()) return;

		beanCommentHandler.deleteComment(comment);

		if (explicitSave)
		{
LOGGER.debug("deleteComment(): explicitSave");
		}
		else
		{	//	this cannot be a new instance of the parent ToDoList
LOGGER.debug("deleteComment(): NOT explicitSave");
			ITrug server = TrugServer.getTrugServer().getTrug();
			baseItem = server.getToDoListBuilder(baseItem).deleteComment(comment.get().get()).save();
			setValues();	//	2.9.6
		}
		LOGGER.traceExit(log4jEntryMsg);
	}	//	deleteComment()

	public boolean isSaveRequired()
	{
		return explicitSave;
	}
	public void setSaveRequired(boolean reqd)
	{
		saveRequiredProperty.set(reqd);
	}
	public BooleanProperty saveRequiredProperty()
	{
		return saveRequiredProperty;
	}

	public boolean needSave()
	{
		if (!explicitSave)
			return false;

		return explicitBuilder.needSave() || beanCommentHandler.needSave();
	}

	public boolean canSave()
	{
		if (!explicitSave)
			return true;

		return explicitBuilder.canSave();
	}

	/**
	*	Save changes to the underlying ToDoList item
	*
	*	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	public void save() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("save()");
		if (!explicitSave) return;

		saveComments();	//	2.9.6 - do this here so that explicitBuilder knows there's a change

		if (!explicitBuilder.needSave())
		{
			return;
		}
		if (!explicitBuilder.canSave())
		{
			throw new IllegalStateException("ToDoListBean: cannot save at this time - mandatory values not set");
		}

		baseItem = explicitBuilder.save();
		LOGGER.debug("save(): after explicitBuilder.save(): comments: {}", ()->baseItem.getComments());	//	2.9.6
		setValues();	//	2.9.6
		saveRequiredProperty.set(false);
		LOGGER.traceExit(log4jEntryMsg);
	}	//	save()

	//	2.9.6
	private void saveComments()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("saveComments()");

		beanCommentHandler.saveComments(
				cb -> explicitBuilder.addComment(cb.getComment()),	//	add
				cb -> explicitBuilder.changeComment(cb.get().get(), cb.getComment()),	//	change text
				cb -> explicitBuilder.changeComment(cb.get().get(), cb.getDate()),	//	change date
				cb -> explicitBuilder.deleteComment(cb.get().get())		//	delete
		);
	}

	/**
	*	Delete the underlying ToDoList item
	*
	*	@throws	GNDBException	if the underlying persisted storage engine (e.g. database server) throws an exception
	*				The original error can be retrieved by <code>getCause()</code>
	*/
	public void delete() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("delete()");
		if (newItem) return;

		if (explicitSave)
		{
			explicitBuilder.delete();
			saveRequiredProperty.set(false);
		}
		else
		{
			ITrug server = TrugServer.getTrugServer().getTrug();
			server.getToDoListBuilder(baseItem).delete();
		}
	}	//	delete()

	public void cancelEdit()
	{
		if (!explicitSave) return;
		if (newItem) return;

		saveRequiredProperty.set(false);
		setValues();
	}

	private void setDefaults()
	{
		saveRequiredProperty.setValue(false);
		descriptionProperty.setValue("");
		lastUpdatedProperty.setValue(LocalDateTime.now());
		createdProperty.setValue(LocalDateTime.now());

		//	2.9.6
		isNewProperty.set(true);
		beanCommentHandler = new BeanCommentHandler<>(this, baseItem);
		commentTextProperty.set(beanCommentHandler.commentTextProperty().get());
	}

	private void setValues()
	{
		saveRequiredProperty.setValue(false);
		if (baseItem.getPlantSpecies().isPresent())
		{
			hasParentPlantSpeciesProperty.set(true);
			parentPlantSpeciesProperty.setValue(new PlantSpeciesBean(baseItem.getPlantSpecies().get()));
		}
		else
		{
			hasParentPlantSpeciesProperty.set(false);
			parentPlantSpeciesProperty.setValue(null);
		}
		if (baseItem.getPlantVariety().isPresent())
		{
			hasParentPlantVarietyProperty.set(true);
			parentPlantVarietyProperty.setValue(new PlantVarietyBean(baseItem.getPlantVariety().get()));
		}
		else
		{
			hasParentPlantVarietyProperty.set(false);
			parentPlantVarietyProperty.setValue(null);
		}
		if (baseItem.getHusbandryClass().isPresent())
		{
			hasParentHusbandryClassProperty.set(true);
			parentHusbandryClassProperty.setValue(new HusbandryClassBean(baseItem.getHusbandryClass().get()));
		}
		else
		{
			hasParentHusbandryClassProperty.set(false);
			parentHusbandryClassProperty.setValue(null);
		}
		if (baseItem.getGroundworkActivity().isPresent())
		{
			hasParentGroundworkActivityProperty.set(true);
			parentGroundworkActivityProperty.setValue(new GroundworkActivityBean(baseItem.getGroundworkActivity().get()));
		}
		else
		{
			hasParentGroundworkActivityProperty.set(false);
			parentGroundworkActivityProperty.setValue(null);
		}

		if (baseItem.getHusbandry().isPresent())
		{
			hasParentHusbandryProperty.set(true);
			parentHusbandryProperty.setValue(new HusbandryBean(baseItem.getHusbandry().get()));
		}
		else
		{
			hasParentHusbandryProperty.set(false);
			parentHusbandryProperty.setValue(null);
		}

		if (baseItem.getPurchaseItem().isPresent())
		{
			hasParentPurchaseItemProperty.set(true);
			parentPurchaseItemProperty.setValue(new PurchaseItemBean(baseItem.getPurchaseItem().get()));
		}
		else
		{
			hasParentPurchaseItemProperty.set(false);
			parentPurchaseItemProperty.setValue(null);
		}
		LOGGER.debug("setValues: hasHusbandry: {}, hasPurchaseItem: {}", hasHusbandry(), hasPurchaseItem());

		descriptionProperty.setValue(baseItem.getDescription().orElse(""));
		lastUpdatedProperty.setValue(baseItem.getLastUpdated());
		createdProperty.setValue(baseItem.getCreated());

		itemKey = baseItem.getKey();
		newItem = false;
		isNewProperty.set(false);	//	2.9.6

		LOGGER.debug("setvalues(): about to change BeanCommentHandler");
		beanCommentHandler = new BeanCommentHandler<>(this, baseItem);
		commentTextProperty.set(beanCommentHandler.commentTextProperty().get());
	}

	private void addListeners()
	{
		parentPlantSpeciesProperty.addListener(plantSpeciesIdListener);
		parentPlantVarietyProperty.addListener(plantVarietyIdListener);
		parentHusbandryClassProperty.addListener(husbandryClassIdListener);
		parentGroundworkActivityProperty.addListener(groundWorkActivityIdListener);
		parentHusbandryProperty.addListener(husbandryIdListener);
		parentPurchaseItemProperty.addListener(purchaseItemIdListener);
		descriptionProperty.addListener(descriptionListener);
	}
	private void removeListeners()
	{
		parentPlantSpeciesProperty.removeListener(plantSpeciesIdListener);
		parentPlantVarietyProperty.removeListener(plantVarietyIdListener);
		parentHusbandryClassProperty.removeListener(husbandryClassIdListener);
		parentGroundworkActivityProperty.removeListener(groundWorkActivityIdListener);
		parentHusbandryProperty.removeListener(husbandryIdListener);
		parentPurchaseItemProperty.removeListener(purchaseItemIdListener);
		descriptionProperty.removeListener(descriptionListener);
	}
	private void declareBaseListeners()
	{
		// handle changes to the base item itself
		baseItemDeleted = evt -> {
				removeListeners();
				removeBaseListeners();
				setDefaults();
				baseItem = null;
			};
		baseItemReplaced = evt -> {
				if (evt.getNewValue() != null)
				{
					removeBaseListeners();
					baseItem = (IToDoList)(evt.getNewValue());
					setValues();
					addBaseListeners();
				}
			};

	}
    
	private void addBaseListeners()
	{
        if (baseItem == null) return;
		baseItem.addPropertyChangeListener("deleted", baseItemDeleted);
		baseItem.addPropertyChangeListener("replaced", baseItemReplaced);

	}
	private void removeBaseListeners()
	{
        if (baseItem == null) return;
		baseItem.removePropertyChangeListener("deleted", baseItemDeleted);
		baseItem.removePropertyChangeListener("replaced", baseItemReplaced);

	}

	@Override
	public String toString()
	{
		return "ToDoListBean wrapping " + baseItem;
	}

}

