/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook.mysql;

import org.junit.jupiter.api.*;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.IComment;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;

//@Disabled("need to reset database")
@TestInstance(TestInstance.Lifecycle.PER_CLASS)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
class WeatherConditionBuilderTest
{

    private static WeatherConditionBuilder myTest;
    private static WeatherCondition wls;
    private static String name = "test " + LocalDateTime.now();
    private static ArrayList<String> testComments = new ArrayList(List.of("this is the first comment", "this is the second comment", "this is the third comment",
            "this is the fourth comment", "this is the fifth comment", "this is the sixth comment", "this is the seventh comment", "this is the eighth comment"));
    private static final LocalDate today = LocalDate.now();

    @BeforeAll
    static void startUp()
    {
        SQLTrug myTrug = new SQLTrug();
        myTrug.getServers();
        DBConnection.forceConnectionString(DBConnection.RDBMS_ENUM.MySQL,"jdbc:mysql://localhost:3306/gardennotebook?user=root&password=password&useSSL=false&allowPublicKeyRetrieval=true&sslMode=DISABLED&connectTimeout=1000&cachePrepStmts=true&prepStmtCacheSize=250&prepStmtCacheSqlLimit=2048&maintainTimeStats=false&disableMariaDbDriver=true&serverTimezone=UTC");
    }

    @AfterAll
    void tearDown()
    {
        try
        {
            myTest.delete();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @BeforeEach
    void setUp()
    {
        myTest = new WeatherConditionBuilder(wls);
    }

    @Test
    @Order(1)
    void name()
    {
        name = "test " + LocalDateTime.now();
        myTest = new WeatherConditionBuilder();
        assertFalse(myTest.needSave());
        assertFalse(myTest.canSave());
        try
        {
            myTest.name(name);
            assertTrue(myTest.needSave());
            assertTrue(myTest.canSave());
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertEquals(wls.getName(), name);
    }

    @Test
    @Order(2)
    void isNameDuplicate()
    {
        myTest = new WeatherConditionBuilder();
        try
        {
            assertTrue(myTest.isNameDuplicate(name));
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    @Order(3)
    void description()
    {
        assertFalse(myTest.needSave());
        myTest.description("test species, delete");
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        assertTrue(wls.getDescription().isPresent());
        assertEquals(wls.getDescription().get(), "test species, delete");
    }

    @Test
    @Order(4)
    void addComment()
    {
        assertFalse(myTest.needSave());
//        myTest.addComment("this is the first comment");
        myTest.addComment(testComments.get(0));
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 1);
        assertEquals(comments.get(0).getComment(), testComments.get(0));
        assertTrue(comments.get(0).getDate().equals((today)));
    }

    @Test
    @Order(5)
    void addMultiComment()
    {
        assertFalse(myTest.needSave());
        myTest.addComment(testComments.get(1));
        assertTrue(myTest.needSave());
        myTest.addComment(testComments.get(2), testComments.get(3));
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 4);
        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 4)));
        assertTrue(testComments.subList(0, 4).containsAll(foundComments));

        assertTrue(comments.get(1).getDate().equals((today)));
        assertTrue(comments.get(2).getDate().equals((today)));
        assertTrue(comments.get(3).getDate().equals((today)));
   }

    @Test
    @Order(6)
    void addCommentListOfStrings()
    {
        assertFalse(myTest.needSave());
        List<String> commentValues = List.of(testComments.get(4));
        myTest.addComment(commentValues);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 5);
        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 5)));
        assertTrue(testComments.subList(0, 5).containsAll(foundComments));
        assertTrue(comments.get(4).getDate().equals((today)));
    }

    @Test
    @Order(7)
    void addCommentMultiListOfStrings()
    {
        assertFalse(myTest.needSave());
        List<String> commentValues = List.of(testComments.get(5));
        myTest.addComment(commentValues);
        assertTrue(myTest.needSave());
        commentValues = List.of(testComments.get(6), testComments.get(7));
        myTest.addComment(commentValues);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 8);

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));
        assertTrue(comments.get(5).getDate().equals((today)));
        assertTrue(comments.get(6).getDate().equals((today)));
        assertTrue(comments.get(7).getDate().equals((today)));
    }

    @Test
    @Order(8)
    void changeCommentString()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 8);
        testComments.remove(comments.get(0).getComment());
        myTest.changeComment(comments.get(0), "first comment changed");
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(comments.size(), 8);
        assertEquals(comments.get(0).getComment(), "first comment changed");
        assertTrue(comments.get(0).getDate().equals((today)));
        List<String> foundComments = comments.stream().skip(1).map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 7)));
        assertTrue(testComments.subList(0, 7).containsAll(foundComments));

        assertTrue(comments.get(1).getDate().equals((today)));
        assertTrue(comments.get(2).getDate().equals((today)));
        assertTrue(comments.get(3).getDate().equals((today)));
        assertTrue(comments.get(4).getDate().equals((today)));
        assertTrue(comments.get(5).getDate().equals((today)));
        assertTrue(comments.get(6).getDate().equals((today)));
        assertTrue(comments.get(7).getDate().equals((today)));
        testComments.add("first comment changed");
    }

    @Test
    @Order(9)
    void changeCommentStringNull()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 8);
        myTest.changeComment(comments.get(0), (String)null);
        assertFalse(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(comments.size(), 8);

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));

        assertTrue(comments.get(0).getDate().equals((today)));
        assertTrue(comments.get(1).getDate().equals((today)));
        assertTrue(comments.get(2).getDate().equals((today)));
        assertTrue(comments.get(3).getDate().equals((today)));
        assertTrue(comments.get(4).getDate().equals((today)));
        assertTrue(comments.get(5).getDate().equals((today)));
        assertTrue(comments.get(6).getDate().equals((today)));
        assertTrue(comments.get(7).getDate().equals((today)));
    }

    @Test
    @Order(10)
    void changeCommentStringEmpty()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 8);
        myTest.changeComment(comments.get(0), "");
        assertFalse(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(comments.size(), 8);

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));

        assertTrue(comments.get(0).getDate().equals((today)));
        assertTrue(comments.get(1).getDate().equals((today)));
        assertTrue(comments.get(2).getDate().equals((today)));
        assertTrue(comments.get(3).getDate().equals((today)));
        assertTrue(comments.get(4).getDate().equals((today)));
        assertTrue(comments.get(5).getDate().equals((today)));
        assertTrue(comments.get(6).getDate().equals((today)));
        assertTrue(comments.get(7).getDate().equals((today)));
    }

    @Test
    @Order(11)
    void changeCommentDate()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 8);
        LocalDate targetDate = today.minusDays(2);
        String targetText = comments.get(1).getComment();
        myTest.changeComment(comments.get(1), targetDate);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();   //  comments are sorted by date!
        assertEquals(comments.size(), 8);
        assertEquals(comments.get(0).getComment(), targetText);
        assertTrue(comments.get(0).getDate().equals(targetDate));

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));

        assertTrue(comments.get(1).getDate().equals(today));

        assertTrue(comments.get(2).getDate().equals(today));
        assertTrue(comments.get(3).getDate().equals(today));
        assertTrue(comments.get(4).getDate().equals((today)));
        assertTrue(comments.get(5).getDate().equals((today)));
        assertTrue(comments.get(6).getDate().equals(today));
        assertTrue(comments.get(7).getDate().equals(today));
    }

    @Test
    @Order(12)
    void changeCommentDateNull()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 8);
        LocalDate oldDate = comments.get(0).getDate();
        String targetText = comments.get(0).getComment();
        myTest.changeComment(comments.get(0), (LocalDate)null);
        assertFalse(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();   //  comments are sorted by date!
        assertEquals(comments.size(), 8);
        assertEquals(comments.get(0).getComment(), targetText);
        assertNotNull(comments.get(0).getDate());
        assertTrue(comments.get(0).getDate().equals(oldDate));

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));

        assertTrue(comments.get(1).getDate().equals(today));

        assertTrue(comments.get(2).getDate().equals(today));
        assertTrue(comments.get(3).getDate().equals(today));
        assertTrue(comments.get(4).getDate().equals((today)));
        assertTrue(comments.get(5).getDate().equals((today)));
        assertTrue(comments.get(6).getDate().equals(today));
        assertTrue(comments.get(7).getDate().equals(today));
    }

    @Test
    @Order(14)
    void changeCommentDateFuture()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 8);
        LocalDate oldDate = comments.get(0).getDate();
        LocalDate targetDate = today.plusDays(2);
        String targetText = comments.get(0).getComment();
        myTest.changeComment(comments.get(0), targetDate);
        assertFalse(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();   //  comments are sorted by date!
        assertEquals(comments.size(), 8);
        assertEquals(comments.get(0).getComment(), targetText);
        assertFalse(comments.get(0).getDate().equals(targetDate));
        assertTrue(comments.get(0).getDate().equals(oldDate));

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));

        assertTrue(comments.get(1).getDate().equals(today));

        assertTrue(comments.get(2).getDate().equals(today));
        assertTrue(comments.get(3).getDate().equals(today));
        assertTrue(comments.get(4).getDate().equals((today)));
        assertTrue(comments.get(5).getDate().equals((today)));
        assertTrue(comments.get(6).getDate().equals(today));
        assertTrue(comments.get(7).getDate().equals(today));
    }

    @Test
    @Order(15)
    void changeCommentStringDate()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 8);
        LocalDate targetDate = today.minusDays(3);
        String targetText = comments.get(2).getComment();
        testComments.remove(targetText);
        myTest.changeComment(comments.get(2), targetDate, "third comment changed");
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();   //  comments are sorted by date!
        assertEquals(comments.size(), 8);
        assertEquals(comments.get(0).getComment(), "third comment changed");
        assertTrue(comments.get(0).getDate().equals(targetDate));

        List<String> foundComments = comments.stream().skip(1).map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 7)));
        assertTrue(testComments.subList(0, 7).containsAll(foundComments));

        assertTrue(comments.get(1).getDate().equals(today.minusDays(2)));

        assertTrue(comments.get(2).getDate().equals(today));

        assertTrue(comments.get(3).getDate().equals(today));
        assertTrue(comments.get(4).getDate().equals((today)));
        assertTrue(comments.get(5).getDate().equals((today)));
        assertTrue(comments.get(6).getDate().equals(today));
        assertTrue(comments.get(7).getDate().equals(today));
        testComments.add("third comment changed");
    }

    @Test
    @Order(50)
    void deleteCommentById()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 8);
        int key1 = comments.get(0).getKey();
        String targetText = comments.get(0).getComment();
        testComments.remove(targetText);
        myTest.deleteComment(key1);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(comments.size(), 7);

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 7)));
        assertTrue(testComments.subList(0, 7).containsAll(foundComments));

    }

    @Test
    @Order(51)
    void deleteMultiCommentById()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 7);
        int key1 = comments.get(0).getKey();
        int key2 = comments.get(1).getKey();
        int key3 = comments.get(2).getKey();

        String targetText = comments.get(0).getComment();
        testComments.remove(targetText);
        targetText = comments.get(1).getComment();
        testComments.remove(targetText);
        targetText = comments.get(2).getComment();
        testComments.remove(targetText);

        myTest.deleteComment(key1);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        myTest.deleteComment(key2, key3);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(comments.size(), 4);

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 4)));
        assertTrue(testComments.subList(0, 4).containsAll(foundComments));

    }

    @Test
    @Order(52)
    void deleteCommentByArray()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 4);

        String targetText = comments.get(0).getComment();
        testComments.remove(targetText);

        myTest.deleteComment(comments.get(0));
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(comments.size(), 3);

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 3)));
        assertTrue(testComments.subList(0, 3).containsAll(foundComments));

    }

    @Test
    @Order(53)
    void deleteMultiCommentByArray()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 3);
        myTest.deleteComment(comments.get(0));
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        myTest.deleteComment(comments.get(1), comments.get(2));
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(comments.size(), 0);
    }

    @Test
    @Order(60)
    void setUpForDeleteCommentByList()
    {
        testComments = new ArrayList(List.of("this is the first comment", "this is the second comment", "this is the third comment", "this is the fourth comment"));
        myTest.addComment(testComments);
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 4);


        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments));
        assertTrue(testComments.containsAll(foundComments));

        assertTrue(comments.get(0).getDate().equals((today)));
        assertTrue(comments.get(1).getDate().equals((today)));
        assertTrue(comments.get(2).getDate().equals((today)));
        assertTrue(comments.get(3).getDate().equals((today)));
    }

    @Test
    @Order(61)
    void deleteCommentByList()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 4);
        List<IComment> listComments = List.of(comments.get(0));
        testComments.remove(comments.get(0).getComment());
        myTest.deleteComment(listComments);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(comments.size(), 3);

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 3)));
        assertTrue(testComments.subList(0, 3).containsAll(foundComments));

        assertTrue(comments.get(0).getDate().equals((today)));
        assertTrue(comments.get(1).getDate().equals((today)));
        assertTrue(comments.get(2).getDate().equals((today)));
    }

    @Test
    @Order(62)
    void deleteMultiCommentByList()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 3);
        List<IComment> testComments = List.of(comments.get(0));
        myTest.deleteComment(testComments);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());

       testComments = List.of(comments.get(1), comments.get(2));
        myTest.deleteComment(testComments);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (WeatherCondition) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(comments.size(), 0);
    }

    @Test
    @Order(90)
    void canDelete()
    {
        try
        {
            assertTrue(myTest.canDelete());
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    @Order(91)
    void delete()
    {
        try
        {
            myTest.delete();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    //  Failing tests

    @Test
    @Order(100)
    void cannotSave()
    {
        myTest = new WeatherConditionBuilder();
        assertFalse(myTest.needSave());
        myTest.description("test species, delete");
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());
        assertThrows(IllegalStateException.class, myTest::save);
    }
}