/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook.mysql;

import org.junit.jupiter.api.*;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.IReview;

import java.time.MonthDay;
import java.time.Year;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;

@TestInstance(TestInstance.Lifecycle.PER_CLASS)
//@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
class ReviewListerTest_MySQL
{

    static IReview rev1 = null; //  save() sets default coverFrom/To
    static IReview rev2 = null; //  explicit coverFrom
    static IReview rev3 = null; //  explicit coverTo
    static IReview rev4 = null; //  explicit coverFrom/To
    static IReview rev5 = null; //  next year, save() sets default coverFrom/To

    @BeforeAll
    static void startUp()
    {
        SQLTrug myTrug = new SQLTrug();
        myTrug.getServers();
        DBConnection.forceConnectionString(DBConnection.RDBMS_ENUM.MySQL, "jdbc:mysql://localhost:3306/gardennotebook?user=root&password=password&useSSL=false&allowPublicKeyRetrieval=true&sslMode=DISABLED&connectTimeout=1000&cachePrepStmts=true&prepStmtCacheSize=250&prepStmtCacheSqlLimit=2048&maintainTimeStats=false&disableMariaDbDriver=true&serverTimezone=UTC");

        try
        {
            rev1 = new ReviewBuilder().yearInReview(Year.now()).title("Test Review 1").description("Just a test 1").save();
            rev2 = new ReviewBuilder().yearInReview(Year.now()).title("Test Review 2").description("Just a test 2").coverFrom(Year.now().atDay(32)).save(); //  1 Feb
            rev3 = new ReviewBuilder().yearInReview(Year.now()).title("Test Review 3").description("Just a test 3").coverTo(Year.now().atMonthDay(MonthDay.of(11,30))).save();
            rev4 = new ReviewBuilder().yearInReview(Year.now()).title("Test Review 4").description("Just a test 4").
                    coverFrom(Year.now().atDay(32)).
                    coverTo(Year.now().atMonthDay(MonthDay.of(11,30))).save();
            rev5 = new ReviewBuilder().yearInReview(Year.now().plusYears(1)).title("Test Review 5").description("Just a test 5").save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @AfterAll
    static void closeDown()
    {
        try
        {
            new ReviewBuilder(rev1).delete();
            new ReviewBuilder(rev2).delete();
            new ReviewBuilder(rev3).delete();
            new ReviewBuilder(rev4).delete();
            new ReviewBuilder(rev5).delete();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @BeforeEach
    void setUp()
    {
    }

    @AfterEach
    void tearDown()
    {
    }

    @Test
    void fetch()
    {
        try
        {
            List<? extends IReview> revs = new ReviewLister().fetch();
            assertEquals(5, revs.size());
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void reviewWithDefaultFromTo()
    {
        try
        {
            List<? extends IReview> revs = new ReviewLister().review(rev1).fetch();
            assertEquals(1, revs.size());
            assertTrue(revs.get(0).sameAs(rev1));
            assertEquals(Year.now().atMonthDay(MonthDay.of(1,1)), revs.get(0).getCoverFrom().orElse(null));
            assertEquals(Year.now().atMonthDay(MonthDay.of(12,31)), revs.get(0).getCoverTo().orElse(null));
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void reviewWithDefaultTo()
    {
        try
        {
            List<? extends IReview> revs = new ReviewLister().review(rev2).fetch();
            assertEquals(1, revs.size());
            assertTrue(revs.get(0).sameAs(rev2));
            assertEquals(Year.now().atMonthDay(MonthDay.of(2,1)), revs.get(0).getCoverFrom().orElse(null));
            assertEquals(Year.now().atMonthDay(MonthDay.of(12,31)), revs.get(0).getCoverTo().orElse(null));
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void reviewWithDefaultFrom()
    {
        try
        {
            List<? extends IReview> revs = new ReviewLister().review(rev3).fetch();
            assertEquals(1, revs.size());
            assertTrue(revs.get(0).sameAs(rev3));
            assertEquals(Year.now().atMonthDay(MonthDay.of(1,1)), revs.get(0).getCoverFrom().orElse(null));
            assertEquals(Year.now().atMonthDay(MonthDay.of(11,30)), revs.get(0).getCoverTo().orElse(null));
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void reviewWithExplicitFromTo()
    {
        try
        {
            List<? extends IReview> revs = new ReviewLister().review(rev4).fetch();
            assertEquals(1, revs.size());
            assertTrue(revs.get(0).sameAs(rev4));
            assertEquals(Year.now().atMonthDay(MonthDay.of(2,1)), revs.get(0).getCoverFrom().orElse(null));
            assertEquals(Year.now().atMonthDay(MonthDay.of(11,30)), revs.get(0).getCoverTo().orElse(null));
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void reviewNextYearWithDefaultFromTo()
    {
        try
        {
            List<? extends IReview> revs = new ReviewLister().review(rev5).fetch();
            assertEquals(1, revs.size());
            assertTrue(revs.get(0).sameAs(rev5));
            assertEquals(Year.now().plusYears(1).atMonthDay(MonthDay.of(1,1)), revs.get(0).getCoverFrom().orElse(null));
            assertEquals(Year.now().plusYears(1).atMonthDay(MonthDay.of(12,31)), revs.get(0).getCoverTo().orElse(null));
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void multiReview()
    {
        try
        {
            List<IReview> revs = (List<IReview>) new ReviewLister().review(rev1).review(rev2).fetch();
            assertEquals(2, revs.size());
            assertTrue(revs.get(0).sameAs(rev1));
            assertTrue(revs.get(1).sameAs(rev2));
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void multiReviewItems()
    {
        try
        {
            List<IReview> revs = (List<IReview>) new ReviewLister().review(rev1, rev2).fetch();
            assertEquals(2, revs.size());
            assertTrue(revs.get(0).sameAs(rev1));
            assertTrue(revs.get(1).sameAs(rev2));
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void multiReviewList()
    {
        try
        {
            List<IReview> revs = (List<IReview>) new ReviewLister().review(List.of(rev1, rev2)).fetch();
            assertEquals(2, revs.size());
            assertTrue(revs.get(0).sameAs(rev1));
            assertTrue(revs.get(1).sameAs(rev2));
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void yearInReview()
    {
        try
        {
            List<IReview> revs = (List<IReview>) new ReviewLister().yearInReview(Year.now()).fetch();
            assertEquals(4, revs.size());
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void yearInReviewNext()
    {
        try
        {
            List<IReview> revs = (List<IReview>) new ReviewLister().yearInReview(Year.now().plusYears(1)).fetch();
            assertEquals(1, revs.size());
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void yearInReviewLast()
    {
        try
        {
            List<IReview> revs = (List<IReview>) new ReviewLister().yearInReview(Year.now().minusYears(1)).fetch();
            assertEquals(0, revs.size());
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    void coverFrom()
    {
    }

    @Test
    void coverTo()
    {
    }

    @Test
    void fromDate()
    {
    }

    @Test
    void toDate()
    {
    }
}