/*
 *
 *  Copyright (C) 2021 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook.mysql;

import org.junit.jupiter.api.*;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.IComment;
import uk.co.gardennotebook.spi.ICropRotationGroup;
import uk.co.gardennotebook.spi.ILocation;

import java.time.LocalDate;
import java.time.Year;
import java.util.ArrayList;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;

@TestInstance(TestInstance.Lifecycle.PER_CLASS)
@TestMethodOrder(MethodOrderer.OrderAnnotation.class)
class CroppingPlanBuilderTest
{
    private static CroppingPlanBuilder myTest;
    private static CroppingPlan wls;
    private static ArrayList<String> testComments = new ArrayList<>(List.of("this is the first comment", "this is the second comment", "this is the third comment",
            "this is the fourth comment", "this is the fifth comment", "this is the sixth comment", "this is the seventh comment", "this is the eighth comment"));
    private static final LocalDate today = LocalDate.now();

    @BeforeAll
    static void startUp()
    {
        SQLTrug myTrug = new SQLTrug();
        myTrug.getServers();
        DBConnection.forceConnectionString(DBConnection.RDBMS_ENUM.MySQL,"jdbc:mysql://localhost:3306/gardennotebook?user=root&password=password&useSSL=false&allowPublicKeyRetrieval=true&sslMode=DISABLED&connectTimeout=1000&cachePrepStmts=true&prepStmtCacheSize=250&prepStmtCacheSqlLimit=2048&maintainTimeStats=false&disableMariaDbDriver=true&serverTimezone=UTC");
//        DBConnection.forceConnectionString(DBConnection.RDBMS_ENUM.MySQL,"jdbc:mysql://localhost:3306/notebook3?user=root&password=password&useSSL=false&allowPublicKeyRetrieval=true&sslMode=DISABLED&connectTimeout=1000&cachePrepStmts=true&prepStmtCacheSize=250&prepStmtCacheSqlLimit=2048&maintainTimeStats=false&disableMariaDbDriver=true&serverTimezone=UTC");
    }

    @AfterAll
    void tearDown()
    {
        try
        {
            myTest.delete();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @BeforeEach
    void setUp()
    {
        myTest = new CroppingPlanBuilder(wls);
    }

    @Test
    @Order(1)
    void cropRotationGroupId()
    {
        myTest = new CroppingPlanBuilder();
        assertFalse(myTest.needSave());
        assertFalse(myTest.canSave());
        myTest.cropRotationGroupId(1);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());
        assertThrows(IllegalStateException.class, ()->myTest.save());
    }

    @Test
    @Order(2)
    void cropRotationGroup()
    {
        myTest = new CroppingPlanBuilder();
        assertFalse(myTest.needSave());
        assertFalse(myTest.canSave());
        ICropRotationGroup crg = null;
        try
        {
            crg = new CropRotationGroupLister().id(1).fetch().get(0);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        myTest.cropRotationGroup(crg);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());
        assertThrows(IllegalStateException.class, ()->myTest.save());
    }

    @Test
    @Order(20)
    void locationId()
    {
        myTest = new CroppingPlanBuilder();
        assertFalse(myTest.needSave());
        assertFalse(myTest.canSave());
        myTest.locationId(1);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());
        assertThrows(IllegalStateException.class, ()->myTest.save());
    }

    @Test
    @Order(21)
    void location()
    {
        myTest = new CroppingPlanBuilder();
        assertFalse(myTest.needSave());
        assertFalse(myTest.canSave());
        ILocation loc = null;
        try
        {
            loc = new LocationLister().id(1).fetch().get(0);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        myTest.location(loc);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());
        assertThrows(IllegalStateException.class, ()->myTest.save());
    }

    @Test
    @Order(30)
    void yearOfPlan()
    {
        myTest = new CroppingPlanBuilder();
        assertFalse(myTest.needSave());
        assertFalse(myTest.canSave());
        myTest.yearOfPlan(Year.now());
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());
        assertThrows(IllegalStateException.class, ()->myTest.save());
    }

    @Test
    @Order(40)
    void save()
    {
        myTest = new CroppingPlanBuilder();
        assertFalse(myTest.needSave());
        assertFalse(myTest.canSave());
        ICropRotationGroup crg = null;
        try
        {
            crg = new CropRotationGroupLister().id(1).fetch().get(0);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        myTest.cropRotationGroup(crg);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());

        ILocation loc = null;
        try
        {
            loc = new LocationLister().id(1).fetch().get(0);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        myTest.location(loc);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());

        myTest.yearOfPlan(Year.now());
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());

        try
        {
            wls = (CroppingPlan) (myTest.save());
//            System.out.println("test 40: wls: "+wls);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        assertNotNull(wls);
   }

    @Test
    @Order(50)
    void addComment()
    {
        assertNotNull(wls);
        assertFalse(myTest.needSave());
        myTest.addComment(testComments.get(0));
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(1, comments.size());
        assertEquals(testComments.get(0), comments.get(0).getComment());
        assertEquals(today, comments.get(0).getDate());
    }

    @Test
    @Order(51)
    void addMultiComment()
    {
        assertFalse(myTest.needSave());
        myTest.addComment(testComments.get(1));
        assertTrue(myTest.needSave());
        myTest.addComment(testComments.get(2), testComments.get(3));
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 4);
        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 4)));
        assertTrue(testComments.subList(0, 4).containsAll(foundComments));

        assertEquals(today, comments.get(1).getDate());
        assertEquals(today, comments.get(2).getDate());
        assertEquals(today, comments.get(3).getDate());
    }

    @Test
    @Order(52)
    void addCommentListOfStrings()
    {
        assertFalse(myTest.needSave());
        List<String> commentValues = List.of(testComments.get(4));
        myTest.addComment(commentValues);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 5);
        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 5)));
        assertTrue(testComments.subList(0, 5).containsAll(foundComments));
        assertEquals(today, comments.get(4).getDate());
    }

    @Test
    @Order(53)
    void addCommentMultiListOfStrings()
    {
        assertFalse(myTest.needSave());
        List<String> commentValues = List.of(testComments.get(5));
        myTest.addComment(commentValues);
        assertTrue(myTest.needSave());
        commentValues = List.of(testComments.get(6), testComments.get(7));
        myTest.addComment(commentValues);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 8);

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));
        assertEquals(today, comments.get(5).getDate());
        assertEquals(today, comments.get(6).getDate());
        assertEquals(today, comments.get(7).getDate());
    }

    @Test
    @Order(54)
    void changeCommentString()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(comments.size(), 8);
        testComments.remove(comments.get(0).getComment());
        myTest.changeComment(comments.get(0), "first comment changed");
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(8, comments.size());
        assertEquals(comments.get(0).getComment(), "first comment changed");
        assertEquals(today, comments.get(0).getDate());
        List<String> foundComments = comments.stream().skip(1).map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 7)));
        assertTrue(testComments.subList(0, 7).containsAll(foundComments));

        assertEquals(today, comments.get(1).getDate());
        assertEquals(today, comments.get(2).getDate());
        assertEquals(today, comments.get(3).getDate());
        assertEquals(today, comments.get(4).getDate());
        assertEquals(today, comments.get(5).getDate());
        assertEquals(today, comments.get(6).getDate());
        assertEquals(today, comments.get(7).getDate());
        testComments.add("first comment changed");
    }

    @Test
    @Order(55)
    void changeCommentStringNull()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(8, comments.size());
        myTest.changeComment(comments.get(0), (String)null);
        assertFalse(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(8, comments.size());

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));

        assertEquals(today, comments.get(0).getDate());
        assertEquals(today, comments.get(1).getDate());
        assertEquals(today, comments.get(2).getDate());
        assertEquals(today, comments.get(3).getDate());
        assertEquals(today, comments.get(4).getDate());
        assertEquals(today, comments.get(5).getDate());
        assertEquals(today, comments.get(6).getDate());
        assertEquals(today, comments.get(7).getDate());
    }

    @Test
    @Order(56)
    void changeCommentStringEmpty()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(8, comments.size());
        myTest.changeComment(comments.get(0), "");
        assertFalse(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(8, comments.size());

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));

        assertEquals(today, comments.get(0).getDate());
        assertEquals(today, comments.get(1).getDate());
        assertEquals(today, comments.get(2).getDate());
        assertEquals(today, comments.get(3).getDate());
        assertEquals(today, comments.get(4).getDate());
        assertEquals(today, comments.get(5).getDate());
        assertEquals(today, comments.get(6).getDate());
        assertEquals(today, comments.get(7).getDate());
    }

    @Test
    @Order(57)
    void changeCommentDate()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(8, comments.size());
        LocalDate targetDate = today.minusDays(2);
        String targetText = comments.get(1).getComment();
        myTest.changeComment(comments.get(1), targetDate);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();   //  comments are sorted by date!
        assertEquals(8, comments.size());
        assertEquals(targetText, comments.get(0).getComment());
        assertEquals(targetDate, comments.get(0).getDate());

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));

        assertEquals(today, comments.get(1).getDate());

        assertEquals(today, comments.get(2).getDate());
        assertEquals(today, comments.get(3).getDate());
        assertEquals(today, comments.get(4).getDate());
        assertEquals(today, comments.get(5).getDate());
        assertEquals(today, comments.get(6).getDate());
        assertEquals(today, comments.get(7).getDate());
    }

    @Test
    @Order(58)
    void changeCommentDateNull()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(8, comments.size());
        LocalDate oldDate = comments.get(0).getDate();
        String targetText = comments.get(0).getComment();
        myTest.changeComment(comments.get(0), (LocalDate)null);
        assertFalse(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();   //  comments are sorted by date!
        assertEquals(8, comments.size());
        assertEquals(comments.get(0).getComment(), targetText);
        assertNotNull(comments.get(0).getDate());
        assertEquals(oldDate, comments.get(0).getDate());

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));

        assertEquals(today, comments.get(1).getDate());

        assertEquals(today, comments.get(2).getDate());
        assertEquals(today, comments.get(3).getDate());
        assertEquals(today, comments.get(4).getDate());
        assertEquals(today, comments.get(5).getDate());
        assertEquals(today, comments.get(6).getDate());
        assertEquals(today, comments.get(7).getDate());
    }

    @Test
    @Order(59)
    void changeCommentDateFuture()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(8, comments.size());
        LocalDate oldDate = comments.get(0).getDate();
        LocalDate targetDate = today.plusDays(2);
        String targetText = comments.get(0).getComment();
        myTest.changeComment(comments.get(0), targetDate);
        assertFalse(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();   //  comments are sorted by date!
        assertEquals(8, comments.size());
        assertEquals(targetText, comments.get(0).getComment());
        assertNotEquals(targetDate, comments.get(0).getDate());
        assertEquals(oldDate, comments.get(0).getDate());

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 8)));
        assertTrue(testComments.subList(0, 8).containsAll(foundComments));

        assertEquals(today, comments.get(1).getDate());

        assertEquals(today, comments.get(2).getDate());
        assertEquals(today, comments.get(3).getDate());
        assertEquals(today, comments.get(4).getDate());
        assertEquals(today, comments.get(5).getDate());
        assertEquals(today, comments.get(6).getDate());
        assertEquals(today, comments.get(7).getDate());
    }

    @Test
    @Order(60)
    void changeCommentStringDate()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(8, comments.size());
        LocalDate targetDate = today.minusDays(3);
        String targetText = comments.get(2).getComment();
        testComments.remove(targetText);
        myTest.changeComment(comments.get(2), targetDate, "third comment changed");
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();   //  comments are sorted by date!
        assertEquals(8, comments.size());
        assertEquals("third comment changed", comments.get(0).getComment());
        assertEquals(targetDate, comments.get(0).getDate());

        List<String> foundComments = comments.stream().skip(1).map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 7)));
        assertTrue(testComments.subList(0, 7).containsAll(foundComments));

        assertEquals(today.minusDays(2), comments.get(1).getDate());

        assertEquals(today, comments.get(2).getDate());

        assertEquals(today, comments.get(3).getDate());
        assertEquals(today, comments.get(4).getDate());
        assertEquals(today, comments.get(5).getDate());
        assertEquals(today, comments.get(6).getDate());
        assertEquals(today, comments.get(7).getDate());
        testComments.add("third comment changed");
    }

    @Test
    @Order(70)
    void deleteCommentById()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(8, comments.size());
        int key1 = comments.get(0).getKey();
        String targetText = comments.get(0).getComment();
        testComments.remove(targetText);
        myTest.deleteComment(key1);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(7, comments.size());

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 7)));
        assertTrue(testComments.subList(0, 7).containsAll(foundComments));
    }

    @Test
    @Order(71)
    void deleteMultiCommentById()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(7, comments.size());
        int key1 = comments.get(0).getKey();
        int key2 = comments.get(1).getKey();
        int key3 = comments.get(2).getKey();

        String targetText = comments.get(0).getComment();
        testComments.remove(targetText);
        targetText = comments.get(1).getComment();
        testComments.remove(targetText);
        targetText = comments.get(2).getComment();
        testComments.remove(targetText);

        myTest.deleteComment(key1);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        myTest.deleteComment(key2, key3);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(4, comments.size());

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 4)));
        assertTrue(testComments.subList(0, 4).containsAll(foundComments));
    }

    @Test
    @Order(72)
    void deleteCommentByArray()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(4, comments.size());

        String targetText = comments.get(0).getComment();
        testComments.remove(targetText);

        myTest.deleteComment(comments.get(0));
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(3, comments.size());

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 3)));
        assertTrue(testComments.subList(0, 3).containsAll(foundComments));
    }

    @Test
    @Order(73)
    void deleteMultiCommentByArray()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(3, comments.size());
        myTest.deleteComment(comments.get(0));
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        myTest.deleteComment(comments.get(1), comments.get(2));
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(0, comments.size());
    }

    @Test
    @Order(80)
    void setUpForDeleteCommentByList()
    {
        testComments = new ArrayList<>(List.of("this is the first comment", "this is the second comment", "this is the third comment", "this is the fourth comment"));
        myTest.addComment(testComments);
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        List<IComment> comments = wls.getComments();
        assertEquals(4, comments.size());


        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments));
        assertTrue(testComments.containsAll(foundComments));

        assertEquals(today, comments.get(0).getDate());
        assertEquals(today, comments.get(1).getDate());
        assertEquals(today, comments.get(2).getDate());
        assertEquals(today, comments.get(3).getDate());
    }

    @Test
    @Order(81)
    void deleteCommentByList()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(4, comments.size());
        List<IComment> listComments = List.of(comments.get(0));
        testComments.remove(comments.get(0).getComment());
        myTest.deleteComment(listComments);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(3, comments.size());

        List<String> foundComments = comments.stream().map(IComment::getComment).toList();
        assertTrue(foundComments.containsAll(testComments.subList(0, 3)));
        assertTrue(testComments.subList(0, 3).containsAll(foundComments));

        assertEquals(today, comments.get(0).getDate());
        assertEquals(today, comments.get(1).getDate());
        assertEquals(today, comments.get(2).getDate());
    }

    @Test
    @Order(82)
    void deleteMultiCommentByList()
    {
        assertFalse(myTest.needSave());
        List<IComment> comments = wls.getComments();
        assertEquals(3, comments.size());
        List<IComment> testComments = List.of(comments.get(0));
        myTest.deleteComment(testComments);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());

        testComments = List.of(comments.get(1), comments.get(2));
        myTest.deleteComment(testComments);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) myTest.save();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        comments = wls.getComments();
        assertEquals(0, comments.size());
    }

    @Test
    @Order(90)
    void canDelete()
    {
        try
        {
            assertTrue(myTest.canDelete());
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    @Test
    @Order(91)
    void delete()
    {
        try
        {
            myTest.delete();
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
    }

    //  Modification tests

    @Test
    @Order(100)
    void modifyCropGroup()
    {
        myTest = new CroppingPlanBuilder();
        assertFalse(myTest.needSave());
        assertFalse(myTest.canSave());
        ICropRotationGroup crg = null;
        try
        {
            crg = new CropRotationGroupLister().id(1).fetch().get(0);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        myTest.cropRotationGroup(crg);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());

        ILocation loc = null;
        try
        {
            loc = new LocationLister().id(1).fetch().get(0);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        myTest.location(loc);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());

        myTest.yearOfPlan(Year.now());
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());

        try
        {
            wls = (CroppingPlan) (myTest.save());
//            System.out.println("test 40: wls: "+wls);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        assertNotNull(wls);

        //  setup complete!
        myTest = new CroppingPlanBuilder(wls);
        assertFalse(myTest.needSave());
//        assertFalse(myTest.canSave());
        myTest.cropRotationGroupId(2);
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) (myTest.save());
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertEquals(2, wls.getCropRotationGroupId());
    }

    @Test
    @Order(101)
    void modifyLocation()
    {
        myTest = new CroppingPlanBuilder();
        assertFalse(myTest.needSave());
        assertFalse(myTest.canSave());
        ICropRotationGroup crg = null;
        try
        {
            crg = new CropRotationGroupLister().id(1).fetch().get(0);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        myTest.cropRotationGroup(crg);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());

        ILocation loc = null;
        try
        {
            loc = new LocationLister().id(1).fetch().get(0);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        myTest.location(loc);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());

        myTest.yearOfPlan(Year.now());
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());

        try
        {
            wls = (CroppingPlan) (myTest.save());
//            System.out.println("test 40: wls: "+wls);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        assertNotNull(wls);

        //  setup complete!
        myTest = new CroppingPlanBuilder(wls);
        assertFalse(myTest.needSave());
//        assertFalse(myTest.canSave());
        myTest.locationId( 2);  //  non-existant!
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        assertThrows(GNDBException.class, ()->myTest.save());
    }

    @Test
    @Order(102)
    void modifyYear()
    {
        myTest = new CroppingPlanBuilder();
        assertFalse(myTest.needSave());
        assertFalse(myTest.canSave());
        ICropRotationGroup crg = null;
        try
        {
            crg = new CropRotationGroupLister().id(1).fetch().get(0);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        myTest.cropRotationGroup(crg);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());

        ILocation loc = null;
        try
        {
            loc = new LocationLister().id(1).fetch().get(0);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        myTest.location(loc);
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());

        myTest.yearOfPlan(Year.now());
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());

        try
        {
            wls = (CroppingPlan) (myTest.save());
//            System.out.println("test 40: wls: "+wls);
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertFalse(myTest.needSave());
        assertNotNull(wls);

        //  setup complete!
        myTest = new CroppingPlanBuilder(wls);
        assertFalse(myTest.needSave());
//        assertFalse(myTest.canSave());
        myTest.yearOfPlan(Year.now().plusYears(1));
        assertTrue(myTest.needSave());
        assertTrue(myTest.canSave());
        try
        {
            wls = (CroppingPlan) (myTest.save());
        }
        catch (GNDBException e)
        {
            e.printStackTrace();
        }
        assertEquals(Year.now().plusYears(1), wls.getYearOfPlan());
    }



    //  Failing tests

    @Test
    @Order(200)
    void cannotSave()
    {
        myTest = new CroppingPlanBuilder();
        assertFalse(myTest.needSave());
        myTest.yearOfPlan(Year.now());
        assertTrue(myTest.needSave());
        assertFalse(myTest.canSave());
        assertThrows(IllegalStateException.class, myTest::save);
    }
}