/*
 * Copyright (C) 2018, 2019, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
    3.0.0	Use DBKeyHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;
import java.util.ArrayList;
import java.util.Collections;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;
import jakarta.json.JsonObjectBuilder;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class WeatherLister implements IWeatherLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private final DBKeyHandler<IWeather> useWeather = new DBKeyHandler<>("weatherId");

	private final DBKeyHandler<IWeatherCondition> useWeatherCondition = new DBKeyHandler<>("weatherConditionId");

	private boolean useFromDate = false;
	private LocalDate fromDate;
	private boolean useToDate = false;
	private LocalDate toDate;

	private boolean useWhere = false;

	WeatherLister() {}

	@Override
	public List<IWeather> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		List<IWeather> vals = new ArrayList<>();
		boolean fetchAll = !useWhere;
        String query = "";
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = buildSQL_MySQL();
			case hsqldb -> query = buildSQL_hsqldb();
			case MSSQLServer -> query = buildSQL_MSSQLServer();
			default -> {
				LOGGER.error("fetch(): no known rdbms");
				throw new GNDBException(new IllegalStateException("WeatherLister: fetch(): no known RDBMS"));
			}
		}
LOGGER.debug("fetch(): query: {}", query);

		try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
		{
			ResultSet rs = stmt.executeQuery(query);
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> vals = processResults_MySQL(rs);
				case hsqldb -> vals = processResults_hsqldb(rs);
				case MSSQLServer -> vals = processResults_MSSQLServer(rs);
				default -> {
					LOGGER.error("fetch(): no known rdbms");
					throw new GNDBException(new IllegalStateException("WeatherLister: fetch(): no known RDBMS"));
				}
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("fetch(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

		if (vals.isEmpty()) return Collections.emptyList();

		for (IWeather vx : vals)
		{
			MySQLCache.cacheWeather.putIfAbsent(vx.getKey(), vx);
		}
		if (fetchAll) { MySQLCache.completeWeather = true; }

		populateWeatherCondition();

LOGGER.traceExit(log4jEntryMsg);
		return vals;
	}	// fetch()

    private String buildSQL_MySQL()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from weather as d ");
        query.append("left join (select * from comment where ownerType = 'WE') as c ");
        query.append("on d.weatherId = c.ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.weatherConditionId, d.weatherId, c.date");
        return query.toString();
    }

    private String buildSQL_hsqldb()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from weather as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'WE') as c ");
        query.append("on d.weatherId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.weatherConditionId, d.weatherId, c_date");
        return query.toString();
    }

    private String buildSQL_MSSQLServer()
    {
        StringBuilder query = new StringBuilder("select d.weatherId as d_weatherId, " +
                                                        "d.weatherConditionId as d_weatherConditionId, " +
                                                        "d.date as d_date, " +
                                                        "d.severity as d_severity, " +
                                                        "d.lastUpdated as d_lastUpdated, " +
                                                        "d.created as d_created, " +
                                                        " c.* from weather as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'WE') as c ");
        query.append("on d.weatherId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.date, d.weatherConditionId, d.weatherId, c_date");
        return query.toString();
    }

    private void buildCommonSQL(StringBuilder query)
    {
        boolean first = true;

		if (useWeather.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(useWeather.getSQLClause("d"));
			first = false;
			useWeather.clear();
		}

		if (useWeatherCondition.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(useWeatherCondition.getSQLClause("d"));
			first = false;
			useWeatherCondition.clear();
		}

        if (this.useFromDate)
        {
            if (first) query.append(" where ");
            else query.append(" and");

            query.append(" d.date >= '").append(this.fromDate).append("'");
            first = false;
            this.useFromDate = false;
        }

        if (this.useToDate)
        {
            if (first) query.append(" where ");
            else query.append(" and");

            query.append(" d.date <= '").append(this.toDate).append("'");
            this.useToDate = false;
        }  
    }   //  buildCommonSQL()

	private List<IWeather> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<IWeather> tempList = new ArrayList<>();

		Weather item = null;

		while (rs.next())
		{
			int weatherId = rs.getInt("d.weatherId");
			int weatherConditionId = rs.getInt("d.weatherConditionId");
			LocalDate date = rs.getDate("d.date").toLocalDate();
			String severity = rs.getString("d.severity");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("weatherId: {}, weatherConditionId: {}, date: {}, severity: {}, lastUpdated: {}, created: {}",
                weatherId, weatherConditionId, date, severity, lastUpdated, created);
			if (item != null && weatherId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_MySQL(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("commentId"),
					rs.getInt("ownerId"),
					"WE",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): extra comment is: {}", comm);
				item = new Weather(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
					item = new Weather(weatherId, weatherConditionId, date, severity, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						weatherId,
						"WE",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): first comment is: {}", comm);
					item = new Weather(weatherId, weatherConditionId, date, severity, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<IWeather> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<IWeather> tempList = new ArrayList<>();

		Weather item = null;

		while (rs.next())
		{
			int weatherId = rs.getInt("weather.weatherId");
			int weatherConditionId = rs.getInt("weather.weatherConditionId");
			LocalDate date = rs.getDate("weather.date").toLocalDate();
			String severity = rs.getString("weather.severity");
			LocalDateTime lastUpdated = rs.getTimestamp("weather.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("weather.created").toLocalDateTime();
LOGGER.debug("weatherId: {}, weatherConditionId: {}, date: {}, severity: {}, lastUpdated: {}, created: {}",
                weatherId, weatherConditionId, date, severity, lastUpdated, created);
			if (item != null && weatherId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"WE",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new Weather(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new Weather(weatherId, weatherConditionId, date, severity, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						weatherId,
						"WE",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new Weather(weatherId, weatherConditionId, date, severity, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<IWeather> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<IWeather> tempList = new ArrayList<>();

		Weather item = null;

		while (rs.next())
		{
			int weatherId = rs.getInt("d_weatherId");
			int weatherConditionId = rs.getInt("d_weatherConditionId");
			LocalDate date = rs.getDate("d_date").toLocalDate();
			String severity = rs.getString("d_severity");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("weatherId: {}, weatherConditionId: {}, date: {}, severity: {}, lastUpdated: {}, created: {}",
                weatherId, weatherConditionId, date, severity, lastUpdated, created);
			if (item != null && weatherId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"WE",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new Weather(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new Weather(weatherId, weatherConditionId, date, severity, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						weatherId,
						"WE",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new Weather(weatherId, weatherConditionId, date, severity, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	/*
	*	Populate the parents slot(s)
	*/
	private void populateWeatherCondition() throws GNDBException
	{
// parent table type: TF
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populateWeatherCondition()");

// just make sure everything is cached
		new WeatherConditionLister().load();
LOGGER.traceExit(log4jEntryMsg);
	}

	void clear()
	{
		MySQLCache.cacheWeather.clear();
		MySQLCache.completeWeather = false;
	}

	/**
	*
	*Select only the Weather entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	IWeatherLister id(int... vals)
	{
		useWeather.id(vals);
//		useWhere = true;
		useWhere = useWhere || useWeather.isUsed();
		return this;
	}

	@Override
	public IWeatherLister weather(IWeather... items)
	{
		useWeather.item(items);
//		useWhere = true;
		useWhere = useWhere || useWeather.isUsed();
		return this;
	}

	@Override
	public IWeatherLister weather(List<IWeather> items)
	{
		useWeather.item(items);
//		useWhere = true;
		useWhere = useWhere || useWeather.isUsed();
		return this;
	}

	@Override
	public IWeatherLister weatherCondition(IWeatherCondition... items)
	{
		useWeatherCondition.item(items);
//		useWhere = true;
		useWhere = useWhere || useWeatherCondition.isUsed();
		return this;
	}

	@Override
	public IWeatherLister weatherCondition(List<IWeatherCondition> items)
	{
		useWeatherCondition.item(items);
//		useWhere = true;
		useWhere = useWhere || useWeatherCondition.isUsed();
		return this;
	}

	@Override
	public IWeatherLister fromDate(LocalDate item)
	{
		if (item == null) return this;
		this.fromDate = item;
		this.useFromDate = true;
		this.useWhere = true;
		return this;
	}

	@Override
	public IWeatherLister toDate(LocalDate item)
	{
		if (item == null) return this;
		this.toDate = item;
		this.useToDate = true;
		this.useWhere = true;
		return this;
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		useWhere = false;
		fetch();

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (IWeather ihc : MySQLCache.cacheWeather.values())
		{
			Weather hc = (Weather)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "Weather");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "Weather.json"), false)))
		{
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJson(): IOException", ex);
		}
	}	// toJson

}
