/*
 * Copyright (C) 2018, 2019, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
	2.5.0   Replace description with quantity and unit - can ONLY record produce sales.
    3.0.0	Use DBKeyHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.time.LocalDateTime;
import java.util.*;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;
import jakarta.json.JsonObjectBuilder;

import java.math.BigDecimal;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class SaleItemLister implements ISaleItemLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private final DBKeyHandler<ISaleItem> useSaleItem = new DBKeyHandler<>("saleItemId");

	private final DBKeyHandler<IPlantSpecies> usePlantSpecies = new DBKeyHandler<>("plantSpeciesId");

	private final DBKeyHandler<IPlantVariety> usePlantVariety = new DBKeyHandler<>("plantVarietyId");

	private final DBKeyHandler<ISale> useSale = new DBKeyHandler<>("saleId");

	private boolean useWhere = false;

	SaleItemLister() {}

	@Override
	public List<ISaleItem> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		List<ISaleItem> vals = new ArrayList<>();
		boolean fetchAll = !useWhere;
        String query = "";
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = buildSQL_MySQL();
			case hsqldb -> query = buildSQL_hsqldb();
			case MSSQLServer -> query = buildSQL_MSSQLServer();
			default -> {
				LOGGER.error("fetch(): no known rdbms");
				throw new GNDBException(new IllegalStateException("SaleItemLister: fetch(): no known RDBMS"));
			}
		}
LOGGER.debug("fetch(): query: {}", query);

		try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
		{
			ResultSet rs = stmt.executeQuery(query);
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> vals = processResults_MySQL(rs);
				case hsqldb -> vals = processResults_hsqldb(rs);
				case MSSQLServer -> vals = processResults_MSSQLServer(rs);
				default -> {
					LOGGER.error("fetch(): no known rdbms");
					throw new GNDBException(new IllegalStateException("SaleItemLister: fetch(): no known RDBMS"));
				}
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("fetch(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

		if (vals.isEmpty()) return Collections.emptyList();

		for (ISaleItem vx : vals)
		{
			MySQLCache.cacheSaleItem.putIfAbsent(vx.getKey(), vx);
		}
		if (fetchAll) { MySQLCache.completeSaleItem = true; }

		populatePlantSpecies(vals, fetchAll);

		populatePlantVariety(vals, fetchAll);

		populateSale(vals, fetchAll);

LOGGER.traceExit(log4jEntryMsg);
		return vals;
	}	// fetch()

    private String buildSQL_MySQL()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from saleitem as d ");
        query.append("left join (select * from comment where ownerType = 'SI') as c ");
        query.append("on d.saleItemId = c.ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.saleId, d.saleItemId, c.date");
        return query.toString();
    }   //  buildSQL_MySQL()

    private String buildSQL_hsqldb()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from saleitem as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'SI') as c ");
        query.append("on d.saleItemId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.saleId, d.saleItemId, c_date");
        return query.toString();
    }   //  buildSQL_hsqldb()

    private String buildSQL_MSSQLServer()
    {
        StringBuilder query = new StringBuilder("select d.saleItemId as d_saleItemId, " +
                                                        "d.saleId as d_saleId, " +
                                                        "d.plantSpeciesId as d_plantSpeciesId, " +
                                                        "d.plantVarietyId as d_plantVarietyId, " +
                                                        "d.quantity as d_quantity, " +
                                                        "d.unit as d_unit, " +
                                                        "d.itemCost as d_itemCost, " +
                                                        "d.currency as d_currency, " +
                                                        "d.lastUpdated as d_lastUpdated, " +
                                                        "d.created as d_created," +
                                                        " c.* from saleitem as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'SI') as c ");
        query.append("on d.saleItemId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.saleId, d.saleItemId, c_date");
        return query.toString();
    }   //  buildSQL_MSSQLServer()

    private void buildCommonSQL(StringBuilder query)
    {
        boolean first = true;
		if (useSaleItem.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and ");
			query.append(useSaleItem.getSQLClause("d"));
			first = false;
			useSaleItem.clear();
		}

//		if (this.usePlantSpeciesId || this.usePlantVarietyId)
		if (this.usePlantSpecies.isUsed() || this.usePlantVariety.isUsed())
        {
            if (first)
                query.append(" where ");
            else
                query.append(" and ");

			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" ( ");

			if (this.usePlantSpecies.isUsed())
			{
				query.append(usePlantSpecies.getSQLClause("d"));
			}

			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" or ");
			if (this.usePlantVariety.isUsed())
			{
				query.append(usePlantVariety.getSQLClause("d"));
			}

			if (this.usePlantSpecies.isUsed() && this.usePlantVariety.isUsed())
				query.append(" ) ");
			first = false;
			usePlantSpecies.clear();
			usePlantVariety.clear();
        }
		if (useSale.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and ");
			query.append(useSale.getSQLClause("d"));
			first = false;
			useSale.clear();
		}

	}   //  buildCommonSQL()

	private List<ISaleItem> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<ISaleItem> tempList = new ArrayList<>();

		SaleItem item = null;

		while (rs.next())
		{
			int saleItemId = rs.getInt("d.saleItemId");
			int saleId = rs.getInt("d.saleId");
			int tmp_plantSpeciesId = rs.getInt("d.plantSpeciesId");	// plantSpeciesId is now mandatory
			Integer plantSpeciesId = rs.wasNull() ? null : tmp_plantSpeciesId;
			int tmp_plantVarietyId = rs.getInt("d.plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			BigDecimal quantity = rs.getBigDecimal("d.quantity");
			String unit = rs.getString("d.unit");
			BigDecimal itemCost = rs.getBigDecimal("d.itemCost");
			String currency = rs.getString("d.currency");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
LOGGER.debug("lastUpdated: {}", lastUpdated);
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("created: {}", created);
LOGGER.debug("saleItemId: {}, saleId: {}, plantSpeciesId: {}, plantVarietyId: {}, quantity: {}, unit: {}, itemCost: {}, currency: {}, lastUpdated: {}, created: {}",
                saleItemId, saleId, plantSpeciesId, plantVarietyId, quantity, unit, itemCost, currency, lastUpdated, created);
			if (item != null && saleItemId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_MySQL(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("commentId"),
					rs.getInt("ownerId"),
					"SI",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): extra comment is: {}", comm);
				item = new SaleItem(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
					item = new SaleItem(saleItemId, saleId, plantSpeciesId, plantVarietyId, quantity, unit, itemCost, currency, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						saleItemId,
						"SI",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): first comment is: {}", comm);
					item = new SaleItem(saleItemId, saleId, plantSpeciesId, plantVarietyId, quantity, unit, itemCost, currency, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<ISaleItem> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<ISaleItem> tempList = new ArrayList<>();

		SaleItem item = null;

		while (rs.next())
		{
			int saleItemId = rs.getInt("saleitem.saleItemId");
			int saleId = rs.getInt("saleitem.saleId");
			int tmp_plantSpeciesId = rs.getInt("saleitem.plantSpeciesId");
			Integer plantSpeciesId = rs.wasNull() ? null : tmp_plantSpeciesId;
			int tmp_plantVarietyId = rs.getInt("saleitem.plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			BigDecimal quantity = rs.getBigDecimal("saleitem.quantity");
			String unit = rs.getString("saleitem.unit");
			BigDecimal itemCost = rs.getBigDecimal("saleitem.itemCost");
			String currency = rs.getString("saleitem.currency");
			LocalDateTime lastUpdated = rs.getTimestamp("saleitem.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("saleitem.created").toLocalDateTime();
LOGGER.debug("saleItemId: {}, saleId: {}, plantSpeciesId: {}, plantVarietyId: {}, quantity: {}, unit: {}, itemCost: {}, currency: {}, lastUpdated: {}, created: {}",
                saleItemId, saleId, plantSpeciesId, plantVarietyId, quantity, unit, itemCost, currency, lastUpdated, created);
			if (item != null && saleItemId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"SI",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new SaleItem(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new SaleItem(saleItemId, saleId, plantSpeciesId, plantVarietyId, quantity, unit, itemCost, currency, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						saleItemId,
						"SI",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new SaleItem(saleItemId, saleId, plantSpeciesId, plantVarietyId, quantity, unit, itemCost, currency, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<ISaleItem> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<ISaleItem> tempList = new ArrayList<>();

		SaleItem item = null;

		while (rs.next())
		{
			int saleItemId = rs.getInt("d_saleItemId");
			int saleId = rs.getInt("d_saleId");
			int tmp_plantSpeciesId = rs.getInt("d_plantSpeciesId");
			Integer plantSpeciesId = rs.wasNull() ? null : tmp_plantSpeciesId;
			int tmp_plantVarietyId = rs.getInt("d_plantVarietyId");
			Integer plantVarietyId = rs.wasNull() ? null : tmp_plantVarietyId;
			BigDecimal quantity = rs.getBigDecimal("d_quantity");
			String unit = rs.getString("d_unit");
			BigDecimal itemCost = rs.getBigDecimal("d_itemCost");
			String currency = rs.getString("d_currency");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("saleItemId: {}, saleId: {}, plantSpeciesId: {}, plantVarietyId: {}, quantity: {}, unit: {}, itemCost: {}, currency: {}, lastUpdated: {}, created: {}",
                saleItemId, saleId, plantSpeciesId, plantVarietyId, quantity, unit, itemCost, currency, lastUpdated, created);
			if (item != null && saleItemId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"SI",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new SaleItem(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new SaleItem(saleItemId, saleId, plantSpeciesId, plantVarietyId, quantity, unit, itemCost, currency, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						saleItemId,
						"SI",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new SaleItem(saleItemId, saleId, plantSpeciesId, plantVarietyId, quantity, unit, itemCost, currency, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	/*
	*	Populate the parents slot(s)
	*/
	private void populatePlantSpecies(List<ISaleItem> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: TD
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populatePlantSpecies()");

		if (fetchAll)
		{
			new PlantSpeciesLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			map(c -> ((SaleItem)c).getPlantSpeciesId()).			// plantSpeciesId is now mandatory
			mapToInt(Integer::intValue).filter(c -> c>0).distinct().
			toArray();

		if (keys.length > 0)
		{
			new PlantSpeciesLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	private void populatePlantVariety(List<ISaleItem> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: NL
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populatePlantVariety()");

		if (fetchAll)
		{
			new PlantVarietyLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			map(c -> ((SaleItem)c).getPlantVarietyId()).filter(Objects::nonNull).
			mapToInt(Integer::intValue).filter(c -> c>0).distinct().
			toArray();

		if (keys.length > 0)
		{
			new PlantVarietyLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	private void populateSale(List<ISaleItem> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: NL
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populateSale()");

		if (fetchAll)
		{
			new SaleLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			mapToInt(c -> ((SaleItem)c).getSaleId()).filter(c -> c>0).distinct().
			filter(c -> !MySQLCache.cacheSale.containsKey(c)).
			toArray();

		if (keys.length > 0)
		{
			new SaleLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	void clear()
	{
		MySQLCache.cacheSaleItem.clear();
		MySQLCache.completeSaleItem = false;
	}

	/**
	*
	*Select only the SaleItem entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	SaleItemLister id(int... vals)
	{
		useSaleItem.id(vals);
		useWhere = useWhere || useSaleItem.isUsed();
		return this;
	}

	@Override
	public ISaleItemLister saleItem(ISaleItem... items)
	{
		useSaleItem.item(items);
		useWhere = useWhere || useSaleItem.isUsed();
		return this;
	}

	@Override
	public ISaleItemLister saleItem(List<ISaleItem> items)
	{
		useSaleItem.item(items);
		useWhere = useWhere || useSaleItem.isUsed();
		return this;
	}

	@Override
	public ISaleItemLister plantSpecies(IPlantSpecies... items)
	{
		usePlantSpecies.item(items);
		useWhere = useWhere || usePlantSpecies.isUsed();
		return this;
	}

	@Override
	public ISaleItemLister plantSpecies(List<IPlantSpecies> items)
	{
		usePlantSpecies.item(items);
		useWhere = useWhere || usePlantSpecies.isUsed();
		return this;
	}

	@Override
	public ISaleItemLister plantVariety(IPlantVariety... items)
	{
		usePlantVariety.item(items);
		useWhere = useWhere || usePlantVariety.isUsed();
		return this;
	}

	@Override
	public ISaleItemLister plantVariety(List<IPlantVariety> items)
	{
		usePlantVariety.item(items);
		useWhere = useWhere || usePlantVariety.isUsed();
		return this;
	}

	@Override
	public ISaleItemLister sale(ISale... items)
	{
		useSale.item(items);
		useWhere = useWhere || useSale.isUsed();
		return this;
	}

	@Override
	public ISaleItemLister sale(List<ISale> items)
	{
		useSale.item(items);
		useWhere = useWhere || useSale.isUsed();
		return this;
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		useWhere = false;
		fetch();

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (ISaleItem ihc : MySQLCache.cacheSaleItem.values())
		{
			SaleItem hc = (SaleItem)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "SaleItem");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "SaleItem.json"), false)))
		{
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJson(): IOException", ex);
		}
	}	// toJson

}
