/*
 *
 *  Copyright (C) 2021, 2023 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

/*
    Change log
	3.1.0	Use jakarta implementation of JSON
*/

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import jakarta.json.*;
import java.beans.PropertyChangeListener;
import java.time.LocalDateTime;
import java.util.*;

final class ReviewReferences implements IReviewReferences
{
    private final FlagHandler<IReviewReferences> flagHandler;
    {
        flagHandler = new FlagHandler<>(this);
    }

    private final int id;

    /*
     *   The references are for this review.
     */
    private final Integer reviewId;

    /*
     *   The review is for plants of this species.
     *@apiNote
     *   plantVarietyId may or may not be given; if absent the activity is for all (current) e.g. tomato varieties, e.g for spraying.
     *   This 'denormalises' the model but enables easy searches for e.g. 'all tomatoes'
     */
    private final Integer plantSpeciesId;

    /*
     *	The review is for plants of this variety.
     *   @apiNote
     *   If present, plantSpeciesId must be given.  This 'denormalises' the model but enables easy searches for e.g. 'all tomatoes'
     */
    private final Integer plantVarietyId;

    /*
     *   The review concerns this type of plant care
     */
    private final Integer husbandryClassId;

    /*
     *   The review concerns this type of activity
     */
    private final Integer groundworkActivityId;

    /*
     *   The review concerns this type of pest or disease or its treatment
     */
    private final Integer afflictionId;

    /*
     *	The review concerns this location
     */
    private final Integer locationId;

    /*
     *   The review concerns this pProduct
     */
    private final Integer productId;

    private final LocalDateTime lastUpdated;
    private final LocalDateTime created;
    private final List<Comment> commentsList;

    /**
     *	Build an immutable ReviewReferences entry one field at a time
     */
    ReviewReferences(
            final int id,
            final int reviewId,
            Integer plantSpeciesId,
            Integer plantVarietyId,
            Integer husbandryClassId,
            Integer groundworkActivityId,
            Integer afflictionId,
            Integer locationId,
            Integer productId,
            final LocalDateTime lastUpdated,
            final LocalDateTime created,
            final Comment... comments)
    {
        this.id = id;
        this.reviewId = reviewId;
        this.plantSpeciesId = plantSpeciesId;
        this.plantVarietyId = plantVarietyId;
        this.husbandryClassId = husbandryClassId;
        this.groundworkActivityId = groundworkActivityId;
        this.afflictionId = afflictionId;
        this.locationId = locationId;
        this.productId = productId;
        this.lastUpdated = lastUpdated;
        this.created = created;

        if (comments != null && comments.length>0)
        {
            this.commentsList = new ArrayList<>(Arrays.asList(comments));
        }
        else
        {
            this.commentsList = Collections.emptyList();
        }
    }

    /**
     *	Build an immutable ReviewReferences entry cloning the given ReviewReferences entry but adding the comments list
     */
    ReviewReferences(
            final ReviewReferences toCopy,
            final List<Comment> comments)
    {
        this.id = toCopy.id;
        this.reviewId = toCopy.reviewId;
        this.plantSpeciesId = toCopy.plantSpeciesId;
        this.plantVarietyId = toCopy.plantVarietyId;
        this.husbandryClassId = toCopy.husbandryClassId;
        this.groundworkActivityId = toCopy.groundworkActivityId;
        this.afflictionId = toCopy.afflictionId;
        this.locationId = toCopy.locationId;
        this.productId = toCopy.productId;
        this.lastUpdated = toCopy.lastUpdated;
        this.created = toCopy.created;

        if (comments != null && comments.size()>0)
        {
            if (toCopy.commentsList.size()>0)
            {
                // append new comments to previous list
                this.commentsList = new ArrayList<>(toCopy.commentsList);
                this.commentsList.addAll(comments);
            }
            else
            {	// no comments on original item
                this.commentsList = new ArrayList<>(comments);
            }
        }
        else
        {	// no new comments to add
            this.commentsList = toCopy.commentsList;
        }
    }

    /**
     *	Build an immutable ReviewReferences entry from a JSON dump.
     *	The dumped object must be complete (all non-nullable fields must have values) except
     *	the id field can be null or absent to indicate that this is a new item to be inserted.
     *
     *	@param	json	a JsonObject holding all the fields for a full initialisation.
     */
    ReviewReferences(JsonObject json)
    {
        this.id = json.getInt("id", -1);
        this.reviewId = json.getInt("reviewId", -1);

        if (json.containsKey("plantSpeciesId") && !json.isNull("plantSpeciesId"))
        {
            this.plantSpeciesId = json.getInt("plantSpeciesId");
        }
        else
        {
            this.plantSpeciesId = null;
        }

        if (json.containsKey("plantVarietyId") && !json.isNull("plantVarietyId"))
        {
            this.plantVarietyId = json.getInt("plantVarietyId");
        }
        else
        {
            this.plantVarietyId = null;
        }

        if (json.containsKey("husbandryClassId") && !json.isNull("husbandryClassId"))
        {
            this.husbandryClassId = json.getInt("husbandryClassId");
        }
        else
        {
            this.husbandryClassId = null;
        }

        if (json.containsKey("groundworkActivityId") && !json.isNull("groundworkActivityId"))
        {
            this.groundworkActivityId = json.getInt("groundworkActivityId");
        }
        else
        {
            this.groundworkActivityId = null;
        }

        if (json.containsKey("afflictionId") && !json.isNull("afflictionId"))
        {
            this.afflictionId = json.getInt("afflictionId");
        }
        else
        {
            this.afflictionId = null;
        }

        if (json.containsKey("locationId") && !json.isNull("locationId"))
        {
            this.locationId = json.getInt("locationId");
        }
        else
        {
            this.locationId = null;
        }

        if (json.containsKey("productId") && !json.isNull("productId"))
        {
            this.productId = json.getInt("productId");
        }
        else
        {
            this.productId = null;
        }

        this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
        this.created = LocalDateTime.parse(json.getString("created"));

        JsonArray jsonComments = json.getJsonArray("comments");
        if (jsonComments != null && !jsonComments.isEmpty())
        {// there is probably only one comment
            this.commentsList = new ArrayList<>(jsonComments.size());
            for (JsonObject ct : jsonComments.getValuesAs(JsonObject.class))
            {
                this.commentsList.add(new Comment(ct));
            }
        }
        else
        {
            this.commentsList = Collections.emptyList();
        }
    }	//constructor from JSON

    int getId()
    {
        return id;
    }
    @Override
    public Integer getKey()
    {
        return id;
    }

    @Override
    public NotebookEntryType getType()
    {
        return NotebookEntryType.REVIEWREFERENCES;
    }

    int getReviewId()
    {
        return reviewId;
    }
    @Override
    public Optional<IReview> getReview()
    {
        return Optional.ofNullable(MySQLCache.cacheReview.get(reviewId));
    }

    Integer getPlantSpeciesId()
    {
        return plantSpeciesId;
    }
    @Override
    public Optional<IPlantSpecies> getPlantSpecies()
    {
        return Optional.ofNullable(MySQLCache.cachePlantSpecies.get(plantSpeciesId));
    }

    Integer getPlantVarietyId()
    {
        return plantVarietyId;
    }
    @Override
    public Optional<IPlantVariety> getPlantVariety()
    {
        return Optional.ofNullable(MySQLCache.cachePlantVariety.get(plantVarietyId));
    }

    Integer getHusbandryClassId()
    {
        return husbandryClassId;
    }
    @Override
    public Optional<IHusbandryClass> getHusbandryClass()
    {
        return Optional.ofNullable(MySQLCache.cacheHusbandryClass.get(husbandryClassId));
    }

    Integer getGroundworkActivityId()
    {
        return groundworkActivityId;
    }
    @Override
    public Optional<IGroundworkActivity> getGroundworkActivity()
    {
        return Optional.ofNullable(MySQLCache.cacheGroundworkActivity.get(groundworkActivityId));
    }

    Integer getAfflictionId()
    {
        return afflictionId;
    }
    @Override
    public Optional<IAffliction> getAffliction()
    {
        return Optional.ofNullable(MySQLCache.cacheAffliction.get(afflictionId));
    }

    Integer getLocationId()
    {
        return locationId;
    }
    @Override
    public Optional<ILocation> getLocation()
    {
        return Optional.ofNullable(MySQLCache.cacheLocation.get(locationId));
    }

    Integer getProductId()
    {
        return productId;
    }
    @Override
    public Optional<IProduct> getProduct()
    {
        return Optional.ofNullable(MySQLCache.cacheProduct.get(productId));
    }

    @Override
    public LocalDateTime getLastUpdated()
    {
        return lastUpdated;
    }

    @Override
    public LocalDateTime getCreated()
    {
        return created;
    }

    @Override
    public boolean sameAs(INotebookEntry other)
    {
        if (other == null || other.getType() != this.getType())
        {
            return false;
        }
        return other.getKey().equals(this.getKey());
    }

    @Override
    public List<IComment> getComments()
    {
        return new ArrayList<>(this.commentsList);
    }

    JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
    {
        JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
        jsonBuilder.add("id", id);
        jsonBuilder.add("reviewId", reviewId);

        if (plantSpeciesId != null)
        {
            jsonBuilder.add("plantSpeciesId", plantSpeciesId);
        }
        else
        {
            jsonBuilder.addNull("plantSpeciesId");
        }
        if (plantVarietyId != null)
        {
            jsonBuilder.add("plantVarietyId", plantVarietyId);
        }
        else
        {
            jsonBuilder.addNull("plantVarietyId");
        }

        if (husbandryClassId != null)
        {
            jsonBuilder.add("husbandryClassId", husbandryClassId);
        }
        else
        {
            jsonBuilder.addNull("husbandryClassId");
        }

        if (groundworkActivityId != null)
        {
            jsonBuilder.add("groundworkActivityId", groundworkActivityId);
        }
        else
        {
            jsonBuilder.addNull("groundworkActivityId");
        }

        if (afflictionId != null)
        {
            jsonBuilder.add("afflictionId", afflictionId);
        }
        else
        {
            jsonBuilder.addNull("afflictionId");
        }

        if (locationId != null)
        {
            jsonBuilder.add("locationId", locationId);
        }
        else
        {
            jsonBuilder.addNull("locationId");
        }

        jsonBuilder.add("lastUpdated", lastUpdated.toString());
        jsonBuilder.add("created", created.toString());
        if (commentsList != null && !commentsList.isEmpty())
        {// no point writing an empty comments array (the loaders handle this)
            JsonArrayBuilder jsonComments = jsonFactory.createArrayBuilder();
            for (Comment ct : commentsList)
            {
                jsonComments.add(ct.toJson(jsonFactory));
            }
            jsonBuilder.add("comments", jsonComments);
        }
        jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "Journal");
        return jsonBuilder;
    }	//	toJson

    @Override
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener)
    {
        flagHandler.addPropertyChangeListener(propertyName, listener);
    }

    @Override
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener)
    {
        flagHandler.removePropertyChangeListener(propertyName, listener);
    }

    @Override
    public void flagDeleted()
    {
        flagHandler.flagDeleted();
    }

    @Override
    public void flagReplaced(IReviewReferences newValue)
    {
        flagHandler.flagReplaced(newValue, newValue::addPropertyChangeListener);
    }

    @Override
    public String toString() {
        return "ReviewReferences: " + "id: " + id + ", " +
                "reviewId" + reviewId + ", " +
                "plantSpeciesId" + plantSpeciesId + ", " +
                "plantVarietyId" + plantVarietyId + ", " +
                "husbandryClassId: " + husbandryClassId + ", " +
                "groundworkActivityId: " + groundworkActivityId + ", " +
                "afflictionId: " + afflictionId + ", " +
                "locationId: " + locationId + ", " +
                "productId: " + productId + ", " +
                "lastUpdated: " + lastUpdated + ", " +
                "created: " + created;
    }

}
