/*
 * Copyright (C) 2018, 2019, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.4.0   Support MS SQLServer
    3.0.0	Use DBKeyHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import java.time.LocalDateTime;
import java.util.List;
import java.util.ArrayList;
import java.util.Collections;

import java.sql.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import java.io.FileWriter;
import java.io.IOException;
import java.io.File;

import jakarta.json.JsonArrayBuilder;
import jakarta.json.JsonBuilderFactory;
import jakarta.json.JsonWriter;
import jakarta.json.JsonWriterFactory;

import jakarta.json.JsonObjectBuilder;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class RetailerHasProductLister implements IRetailerHasProductLister
{
	private static final Logger LOGGER = LogManager.getLogger();

	private final DBKeyHandler<IRetailerHasProduct> useRetailerHasProduct = new DBKeyHandler<>("retailerHasProductId");

	private final DBKeyHandler<IProduct> useProduct = new DBKeyHandler<>("productId");

	private final DBKeyHandler<IRetailer> useRetailer = new DBKeyHandler<>("retailerId");

	private boolean useWhere = false;

	RetailerHasProductLister() {}

	@Override
	public List<IRetailerHasProduct> fetch() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("fetch()");

		List<IRetailerHasProduct> vals = new ArrayList<>();
		boolean fetchAll = !useWhere;
        String query = "";
		switch (DBConnection.DB_IN_USE)
		{
			case MariaDB, MySQL -> query = buildSQL_MySQL();
			case hsqldb -> query = buildSQL_hsqldb();
			case MSSQLServer -> query = buildSQL_MSSQLServer();
			default -> {
				LOGGER.error("RetailerHasProductLister: fetch(): no known rdbms");
				throw new GNDBException(new IllegalStateException("RetailerHasProductLister: fetch(): no known RDBMS"));
			}
		}
LOGGER.debug("fetch(): query: {}", query);

		try (Connection conn = DBConnection.getConnection(); Statement stmt = conn.createStatement();)
		{
			ResultSet rs = stmt.executeQuery(query);
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> vals = processResults_MySQL(rs);
				case hsqldb -> vals = processResults_hsqldb(rs);
				case MSSQLServer -> vals = processResults_MSSQLServer(rs);
				default -> {
					LOGGER.error("RetailerHasProductLister: fetch(): no known rdbms");
					throw new GNDBException(new IllegalStateException("RetailerHasProductLister: fetch(): no known RDBMS"));
				}
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("fetch(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

		if (vals.isEmpty()) return Collections.emptyList();

		for (IRetailerHasProduct vx : vals)
		{
			MySQLCache.cacheRetailerHasProduct.putIfAbsent(vx.getKey(), vx);
		}
		if (fetchAll) { MySQLCache.completeRetailerHasProduct = true; }

		populateProduct(vals, fetchAll);

		populateRetailer(vals, fetchAll);

LOGGER.traceExit(log4jEntryMsg);
		return vals;
	}	// fetch()

    private String buildSQL_MySQL()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from retailerhasproduct as d ");
        query.append("left join (select * from comment where ownerType = 'RP') as c ");
        query.append("on d.retailerHasProductId = c.ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.retailerId, d.productId, d.retailerHasProductId, c.date");
        return query.toString();
    }   //  buildSQL_MySQL()

    private String buildSQL_hsqldb()
    {
        StringBuilder query = new StringBuilder("select d.*, c.* from retailerhasproduct as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'RP') as c ");
        query.append("on d.retailerHasProductId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.retailerId, d.productId, d.retailerHasProductId, c_date");
        return query.toString();
    }   //  buildSQL_hsqldb()

    private String buildSQL_MSSQLServer()
    {
        StringBuilder query = new StringBuilder("select d.retailerHasProductId as d_retailerHasProductId, " +
                                                        "d.retailerId as d_retailerId, " +
                                                        "d.productId as d_productId, " +
                                                        "d.SKU as d_SKU, " +
                                                        "d.lastUpdated as d_lastUpdated, " +
                                                        "d.created as d_created," +
                                                        " c.* from retailerhasproduct as d ");
        query.append("left join (select commentId as c_commentId, ownerId as c_ownerId, date as c_date, comment as c_comment, lastUpdated as c_lastUpdated, created as c_created from comment where ownerType = 'RP') as c ");
        query.append("on d.retailerHasProductId = c_ownerId ");
        if (useWhere)
        {
            useWhere = false;
            buildCommonSQL(query);
        }
        query.append(" order by d.retailerId, d.productId, d.retailerHasProductId, c_date");
        return query.toString();
    }   //  buildSQL_MSSQLServer()

    private void buildCommonSQL(StringBuilder query)
    {
        boolean first = true;

		if (useRetailerHasProduct.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(useRetailerHasProduct.getSQLClause("d"));
			first = false;
			useRetailerHasProduct.clear();
		}

		if (useProduct.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(useProduct.getSQLClause("d"));
			first = false;
			useProduct.clear();
		}

		if (useRetailer.isUsed())
		{
			if (first) query.append(" where ");
			else query.append(" and");
			query.append(useRetailer.getSQLClause("d"));
			first = false;
			useRetailer.clear();
		}

	}   //  buildCommonSQL()

	private List<IRetailerHasProduct> processResults_MySQL(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MySQL()");

		List<IRetailerHasProduct> tempList = new ArrayList<>();

		RetailerHasProduct item = null;

		while (rs.next())
		{
			int retailerHasProductId = rs.getInt("d.retailerHasProductId");
			int retailerId = rs.getInt("d.retailerId");
			int productId = rs.getInt("d.productId");
			String SKU = rs.getString("d.SKU");
			LocalDateTime lastUpdated = rs.getTimestamp("d.lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d.created").toLocalDateTime();
LOGGER.debug("retailerHasProductId: {}, retailerId: {}, productId: {}, SKU: {}, lastUpdated: {}, created: {}",
                retailerHasProductId, retailerId, productId, SKU, lastUpdated, created);
			if (item != null && retailerHasProductId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_MySQL(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("commentId"),
					rs.getInt("ownerId"),
					"RP",
					rs.getDate("c.date").toLocalDate(),
					rs.getString("c.comment"),
					rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): extra comment is: {}", comm);
				item = new RetailerHasProduct(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c.commentId");
				if (rs.wasNull())
				{// no comment
					item = new RetailerHasProduct(retailerHasProductId, retailerId, productId, SKU, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						retailerHasProductId,
						"RP",
						rs.getDate("c.date").toLocalDate(),
						rs.getString("c.comment"),
						rs.getTimestamp("c.lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c.created").toLocalDateTime());
LOGGER.debug("processResults_MySQL(): first comment is: {}", comm);
					item = new RetailerHasProduct(retailerHasProductId, retailerId, productId, SKU, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MySQL()

	private List<IRetailerHasProduct> processResults_hsqldb(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_hsqldb()");

		List<IRetailerHasProduct> tempList = new ArrayList<>();

		RetailerHasProduct item = null;

		while (rs.next())
		{
			int retailerHasProductId = rs.getInt("retailerhasproduct.retailerHasProductId");
//LOGGER.debug("retailerHasProductId: {}", retailerHasProductId);
			int retailerId = rs.getInt("retailerhasproduct.retailerId");
//LOGGER.debug("retailerId: {}", retailerId);
			int productId = rs.getInt("retailerhasproduct.productId");
//LOGGER.debug("productId: {}", productId);
			String SKU = rs.getString("retailerhasproduct.SKU");
//LOGGER.debug("SKU: {}", SKU);
			LocalDateTime lastUpdated = rs.getTimestamp("retailerhasproduct.lastUpdated").toLocalDateTime();
//LOGGER.debug("lastUpdated: {}", lastUpdated);
			LocalDateTime created = rs.getTimestamp("retailerhasproduct.created").toLocalDateTime();
//LOGGER.debug("created: {}", created);
LOGGER.debug("retailerHasProductId: {}, retailerId: {}, productId: {}, SKU: {}, lastUpdated: {}, created: {}",
                retailerHasProductId, retailerId, productId, SKU, lastUpdated, created);
			if (item != null && retailerHasProductId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"RP",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new RetailerHasProduct(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new RetailerHasProduct(retailerHasProductId, retailerId, productId, SKU, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						retailerHasProductId,
						"RP",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new RetailerHasProduct(retailerHasProductId, retailerId, productId, SKU, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_hsqldb()

	private List<IRetailerHasProduct> processResults_MSSQLServer(ResultSet rs) throws SQLException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("processResults_MSSQLServer()");

		List<IRetailerHasProduct> tempList = new ArrayList<>();

		RetailerHasProduct item = null;

		while (rs.next())
		{
			int retailerHasProductId = rs.getInt("d_retailerHasProductId");
			int retailerId = rs.getInt("d_retailerId");
			int productId = rs.getInt("d_productId");
			String SKU = rs.getString("d_SKU");
			LocalDateTime lastUpdated = rs.getTimestamp("d_lastUpdated").toLocalDateTime();
			LocalDateTime created = rs.getTimestamp("d_created").toLocalDateTime();
LOGGER.debug("retailerHasProductId: {}, retailerId: {}, productId: {}, SKU: {}, lastUpdated: {}, created: {}",
                retailerHasProductId, retailerId, productId, SKU, lastUpdated, created);
			if (item != null && retailerHasProductId == item.getId())
			{// additional comment on the item
LOGGER.debug("processResults_hsqldb(): got additional comment for: {}", item);
				Comment comm = new Comment(rs.getInt("c_commentId"),
					rs.getInt("c_ownerId"),
					"RP",
					rs.getDate("c_date").toLocalDate(),
					rs.getString("c_comment"),
					rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
					rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): extra comment is: {}", comm);
				item = new RetailerHasProduct(item, comm);
			}
			else
			{
				if (item != null) tempList.add(item);
				int cid = rs.getInt("c_commentId");
				if (rs.wasNull())
				{// no comment
					item = new RetailerHasProduct(retailerHasProductId, retailerId, productId, SKU, lastUpdated, created);
				}
				else
				{// new item with comment
					Comment comm = new Comment(cid,
						retailerHasProductId,
						"RP",
						rs.getDate("c_date").toLocalDate(),
						rs.getString("c_comment"),
						rs.getTimestamp("c_lastUpdated").toLocalDateTime(),
						rs.getTimestamp("c_created").toLocalDateTime());
LOGGER.debug("processResults_hsqldb(): first comment is: {}", comm);
					item = new RetailerHasProduct(retailerHasProductId, retailerId, productId, SKU, lastUpdated, created, comm);
				}
			}
		}
		if (item != null) tempList.add(item);

LOGGER.traceExit(log4jEntryMsg);
		return tempList;
	}	// processResults_MSSQLServer()

	/*
	*	Populate the parents slot(s)
	*/
	private void populateProduct(List<IRetailerHasProduct> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: NL
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populateProduct()");

		if (fetchAll)
		{
			new ProductLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			mapToInt(c -> ((RetailerHasProduct)c).getProductId()).filter(c -> c>0).distinct().
			filter(c -> !MySQLCache.cacheProduct.containsKey(c)).
			toArray();

		if (keys.length > 0)
		{
			new ProductLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	private void populateRetailer(List<IRetailerHasProduct> vals, boolean fetchAll) throws GNDBException
	{
// parent table type: TD
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("populateRetailer()");

		if (fetchAll)
		{
			new RetailerLister().fetch();
			return;
		}
		int[] keys = vals.stream().
			mapToInt(c -> ((RetailerHasProduct)c).getRetailerId()).filter(c -> c>0).distinct().
			filter(c -> !MySQLCache.cacheRetailer.containsKey(c)).
			toArray();

		if (keys.length > 0)
		{
			new RetailerLister().id(keys).fetch();
		}
LOGGER.traceExit(log4jEntryMsg);
	}

	void clear()
	{
		MySQLCache.cacheRetailerHasProduct.clear();
		MySQLCache.completeRetailerHasProduct = false;
	}

	/**
	*
	*Select only the RetailerHasProduct entries with these ids
	*May be called multiple times to extend the list
	*
	*	@param vals	a list of ids
	*	@return	 this Lister
	*/
	RetailerHasProductLister id(int... vals)
	{
		useRetailerHasProduct.id(vals);
		useWhere = useWhere || useRetailerHasProduct.isUsed();
		return this;
	}

	@Override
	public IRetailerHasProductLister retailerHasProduct(IRetailerHasProduct... items)
	{
		useRetailerHasProduct.item(items);
		useWhere = useWhere || useRetailerHasProduct.isUsed();
		return this;
	}

	@Override
	public IRetailerHasProductLister retailerHasProduct(List<IRetailerHasProduct> items)
	{
		useRetailerHasProduct.item(items);
		useWhere = useWhere || useRetailerHasProduct.isUsed();
		return this;
	}

	@Override
	public IRetailerHasProductLister product(IProduct... items)
	{
		useProduct.item(items);
		useWhere = useWhere || useProduct.isUsed();
		return this;
	}

	@Override
	public IRetailerHasProductLister product(List<IProduct> items)
	{
		useProduct.item(items);
		useWhere = useWhere || useProduct.isUsed();
		return this;
	}

	@Override
	public IRetailerHasProductLister retailer(IRetailer... items)
	{
		useRetailer.item(items);
		useWhere = useWhere || useRetailer.isUsed();
		return this;
	}

	@Override
	public IRetailerHasProductLister retailer(List<IRetailer> items)
	{
		useRetailer.item(items);
		useWhere = useWhere || useRetailer.isUsed();
		return this;
	}

	void toJson(JsonBuilderFactory builderFactory, JsonWriterFactory writerFactory, File dumpDirectory) throws GNDBException
	{
		useWhere = false;
		fetch();

		JsonArrayBuilder jsonHc = builderFactory.createArrayBuilder();
		for (IRetailerHasProduct ihc : MySQLCache.cacheRetailerHasProduct.values())
		{
			RetailerHasProduct hc = (RetailerHasProduct)ihc;
			jsonHc.add(hc.toJson(builderFactory));
		}
        
        JsonObjectBuilder job = builderFactory.createObjectBuilder();
        job.add("JsonMode", "DUMP");
        job.add("JsonNBClass", "RetailerHasProduct");
        job.add("values", jsonHc);
        
		try (JsonWriter writer = writerFactory.createWriter(new FileWriter(new File(dumpDirectory, "RetailerHasProduct.json"), false)))
		{
			writer.writeObject(job.build());
		} catch (IOException ex) {
			LOGGER.error("toJson(): IOException", ex);
		}
	}	// toJson

}
