/*
 * Copyright (C) 2018, 2019, 2021, 2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.2.0   Support hsqldb dialect
    2.2.5   Improve JSON load - make it faster!
    2.3.0   Retrieve generated keys properly!
            Date handling change - seems to be a MySQL 8 thing.
    2.4.0   Support MS SQLServer
    3.0.0	Use DBCommentHandler
	3.1.0	Use jakarta implementation of JSON
 */

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import uk.co.gardennotebook.util.StoryLineTree;

import uk.co.gardennotebook.util.SimpleMoney;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;

import java.math.BigDecimal;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Date;
import java.sql.Timestamp;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import jakarta.json.JsonObject;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import org.apache.logging.log4j.Level;

/**
*
*{@inheritDoc}
*
*	@author	Andy Gegg
*	@version	3.1.0
*	@since	1.0
*/
final class PurchaseBuilder implements IPurchaseBuilder
{
	private static final Logger LOGGER = LogManager.getLogger();

	private IPurchase oldInstance = null;

	private final boolean newInstance;

	private boolean changedAncestor = false;

	private int id;
	private int retailerId;
	private boolean changedRetailerId = false;
	private LocalDate date;
	private boolean changedDate = false;
	private BigDecimal totalCost;
	private boolean changedTotalCost = false;

	/*
	*	ISO 4217 standard currency code (GBP, USD, EUR, etc).  Null means the local currency.
	*/
	private String currency;
	private boolean changedCurrency = false;
	private String orderNo;
	private boolean changedOrderNo = false;
	private String invoiceNo;
	private boolean changedInvoiceNo = false;
	private LocalDate deliveryDate;
	private boolean changedDeliveryDate = false;
	private LocalDateTime lastUpdated;
	private LocalDateTime created;
	private boolean somethingChanged = false;

	private boolean changedComments = false;
	private DBCommentHandler commentHandler;	//	compiler will not allow declaration as final

	/**
	*	constructor to use for a new entry
	*/
	PurchaseBuilder()
	{
		this(null);
	}

	/**
	*	constructor to use to edit or delete an existing entry
	*
	*	@param	oldVal	the existing item to modify or delete; if null a new entry will be created
	*/
	PurchaseBuilder(final IPurchase oldVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("constructor(): oldVal={}", oldVal);
		if (oldVal == null || oldVal.getKey() == null || !(oldVal.getKey() instanceof Integer))
		{
			newInstance = true;
			oldInstance = null;
			this.id = -1;
			//	need to set the default value for dates
			this.date = LocalDate.now();
			commentHandler = new DBCommentHandler(NotebookEntryType.PURCHASE);
			return;
		}

		newInstance = false;
		oldInstance = oldVal;

		Purchase baseObj;
		if (oldVal instanceof Purchase)
		{
			baseObj = (Purchase)oldVal;
			this.id = baseObj.getId();
			this.retailerId = baseObj.getRetailerId();
			this.date = baseObj.getDate();
			this.totalCost = baseObj.getTotalCost().orElse(null);
			this.currency = baseObj.getCurrency().orElse(null);
			this.orderNo = baseObj.getOrderNo().orElse(null);
			this.invoiceNo = baseObj.getInvoiceNo().orElse(null);
			this.deliveryDate = baseObj.getDeliveryDate().orElse(null);
			this.lastUpdated = baseObj.getLastUpdated();
			this.created = baseObj.getCreated();
		}
		else
		{
			Object ky = oldVal.getKey();
			if (ky == null) return;
			if (ky instanceof Integer)
				this.id = (Integer)ky;
			ky = oldVal.getRetailer();
			if (ky == null)
			{
				this.retailerId = 0;
			}
			else
			{
				this.retailerId = ((IRetailer)ky).getKey();
			}
			this.date = oldVal.getDate();
			this.totalCost = oldVal.getTotalCost().orElse(null);
			this.currency = oldVal.getCurrency().orElse(null);
			this.orderNo = oldVal.getOrderNo().orElse(null);
			this.invoiceNo = oldVal.getInvoiceNo().orElse(null);
			this.deliveryDate = oldVal.getDeliveryDate().orElse(null);
			this.lastUpdated = oldVal.getLastUpdated();
			this.created = oldVal.getCreated();
		}
		commentHandler = new DBCommentHandler(NotebookEntryType.PURCHASE, this.id);
		LOGGER.traceExit();
	}	//	constructor()

	/**
	*	give the (new) value of retailerId
	*
	*	@param	newVal	the new value
	*	@return	this Builder
	*/
	IPurchaseBuilder retailerId(final int newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("retailerId(): oldVal={}, newVal={}", this.retailerId, newVal);
		if (this.retailerId == newVal) return this;
		this.retailerId = newVal;
		changedRetailerId = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}
	@Override
	public IPurchaseBuilder retailer(final IRetailer newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("retailer(): oldVal={}, newVal={}", this.retailerId, newVal);
		if (newVal == null) return this;
		if (this.retailerId == newVal.getKey()) return this;
		this.retailerId = newVal.getKey();
		changedRetailerId = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPurchaseBuilder date(final LocalDate newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("date(): oldVal={}, newVal={}", this.date, newVal);

		if (newVal == null) return this;
		if (newVal.equals(this.date)) return this;
		this.date = newVal;
		changedDate = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPurchaseBuilder totalCost(final BigDecimal newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("totalCost(): oldVal={}, newVal={}", this.totalCost, newVal);

		if (newVal == null && this.totalCost == null) return this;
		if (newVal != null && this.totalCost != null && newVal.equals(this.totalCost)) return this;
		this.totalCost = newVal;
		changedTotalCost = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPurchaseBuilder currency(final String newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("currency(): oldVal={}, newVal={}", this.currency, newVal);

		if (newVal == null && this.currency == null) return this;
		if (newVal != null && this.currency != null && newVal.equals(this.currency)) return this;
		this.currency = newVal;
		changedCurrency = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPurchaseBuilder orderNo(final String newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("orderNo(): oldVal={}, newVal={}", this.orderNo, newVal);

		if (newVal == null && this.orderNo == null) return this;
		if (newVal != null && this.orderNo != null && newVal.equals(this.orderNo)) return this;
		this.orderNo = newVal;
		changedOrderNo = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPurchaseBuilder invoiceNo(final String newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("invoiceNo(): oldVal={}, newVal={}", this.invoiceNo, newVal);

		if (newVal == null && this.invoiceNo == null) return this;
		if (newVal != null && this.invoiceNo != null && newVal.equals(this.invoiceNo)) return this;
		this.invoiceNo = newVal;
		changedInvoiceNo = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPurchaseBuilder deliveryDate(final LocalDate newVal)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deliveryDate(): oldVal={}, newVal={}", this.deliveryDate, newVal);

		if (newVal == null && this.deliveryDate == null) return this;
		if (newVal != null && this.deliveryDate != null && newVal.equals(this.deliveryDate)) return this;
		this.deliveryDate = newVal;
		changedDeliveryDate = true;
		somethingChanged = true;
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPurchaseBuilder totalPrice(final SimpleMoney newVal)

	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("totalPrice({})", newVal);

		if (newVal == null)
		{
			currency(null);
			totalCost(null);
		}
		else
		{
			currency(newVal.currency().getCurrencyCode());
			totalCost(newVal.amount());
		}
		LOGGER.traceExit();
		return this;
	}

	@Override
	public IPurchaseBuilder addComment(final String... newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment[array]()");

		commentHandler.addComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit("addComment");
		return this;
	}

	@Override
	public IPurchaseBuilder addComment(final List<String> newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment<list>()");

		commentHandler.addComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit("addComment");
		return this;
	}

	/**
	*	remove a comment from this item
	*
	*	@param	newVals	the comment to remove.  If the comment does not exist, this is a null-op
	*	@return	 this Builder
	*/
	IPurchaseBuilder deleteComment(int... newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

		commentHandler.deleteComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPurchaseBuilder deleteComment(final IComment... newVals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

		commentHandler.deleteComment(newVals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPurchaseBuilder deleteComment(final List<IComment> vals)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

		commentHandler.deleteComment(vals);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPurchaseBuilder changeComment(final IComment base, final String comment)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

		commentHandler.changeComment(base, comment);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPurchaseBuilder changeComment(final IComment base, final LocalDate date)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

		commentHandler.changeComment(base, date);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPurchaseBuilder changeComment(final IComment base, final LocalDate date, final String comment)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

		commentHandler.changeComment(base, date, comment);
		changedComments = commentHandler.isChangedComments();
		LOGGER.traceExit(log4jEntryMsg);
		return this;
	}

	@Override
	public IPurchase save() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("save(): somethingChanged: {}, newInstance: {}, changedComments: {}, changedAncestor: {}",
                                                                somethingChanged, newInstance, changedComments, changedAncestor);

		if (!somethingChanged && !newInstance && !changedComments && !changedAncestor)
		{
			LOGGER.traceExit("nothing changed");
			return MySQLCache.cachePurchase.get(this.id);
		}

		if (newInstance)
		{
			doInsert();
		}
		else if (somethingChanged)
		{
			doUpdate();
		}

		if (changedComments)
		{
			commentHandler.setParentId(this.id);
			commentHandler.save();
		}

// mark cache as dirty
		if (!newInstance &&(somethingChanged || changedComments))
		{
			MySQLCache.cachePurchase.remove(this.id);
		}
// populate the cache
		new PurchaseLister().id(this.id).fetch();
		IPurchase newValue = MySQLCache.cachePurchase.get(this.id);
		if (oldInstance != null)
		{
			oldInstance.flagReplaced(newValue);
		}

	//	tell any parent beans the child list has mutated
	//	only additions and deletions matter, other changes will be reflected through the child bean
		if (newInstance)
			{
			MySQLCache.cacheRetailer.get(retailerId).flagChildAdded(newValue);
		}
		else
		{	//	updated
			if (changedRetailerId)
			{
				if (oldInstance != null)
				{
					MySQLCache.cacheRetailer.get(oldInstance.getRetailer().getKey()).flagChildDeleted(oldInstance);
				}
				MySQLCache.cacheRetailer.get(newValue.getRetailer().getKey()).flagChildAdded(newValue);
			}
		}

		//	stop multiple saves!
		oldInstance = null;

		somethingChanged = false;
		changedComments = false;
		changedAncestor = false;
		changedRetailerId = false;
		changedDate = false;
		changedTotalCost = false;
		changedCurrency = false;
		changedOrderNo = false;
		changedInvoiceNo = false;
		changedDeliveryDate = false;

		LOGGER.traceExit(log4jEntryMsg);
		return newValue;
	}	//	save()

	@Override
	public boolean needSave()
	{
		return somethingChanged || changedComments || changedAncestor;
	}	// needSave()

	@Override
	public boolean canSave()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("canSave(): newInstance={}", newInstance);

		if (!needSave())
		{//	save() will be a null-op but that's OK
			return true;
		}
		if (this.retailerId <= 0)
		{
			LOGGER.debug("retailerId not set");
			return false;
		}
		if (this.date == null ||
			this.date == LocalDate.MAX ||
			this.date == LocalDate.MIN)
		{
			LOGGER.debug("date not set");
			return false;
		}
		return true;
	}	// canSave()

	@Override
	public boolean canDelete() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("canDelete(): newInstance={}", newInstance);

		if (newInstance) return LOGGER.traceExit(log4jEntryMsg, false);

		String query;
		boolean  readValue = false;
		try (	Connection conn = DBConnection.getConnection();
				Statement stmt = conn.createStatement();	)
		{
			switch (DBConnection.DB_IN_USE)
			{
				case MariaDB, MySQL -> query = "select exists (select 1 from purchaseitem where purchaseId = " + this.id + ") as fred";
				case hsqldb -> query = "select exists (select 1 from purchaseitem where purchaseId = " + this.id + ") as fred from (values(99))";
				case MSSQLServer -> query = "select CASE WHEN EXISTS (select 1 from purchaseitem where purchaseId = " + this.id + ") THEN 1 ELSE 0 END as fred";
				default -> {
					LOGGER.error("canDelete(): no known rdbms");
					throw new GNDBException(new IllegalStateException("PurchaseBuilder: canDelete(): no known RDBMS"));
				}
			}
LOGGER.debug("canDelete(): query: {}", query);
			ResultSet rs = stmt.executeQuery(query);
            rs.next();
            readValue = rs.getBoolean("fred");
            rs.close();
LOGGER.debug("canDelete(): readValue: {}", readValue);
			if (readValue)
			{
				return LOGGER.traceExit(log4jEntryMsg, false);
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("canDelete(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
		return LOGGER.traceExit(log4jEntryMsg, !readValue);
	}	// canDelete()

	@Override
	public void delete() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("delete(): newInstance={}", newInstance);

		if (newInstance) return;
		if (!canDelete()) return;

		int res = 0;
		String query = "delete from purchase where purchaseId = " + this.id;
LOGGER.debug("delete(): query: {}", query);
		try (	Connection conn = DBConnection.getConnection();
				Statement stmt = conn.createStatement();	)
		{
			res = stmt.executeUpdate(query);
LOGGER.debug("delete(): result: {}", res);
			// tidy up dependencies with nullable keys
			if (res > 0) {
				query = "delete from comment where ownerId = " + this.id + " and ownerType = 'PU'";
				int res2 = stmt.executeUpdate(query);
LOGGER.debug("delete() comments: result: {}", res2);
			}
			stmt.close();
		}catch (SQLException ex) {
			LOGGER.error("delete(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
		if (res > 0)
		{
			oldInstance.flagDeleted();
			MySQLCache.cachePurchase.remove(this.id);
	//	tell any parent beans the child list has mutated
	//	only additions and deletions matter, other changes will be reflected through the child bean
			MySQLCache.cacheRetailer.get(oldInstance.getRetailer().getKey()).flagChildDeleted(oldInstance);
		}
		oldInstance = null;
LOGGER.traceExit(log4jEntryMsg);
	}	// delete()

	private void doUpdate() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("doUpdate(): newInstance={}, somethingChanged={}", newInstance, somethingChanged);

		if (newInstance) return;
		if (!somethingChanged) return;
		StringBuilder query = new StringBuilder("update purchase set ");
		if (changedRetailerId)
		{
			query.append("retailerId = ?, ");
		}

		if (changedDate)
		{
			query.append("date = ?, ");
		}

		if (changedTotalCost)
		{
			query.append("totalCost = ?, ");
		}

		if (changedCurrency)
		{
			query.append("currency = ?, ");
		}

		if (changedOrderNo)
		{
			query.append("orderNo = ?, ");
		}

		if (changedInvoiceNo)
		{
			query.append("invoiceNo = ?, ");
		}

		if (changedDeliveryDate)
		{
			query.append("deliveryDate = ?, ");
		}

		query.delete(query.length()-2, query.length());
		query.append(" where purchaseId = ").append(this.id);
LOGGER.debug("doUpdate(): query={} ", query.toString());
		try (	Connection conn = DBConnection.getConnection();
				PreparedStatement stmt = conn.prepareStatement(query.toString());	)
		{
			int paramIx = 1;
			if (changedRetailerId)
			{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.retailerId);
				stmt.setInt(paramIx++, this.retailerId);
			}

			if (changedDate)
			{
LOGGER.debug("doUpdate(): param {}={}", paramIx, Date.valueOf(this.date));
				stmt.setDate(paramIx++, Date.valueOf(this.date), java.util.Calendar.getInstance()); //  2.3.0
			}

			if (changedTotalCost)
			{
				if (this.totalCost == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.NUMERIC);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.totalCost);
					stmt.setBigDecimal(paramIx++, this.totalCost);
				}
			}

			if (changedCurrency)
			{
				if (this.currency == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.currency);
					stmt.setString(paramIx++, this.currency);
				}
			}

			if (changedOrderNo)
			{
				if (this.orderNo == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.orderNo);
					stmt.setString(paramIx++, this.orderNo);
				}
			}

			if (changedInvoiceNo)
			{
				if (this.invoiceNo == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, this.invoiceNo);
					stmt.setString(paramIx++, this.invoiceNo);
				}
			}

			if (changedDeliveryDate)
			{
				if (this.deliveryDate == null)
				{
LOGGER.debug("doUpdate(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.DATE);
				}
				else
				{
LOGGER.debug("doUpdate(): param {}={}", paramIx, Date.valueOf(this.deliveryDate));
					stmt.setDate(paramIx++, Date.valueOf(this.deliveryDate), java.util.Calendar.getInstance()); //  2.3.0
				}
			}

			stmt.executeUpdate();

		}catch (SQLException ex) {
			LOGGER.error("doUpdate(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
LOGGER.traceExit(log4jEntryMsg);
	}	// doUpdate

	private void doInsert() throws GNDBException
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("doInsert(): newInstance={}", newInstance);

		if (!newInstance) return;
		if (!canSave())
		{
			throw LOGGER.throwing(Level.ERROR, new IllegalStateException("doInsert(): save criteria not met"));
		}
		if (!this.changedRetailerId)
		{
			throw LOGGER.throwing(Level.ERROR, new IllegalStateException("PurchaseBuilder: doInsert(): retailerId unspecified"));
		}

		StringBuilder query = new StringBuilder("insert into purchase (");
		query.append("retailerId, ");
		query.append("date, ");
		if (changedTotalCost)
		{
			query.append("totalCost, ");
		}

		if (changedCurrency)
		{
			query.append("currency, ");
		}

		if (changedOrderNo)
		{
			query.append("orderNo, ");
		}

		if (changedInvoiceNo)
		{
			query.append("invoiceNo, ");
		}

		if (changedDeliveryDate)
		{
			query.append("deliveryDate, ");
		}

		query.replace(query.length()-2, query.length(), ") values (");
		query.append("?, ");
		query.append("?, ");
		if (changedTotalCost)
		{
			query.append("?, ");
		}

		if (changedCurrency)
		{
			query.append("?, ");
		}

		if (changedOrderNo)
		{
			query.append("?, ");
		}

		if (changedInvoiceNo)
		{
			query.append("?, ");
		}

		if (changedDeliveryDate)
		{
			query.append("?, ");
		}

		query.replace(query.length()-2, query.length(), ")");
LOGGER.debug("doInsert(): query={}", query.toString());

		try (	Connection conn = DBConnection.getConnection();
				PreparedStatement stmt = conn.prepareStatement(query.toString(), Statement.RETURN_GENERATED_KEYS); )
		{
			int paramIx = 1;
LOGGER.debug("doInsert(): param {}={}", paramIx, this.retailerId);
			stmt.setInt(paramIx++, this.retailerId);
LOGGER.debug("doInsert(): param {}={}", paramIx, Date.valueOf(this.date));
			stmt.setDate(paramIx++, Date.valueOf(this.date), java.util.Calendar.getInstance()); //  2.3.0
			if (changedTotalCost) {
				if (this.totalCost == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.NUMERIC);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.totalCost);
					stmt.setBigDecimal(paramIx++, this.totalCost);
				}
			}

			if (changedCurrency) {
				if (this.currency == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.currency);
					stmt.setString(paramIx++, this.currency);
				}
			}

			if (changedOrderNo) {
				if (this.orderNo == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.orderNo);
					stmt.setString(paramIx++, this.orderNo);
				}
			}

			if (changedInvoiceNo) {
				if (this.invoiceNo == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, this.invoiceNo);
					stmt.setString(paramIx++, this.invoiceNo);
				}
			}

			if (changedDeliveryDate) {
				if (this.deliveryDate == null)
				{
LOGGER.debug("doInsert(): param {} null", paramIx);
					stmt.setNull(paramIx++, java.sql.Types.DATE);
				}
				else
				{
LOGGER.debug("doInsert(): param {}={}", paramIx, Date.valueOf(this.deliveryDate));
					stmt.setDate(paramIx++, Date.valueOf(this.deliveryDate), java.util.Calendar.getInstance()); //  2.3.0
				}
			}

			stmt.executeUpdate();

			ResultSet rs = stmt.getGeneratedKeys();
			rs.next();
			int newId = rs.getInt(1);
LOGGER.debug("doInsert(): newId: {}", newId);
			this.id = newId;
		}catch (SQLException ex) {
			LOGGER.error("doInsert(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}

LOGGER.traceExit(log4jEntryMsg);
	}	// doInsert

	/**
     * Process the whole JSON array from a DUMP
     * 
     *  @param newVal    a list of JSON objects representing PlantSpecies as output by a JSON DUMP
     * 
     *	@throws	GNDBException	If the underlying MySQL database throws SQLException it is translated to this.
     *				The causal SQLException can be retrieved by <code>getCause()</code>
     * 
     * @since 2.2.5
     */
    void restoreJsonDump(List<JsonObject> newVal) throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("restoreJsonDump(list JSON)");
        
        if (newVal.isEmpty())
            return;

		StringBuilder query = new StringBuilder("insert into purchase (");
        query.append("purchaseId, ");
		query.append("retailerId, ");
		query.append("date, ");
		query.append("totalCost, ");
		query.append("currency, ");
		query.append("orderNo, ");
		query.append("invoiceNo, ");
		query.append("deliveryDate, ");
		query.append("lastUpdated, ");
		query.append("created) ");
        if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.hsqldb)
        {
            query.append(" overriding system value ");
        }
        query.append(" values (");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?, ");
		query.append("?) ");
LOGGER.debug("restoreJsonDump(): query={}", query.toString());

		try (	Connection conn = DBConnection.getConnection();
				PreparedStatement stmt = conn.prepareStatement(query.toString());	)
		{
            conn.setAutoCommit(false);
            int txCount = 0;
            if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
            {
                conn.createStatement().execute("SET IDENTITY_INSERT purchase ON");
            }
            
            for (JsonObject jo : newVal)
            {
                if (!"DUMP".equals(jo.getString("JsonMode", "DUMP")))
                {
                    LOGGER.error("Purchase DUMP object is not DUMP");
                    throw new IllegalArgumentException("Purchase DUMP object is not DUMP");
                }
                if (!"Purchase".equals(jo.getString("JsonNBClass", "Purchase")))
                {
                    LOGGER.error("Purchase DUMP object is not Purchase");
                    throw new IllegalArgumentException("Purchase DUMP object is not Purchase");
                }
                Purchase ps = new Purchase(jo);
                if (ps.getId() <= 0)
                {//this forces the value of the id field.  The >0 test is a bodge.
                    LOGGER.error("Purchase DUMP object does not have an id");
                    throw new IllegalArgumentException("Purchase DUMP object does not have an id");
                }
                
                int paramIx = 1;
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getId());
                    stmt.setInt(paramIx++, ps.getId());

LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getRetailerId());
                stmt.setInt(paramIx++, ps.getRetailerId());
                
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getDate());
                stmt.setDate(paramIx++, Date.valueOf(ps.getDate()), java.util.Calendar.getInstance()); //  2.3.0
                
                if (!ps.getTotalCost().isPresent())
                {
LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.NUMERIC);
                }
                else
                {
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getTotalCost().get());
                    stmt.setBigDecimal(paramIx++, ps.getTotalCost().get());
                }

                if (!ps.getCurrency().isPresent())
                {
LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getCurrency().get());
                    stmt.setString(paramIx++, ps.getCurrency().get());
                }

                if (!ps.getOrderNo().isPresent())
                {
LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getOrderNo().get());
                    stmt.setString(paramIx++, ps.getOrderNo().get());
                }

                if (!ps.getInvoiceNo().isPresent())
                {
LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getInvoiceNo().get());
                    stmt.setString(paramIx++, ps.getInvoiceNo().get());
                }

                if (!ps.getDeliveryDate().isPresent())
                {
LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.DATE);
                }
                else
                {
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getDeliveryDate().get());
                    stmt.setDate(paramIx++, Date.valueOf(ps.getDeliveryDate().get()), java.util.Calendar.getInstance()); //  2.3.0
                }

LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getLastUpdated());
                stmt.setTimestamp(paramIx++, Timestamp.valueOf(ps.getLastUpdated()));
LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getCreated());
                stmt.setTimestamp(paramIx++, Timestamp.valueOf(ps.getCreated()));
                stmt.executeUpdate();

                if (!ps.getComments().isEmpty())
                {
                    if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
                    {
                        conn.createStatement().execute("SET IDENTITY_INSERT purchase OFF");
                    }
                    CommentBuilder cb = new CommentBuilder(NotebookEntryType.PURCHASE, ps.getId());
                    cb.doJsonInsert(ps.getComments(), conn);
                    if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
                    {
                        conn.createStatement().execute("SET IDENTITY_INSERT purchase ON");
                    }
                }

                if (++txCount > 50)
                {
                    conn.commit();
                    txCount = 0;
                }
            }

            conn.commit();
            if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
            {
                conn.createStatement().execute("SET IDENTITY_INSERT purchase OFF");
            }
		}catch (SQLException ex) {
			LOGGER.error("restoreJsonDump(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
			throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
		}
	}	// restoreJsonDump(JsonObjects)

	@Override
	public boolean hasAncestor() throws GNDBException
	{
		//	this class does not participate in story lines
		return false;
	}	//	hasAncestor()

	@Override
	public StoryLineTree<? extends INotebookEntry> getAncestors() throws GNDBException
	{
		//	this class does not participate in story lines
		return StoryLineTree.emptyTree();
	}	//	getAncestors()

	@Override
	public IPurchaseBuilder ancestor(IPurchaseItem ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(PurchaseItem)

	@Override
	public IPurchaseBuilder ancestor(IGroundwork ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(Groundwork)

	@Override
	public IPurchaseBuilder ancestor(IAfflictionEvent ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(AfflictionEvent)

	@Override
	public IPurchaseBuilder ancestor(IHusbandry ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(Husbandry)

	@Override
	public IPurchaseBuilder ancestor(ISaleItem ancestor) throws GNDBException
	{
		return this;
	}	//	ancestor(SaleItem)

	@Override
	public boolean hasDescendant() throws GNDBException
	{
		//	this class does not participate in story lines
		return false;
	}	//	hasDescendant()

	@Override
	public StoryLineTree<? extends INotebookEntry> getDescendants() throws GNDBException
	{
		//	this class does not participate in story lines
		return StoryLineTree.emptyTree();
	}	//	getDescendants()

	@Override
	public boolean addDescendant(IPurchaseItem descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(PurchaseItem)

	@Override
	public boolean addDescendant(IGroundwork descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(Groundwork)

	@Override
	public boolean addDescendant(IAfflictionEvent descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(AfflictionEvent)

	@Override
	public boolean addDescendant(IHusbandry descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(Husbandry)

	@Override
	public boolean addDescendant(ISaleItem descendant) throws GNDBException
	{
		return false;
	}	//	addDescendant(SaleItem)

}

