/*
 *
 *  Copyright (C) 2021, 2023 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

/*
    Change log
	3.1.0	Use jakarta implementation of JSON
*/

package uk.co.gardennotebook.mysql;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.spi.*;

import jakarta.json.JsonObject;
import java.sql.*;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;

/**
 *
 *{@inheritDoc}
 *
 *	@author	Andy Gegg
 *	@version	3.1.0
 *	@since	3.0.0
 */
final class JournalBuilder implements IJournalBuilder
{
    private static final Logger LOGGER = LogManager.getLogger();

    private IJournal oldInstance = null;

    private final boolean newInstance;

    private int id;

    private LocalDate date;
    private boolean changedDate = false;

    private String title;
    private boolean changedTitle = false;

    private String description;
    private boolean changedDescription = false;

    private LocalDateTime lastUpdated;
    private LocalDateTime created;
    private boolean somethingChanged = false;

    private boolean changedComments = false;

    private DBCommentHandler commentHandler;	//	compiler will not allow declaration as final

    /**
     *	constructor to use for a new entry
     */
    JournalBuilder()
    {
        this(null);
    }

    /**
     *	constructor to use to edit or delete an existing entry
     *
     *	@param	oldVal	the existing item to modify or delete; if null a new entry will be created
     */
    JournalBuilder(final IJournal oldVal)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("constructor(): oldVal={}", oldVal);

        if (oldVal == null || oldVal.getKey() == null || !(oldVal.getKey() instanceof Integer))
        {
            newInstance = true;
            oldInstance = null;
            this.id = -1;
            //	need to set the default value for dates
            this.date = LocalDate.now();
            commentHandler = new DBCommentHandler(NotebookEntryType.JOURNAL);
            return;
        }

        newInstance = false;
        oldInstance = oldVal;

//        Journal baseObj;
        if (oldVal instanceof Journal baseObj)
        {
//            baseObj = (Journal)oldVal;
            this.id = baseObj.getId();
            this.date = baseObj.getDate();
            this.title = baseObj.getTitle();
            this.description = baseObj.getDescription().orElse(null);
            this.lastUpdated = baseObj.getLastUpdated();
            this.created = baseObj.getCreated();
        }
        else
        {
            Integer ky = oldVal.getKey();
            if (ky == null) return;
            this.id = ky;
            this.date = oldVal.getDate();
            this.title = oldVal.getTitle();
            this.description = oldVal.getDescription().orElse(null);
            this.lastUpdated = oldVal.getLastUpdated();
            this.created = oldVal.getCreated();
        }
        commentHandler = new DBCommentHandler(NotebookEntryType.JOURNAL, this.id);

        LOGGER.traceExit();
    }	//	constructor()

    @Override
    public IJournalBuilder date(LocalDate newVal)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("date(): oldVal={}, newVal={}", this.date, newVal);

        if (newVal == null) return this;
        if (newVal.equals(this.date)) return this;
        this.date = newVal;
        changedDate = true;
        somethingChanged = true;
        LOGGER.traceExit();
        return this;
    }

    @Override
    public IJournalBuilder title(String newVal)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("title(): oldVal={}, newVal={}", this.title, newVal);

        if (newVal == null) return this;
        if (newVal.equals(this.title)) return this;

        this.title = newVal;
        changedTitle = true;
        somethingChanged = true;
        LOGGER.traceExit();
        return this;
    }

    @Override
    public IJournalBuilder description(String newVal)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("description(): oldVal={}, newVal={}", this.description, newVal);

        if (newVal == null && this.description == null) return this;
        if (newVal != null && newVal.equals(this.description)) return this;

        this.description = newVal;
        changedDescription = true;
        somethingChanged = true;
        LOGGER.traceExit();
        return this;
    }

    @Override
    public IJournalBuilder addComment(String... newVals)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment[array]()");

        commentHandler.addComment(newVals);
        changedComments = commentHandler.isChangedComments();
        LOGGER.traceExit();
        return this;
    }

    @Override
    public IJournalBuilder addComment(List<String> newVals)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("addComment[array]()");

        commentHandler.addComment(newVals);
        changedComments = commentHandler.isChangedComments();
        LOGGER.traceExit();
        return this;
    }

    /**
     *	remove a comment from this item
     *
     *	@param	newVals	the comment to remove.  If the comment does not exist, this is a null-op
     *	@return	 this Builder
     */
    IJournalBuilder deleteComment(int... newVals)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

        commentHandler.deleteComment(newVals);
        changedComments = commentHandler.isChangedComments();
        LOGGER.traceExit(log4jEntryMsg);
        return this;
    }

    @Override
    public IJournalBuilder deleteComment(IComment... newVals)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

        commentHandler.deleteComment(newVals);
        changedComments = commentHandler.isChangedComments();
        LOGGER.traceExit(log4jEntryMsg);
        return this;
    }

    @Override
    public IJournalBuilder deleteComment(List<IComment> newVals)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("deleteComment()");

        commentHandler.deleteComment(newVals);
        changedComments = commentHandler.isChangedComments();
        LOGGER.traceExit(log4jEntryMsg);
        return this;
    }

    @Override
    public IJournalBuilder changeComment(IComment base, String comment)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

        commentHandler.changeComment(base, comment);
        changedComments = commentHandler.isChangedComments();
        LOGGER.traceExit(log4jEntryMsg);
        return this;
    }

    @Override
    public IJournalBuilder changeComment(IComment base, LocalDate date)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

        commentHandler.changeComment(base, date);
        changedComments = commentHandler.isChangedComments();
        LOGGER.traceExit(log4jEntryMsg);
        return this;
    }

    @Override
    public IJournalBuilder changeComment(IComment base, LocalDate date, String comment)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("changeComment()");

        commentHandler.changeComment(base, date, comment);
        changedComments = commentHandler.isChangedComments();
        LOGGER.traceExit(log4jEntryMsg);
        return this;
    }

    @Override
    public boolean needSave()
    {
        LOGGER.debug("needSave: somethingChanged : {}, changedComments: {}", somethingChanged, changedComments);
        return somethingChanged || changedComments;
    }

    @Override
    public boolean canSave()
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("canSave(): newInstance={}", newInstance);

        if (!needSave())
        {//	save() will be a null-op but that's OK
            return true;
        }

        if (this.date == null ||
            this.date == LocalDate.MAX ||
            this.date == LocalDate.MIN)
        {
            LOGGER.debug("date not set");
            return false;
        }

        if (this.title == null || this.title.isBlank())
        {
            return false;
        }

        return true;
    }	// canSave()

    @Override
    public IJournal save() throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("save(): somethingChanged: {}, newInstance: {}, changedComments: {}",
                somethingChanged, newInstance, changedComments);

        if (!somethingChanged && !newInstance && !changedComments)
        {
            LOGGER.traceExit("nothing changed");
            return MySQLCache.cacheJournal.get(this.id);
        }

        if (newInstance)
        {
            doInsert();
        }
        else if (somethingChanged)
        {
            doUpdate();
        }

        if (changedComments)
        {
            commentHandler.setParentId(this.id);
            commentHandler.save();
        }

// mark cache as dirty
        if (!newInstance &&(somethingChanged || changedComments))
        {
            MySQLCache.cacheJournal.remove(this.id);
        }
// populate the cache
        new JournalLister().id(this.id).fetch();
        IJournal newValue = MySQLCache.cacheJournal.get(this.id);
        if (oldInstance != null)
        {
            oldInstance.flagReplaced(newValue);
        }

        //	stop multiple saves!
        oldInstance = null;

        somethingChanged = false;
        changedComments = false;
        changedDate = false;
        changedTitle = false;
        changedDescription = false;

        LOGGER.traceExit(log4jEntryMsg);
        return newValue;
    }

    @Override
    public boolean canDelete() throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("canDelete(): newInstance={}", newInstance);

        return LOGGER.traceExit(log4jEntryMsg, !newInstance);
    }

    @Override
    public void delete() throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("delete(): newInstance={}", newInstance);

        if (newInstance) return;
        if (!canDelete()) return;

        int res = 0;
        String query = "delete from journal where journalId = " + this.id;
        LOGGER.debug("delete(): query: {}", query);
        try (Connection conn = DBConnection.getConnection();
             Statement stmt = conn.createStatement();	)
        {
            res = stmt.executeUpdate(query);
            LOGGER.debug("delete(): result: {}", res);
            // tidy up dependencies with nullable keys
            if (res > 0) {
                query = "delete from comment where ownerId = " + this.id + " and ownerType = 'JN'";
                int res2 = stmt.executeUpdate(query);
                LOGGER.debug("delete() comments: result: {}", res2);
            }
            stmt.close();
        }catch (SQLException ex) {
            LOGGER.error("delete(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
            throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
        }
        if (res > 0)
        {
            oldInstance.flagDeleted();
            MySQLCache.cacheJournal.remove(this.id);
         }
        oldInstance = null;
        LOGGER.traceExit(log4jEntryMsg);
    }

    private void doUpdate() throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("doUpdate(): newInstance={}, somethingChanged={}", newInstance, somethingChanged);

        if (newInstance) return;
        if (!somethingChanged) return;

        StringBuilder query = new StringBuilder("update journal set ");

        if (changedDate)
        {
            query.append("date = ?, ");
        }

        if (changedTitle)
        {
            query.append("title = ?, ");
        }

        if (changedDescription)
        {
            query.append("description = ?, ");
        }

        query.delete(query.length()-2, query.length());
        query.append(" where journalId = ").append(this.id);
        LOGGER.debug("doUpdate(): query={} ", query.toString());

        try (Connection conn = DBConnection.getConnection();
             PreparedStatement stmt = conn.prepareStatement(query.toString());	)
        {
            int paramIx = 1;

            if (changedDate)
            {
                LOGGER.debug("doUpdate(): param {}={}", paramIx, Date.valueOf(this.date));
                stmt.setDate(paramIx++, Date.valueOf(this.date), java.util.Calendar.getInstance()); //  2.3.0
            }

            if (changedTitle)
            {
                    LOGGER.debug("doUpdate(): param {}={}", paramIx, this.title);
                    stmt.setString(paramIx++, this.title);
            }

            if (changedDescription)
            {
                if (this.description == null)
                {
                    LOGGER.debug("doUpdate(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
                    LOGGER.debug("doUpdate(): param {}={}", paramIx, this.description);
                    stmt.setString(paramIx++, this.description);
                }
            }

            stmt.executeUpdate();

        }
        catch (SQLException ex) {
            LOGGER.error("doUpdate(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
            throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
        }
        LOGGER.traceExit(log4jEntryMsg);
    }

    private void doInsert() throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("doInsert(): newInstance={}", newInstance);

        if (!newInstance) return;
        if (!canSave())
        {
            throw LOGGER.throwing(Level.ERROR, new IllegalStateException("doInsert(): save criteria not met"));
        }

        StringBuilder query = new StringBuilder("insert into journal (");
        query.append("date, ");
        query.append("title, ");

        if (changedDescription)
        {
            query.append("description, ");
        }

        query.replace(query.length()-2, query.length(), ") values (");
        query.append("?, ?, "); //  date, title

        if (changedDescription)
        {
            query.append("?, ");
        }

        query.replace(query.length()-2, query.length(), ")");
        LOGGER.debug("doInsert(): query={}", query.toString());

        try (	Connection conn = DBConnection.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(query.toString(), Statement.RETURN_GENERATED_KEYS); )
        {
            int paramIx = 1;

            LOGGER.debug("doInsert(): param {}={}", paramIx, Date.valueOf(this.date));
            stmt.setDate(paramIx++, Date.valueOf(this.date), java.util.Calendar.getInstance()); //  2.3.0

            LOGGER.debug("doInsert(): param {}={}", paramIx, this.title);
            stmt.setString(paramIx++, this.title);

            if (changedDescription) {
                if (this.description == null)
                {
                    LOGGER.debug("doInsert(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
                    LOGGER.debug("doInsert(): param {}={}", paramIx, this.description);
                    stmt.setString(paramIx++, this.description);
                }
            }

            stmt.executeUpdate();

            ResultSet rs = stmt.getGeneratedKeys();
            rs.next();
            int newId = rs.getInt(1);
            LOGGER.debug("doInsert(): newId: {}", newId);
            this.id = newId;

        }
        catch (SQLException ex) {
            LOGGER.error("doInsert(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
            throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
        }

        LOGGER.traceExit(log4jEntryMsg);
    }   //  doInsert()

    /**
     * Process the whole JSON array from a DUMP
     *
     *  @param newVal    a list of JSON objects representing Journal entries as output by a JSON DUMP
     *  @throws	GNDBException	If the underlying MySQL database throws SQLException it is translated to this.
     *				The causal SQLException can be retrieved by <code>getCause()</code>
     */
    void restoreJsonDump(List<JsonObject> newVal) throws GNDBException
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("restoreJsonDump(list JSON)");

        if (newVal.isEmpty())
            return;

        String query = "insert into journal (";
        query += "journalId, ";
        query += "date, ";
        query += "title, ";
        query += "description, ";
        query += "lastUpdated, ";
        query += "created) ";
        if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.hsqldb)
        {
            query += " overriding system value ";
        }
        query += " values (?, ?, ?, ?, ?, ?)";
        LOGGER.debug("restoreJsonDump(): query={}", query.toString());

        try (	Connection conn = DBConnection.getConnection();
                 PreparedStatement stmt = conn.prepareStatement(query.toString());	)
        {
            conn.setAutoCommit(false);
            int txCount = 0;
            if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
            {
                conn.createStatement().execute("SET IDENTITY_INSERT journal ON");
            }

            for (JsonObject jo : newVal)
            {
                if (!"DUMP".equals(jo.getString("JsonMode", "DUMP")))
                {
                    LOGGER.error("Journal DUMP object is not DUMP");
                    throw new IllegalArgumentException("Journal DUMP object is not DUMP");
                }
                if (!"Journal".equals(jo.getString("JsonNBClass", "Journal")))
                {
                    LOGGER.error("Journal DUMP object is not Journal");
                    throw new IllegalArgumentException("Journal DUMP object is not Journal");
                }
                Journal ps = new Journal(jo);
                if (ps.getId() <= 0)
                {//this forces the value of the id field.  The >0 test is a bodge.
                    LOGGER.error("Journal DUMP object does not have an id");
                    throw new IllegalArgumentException("Journal DUMP object does not have an id");
                }
                int paramIx = 1;
                LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getId());
                stmt.setInt(paramIx++, ps.getId());

                LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getDate());
                stmt.setDate(paramIx++, Date.valueOf(ps.getDate()), java.util.Calendar.getInstance()); //  2.3.0

                LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getTitle());
                stmt.setString(paramIx++, ps.getTitle());

                if (ps.getDescription().isEmpty())
                {
                    LOGGER.debug("restoreJsonDump(): param {} null", paramIx);
                    stmt.setNull(paramIx++, java.sql.Types.VARCHAR);
                }
                else
                {
                    LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getDescription().get());
                    stmt.setString(paramIx++, ps.getDescription().get());
                }

                LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getLastUpdated());
                stmt.setTimestamp(paramIx++, Timestamp.valueOf(ps.getLastUpdated()));
                LOGGER.debug("restoreJsonDump(): param {}={}", paramIx, ps.getCreated());
                stmt.setTimestamp(paramIx++, Timestamp.valueOf(ps.getCreated()));

                stmt.executeUpdate();

                if (!ps.getComments().isEmpty())
                {
                    if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
                    {
                        conn.createStatement().execute("SET IDENTITY_INSERT journal OFF");
                    }
                    CommentBuilder cb = new CommentBuilder(NotebookEntryType.JOURNAL, ps.getId());
                    cb.doJsonInsert(ps.getComments(), conn);
                    if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
                    {
                        conn.createStatement().execute("SET IDENTITY_INSERT journal ON");
                    }
                }

                if (++txCount > 50)
                {
                    conn.commit();
                    txCount = 0;
                }
            }
            conn.commit();
            if (DBConnection.DB_IN_USE == DBConnection.RDBMS_ENUM.MSSQLServer )
            {
                conn.createStatement().execute("SET IDENTITY_INSERT journal OFF");
            }

        }catch (SQLException ex) {
            LOGGER.error("restoreJsonDump(): SQLException: errorCode: {}, SQLstate: {}, message: {}", ex.getErrorCode(), ex.getSQLState(), ex.getMessage());
            throw new GNDBException(ex, ex.getErrorCode(), ex.getSQLState());
        }
    }	// restoreJsonDump(JsonObjects)

}
