/*
 *
 *  Copyright (C) 2021, 2023 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

/*
    Change log
	3.1.0	Use jakarta implementation of JSON
*/

package uk.co.gardennotebook.mysql;

import uk.co.gardennotebook.spi.*;

import jakarta.json.*;
import java.beans.PropertyChangeListener;
import java.time.LocalDateTime;
import java.time.Year;
import java.util.*;

/**
 *{@inheritDoc}
 *
 *	@author	Andy Gegg
 *	@version	3.1.0
 *	@since	3.0.0
 */

final class CroppingPlan implements ICroppingPlan
{
    private final FlagHandler<ICroppingPlan> flagHandler;
    {
        flagHandler = new FlagHandler<>(this);
    }

    private final int id;
    /*
    *   The type of crop being planned
     */
    private final int cropRotationGroupId;
    /*
     *	Where the plan is for.
     */
    private final int locationId;
    /*
    *   The cropping year being planned.
     */
    private final Year yearOfPlan;

    private final LocalDateTime lastUpdated;
    private final LocalDateTime created;
    private final List<Comment> commentsList;

    public CroppingPlan(int id,
                        int cropRotationGroupId,
                        int locationId,
                        Year yearOfPlan,
                        LocalDateTime lastUpdated,
                        LocalDateTime created,
                        Comment... comments)
    {
        this.id = id;
        this.cropRotationGroupId = cropRotationGroupId;
        this.locationId = locationId;
        this.yearOfPlan = yearOfPlan;
        this.lastUpdated = lastUpdated;
        this.created = created;

        if (comments != null && comments.length>0)
        {
            this.commentsList = new ArrayList<>(Arrays.asList(comments));
        }
        else
        {
            this.commentsList = Collections.emptyList();
        }
    }

    /**
     *	Build an immutable CroppingPlan entry cloning the given CroppingPlan entry but adding the comments list
     */
    CroppingPlan(
            final CroppingPlan toCopy,
            final Comment... comments)
    {
        this(toCopy, Arrays.asList(comments));
    }

    /**
     *	Build an immutable CroppingPlan entry cloning the given CroppingPlan entry but adding the comments list
     */
    CroppingPlan(
            final CroppingPlan toCopy,
            final List<Comment> comments)
    {
        this.id = toCopy.id;
        this.cropRotationGroupId = toCopy.cropRotationGroupId;
        this.locationId = toCopy.locationId;
        this.yearOfPlan = toCopy.yearOfPlan;
        this.lastUpdated = toCopy.lastUpdated;
        this.created = toCopy.created;

        if (comments != null && comments.size()>0)
        {
            if (toCopy.commentsList.size()>0)
            {
                // append new comments to previous list
                this.commentsList = new ArrayList<>(toCopy.commentsList);
                this.commentsList.addAll(comments);
            }
            else
            {	// no comments on original item
                this.commentsList = new ArrayList<>(comments);
            }
        }
        else
        {	// no new comments to add
            this.commentsList = toCopy.commentsList;
        }
    }

    /**
     *	Build an immutable CroppingPlan entry from a JSON dump.
     *	The dumped object must be complete (all non-nullable fields must have values) except
     *	the id field can be null or absent to indicate that this is a new item to be inserted.
     *
     *	@param	json	a JsonObject holding all the fields for a full initialisation.
     */
    CroppingPlan(JsonObject json)
    {
        this.id = json.getInt("id", -1);
        this.cropRotationGroupId = json.getInt("cropRotationGroupId");
        this.locationId = json.getInt("locationId");

        this.yearOfPlan = Year.parse(json.getString("yearOfPlan"));

        this.lastUpdated = LocalDateTime.parse(json.getString("lastUpdated"));
        this.created = LocalDateTime.parse(json.getString("created"));

        JsonArray jsonComments = json.getJsonArray("comments");
        if (jsonComments != null && !jsonComments.isEmpty())
        {// there is probably only one comment
            this.commentsList = new ArrayList<>(jsonComments.size());
            for (JsonObject ct : jsonComments.getValuesAs(JsonObject.class))
            {
                this.commentsList.add(new Comment(ct));
            }
        }
        else
        {
            this.commentsList = Collections.emptyList();
        }
    }	//constructor from JSON

    int getId()
    {
        return id;
    }
    @Override
    public Integer getKey()
    {
        return id;
    }

    @Override
    public NotebookEntryType getType()
    {
        return NotebookEntryType.CROPPINGPLAN;
    }

    int getCropRotationGroupId()
    {
        return cropRotationGroupId;
    }
    @Override
    public ICropRotationGroup getCropRotationGroup()
    {
        return MySQLCache.cacheCropRotationGroup.get(cropRotationGroupId);
    }

    int getLocationId()
    {
        return locationId;
    }
    @Override
    public ILocation getLocation()
    {
        return MySQLCache.cacheLocation.get(locationId);
    }

    @Override
    public Year getYearOfPlan()
    {
        return yearOfPlan;
    }

    @Override
    public LocalDateTime getLastUpdated()
    {
        return lastUpdated;
    }

    @Override
    public LocalDateTime getCreated()
    {
        return created;
    }

    @Override
    public ICroppingActualLister getCroppingActual()
    {
        return new CroppingActualLister().croppingPlan(this);
//        return null;

    }

    @Override
    public List<IComment> getComments()
    {
        return new ArrayList<>(this.commentsList);
    }

    JsonObjectBuilder toJson(JsonBuilderFactory jsonFactory)
    {
        JsonObjectBuilder jsonBuilder = jsonFactory.createObjectBuilder();
        jsonBuilder.add("id", id);
        jsonBuilder.add("cropRotationGroupId", cropRotationGroupId);
        jsonBuilder.add("locationId", locationId);

        jsonBuilder.add("yearOfPlan", yearOfPlan.toString());

        jsonBuilder.add("lastUpdated", lastUpdated.toString());
        jsonBuilder.add("created", created.toString());

        if (commentsList != null && !commentsList.isEmpty())
        {// no point writing an empty comments array (the loaders handle this)
            JsonArrayBuilder jsonComments = jsonFactory.createArrayBuilder();
            for (Comment ct : commentsList)
            {
                jsonComments.add(ct.toJson(jsonFactory));
            }
            jsonBuilder.add("comments", jsonComments);
        }
        jsonBuilder.add("JsonMode", "DUMP");
        jsonBuilder.add("JsonNBClass", "CroppingPlan");
        return jsonBuilder;
    }	//	toJson

    @Override
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener)
    {
        flagHandler.addPropertyChangeListener(propertyName, listener);
    }

    @Override
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener)
    {
        flagHandler.removePropertyChangeListener(propertyName, listener);
    }

    @Override
    public void flagDeleted()
    {
        flagHandler.flagDeleted();
    }

    @Override
    public void flagReplaced(ICroppingPlan newValue)
    {
        flagHandler.flagReplaced(newValue, newValue::addPropertyChangeListener);
    }

    @Override
    public void flagChildDeleted(ICroppingActual child)
    {
        flagHandler.flagChildDeleted("CroppingActual", child);
    }

    @Override
    public void flagChildAdded(ICroppingActual child)
    {
        flagHandler.flagChildAdded("CroppingActual", child);
    }

    @Override
    public String toString() {
        return "CroppingPlan: " + "id: " + id + ", " +
                "cropRotationGroupId: " + cropRotationGroupId + ", " +
                "locationId: " + locationId + ", " +
                "yearOfPlan: " + yearOfPlan + ", " +
                "lastUpdated: " + lastUpdated + ", " +
                "created: " + created;
    }

}
