/*
 * Copyright (C) 2018, 2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Support converting ToDoList items (back) into Reminders to delay the action.
			Support 'watch-for' entries from PurchaseItems.
 */

package uk.co.gardennotebook.spi;

import java.util.List;

import java.time.*;

/**
A list of reminders of things to do. <BR>Reminders can be 'one-off' or repeated at various intervals (e.g. 'sow lettuce in March' shown every year.).
*/
public interface IReminderLister
{
	/**
	*	Returns the Reminder entries matching the given criteria.
	*	If no criteria have been specified all Reminder entries will be returned.
	*	Criteria are applied additively (i.e. criterion AND criterion).
	*
	*	@return	A list of Reminder entries matching previously specified criteria.
	*
	*	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
	*			The causal Exception can be retrieved via <code>getCause()</code>
	*/
	public List<IReminder> fetch() throws GNDBException;

	/**
	*	Select only these Reminder entries
	*	May be called repeatedly to add to the list of Reminder entries to be returned
	*
	*	@param items	the list of Reminder entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IReminderLister reminder(IReminder... items);

	/**
	*	Select only these Reminder entries
	*	May be called repeatedly to add to the list of Reminder entries to be returned
	*
	*	@param items	the list of Reminder entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IReminderLister reminder(List<IReminder> items);

	/**
	*	Select only those Reminder entries which are for these GroundworkActivity entries.
	*	May be called repeatedly to add to the list of GroundworkActivity entries to be selected.
	*
	*	@param items	the list of GroundworkActivity items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IReminderLister groundworkActivity(IGroundworkActivity... items);

	/**
	*	Select only those Reminder entries which are for these GroundworkActivity entries.
	*	May be called repeatedly to add to the list of GroundworkActivity entries to be selected
	*
	*	@param items	the list of GroundworkActivity items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IReminderLister groundworkActivity(List<IGroundworkActivity> items);

	/**
	*	Select only those Reminder entries which are for these HusbandryClass entries.
	*	May be called repeatedly to add to the list of HusbandryClass entries to be selected.
	*
	*	@param items	the list of HusbandryClass items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IReminderLister husbandryClass(IHusbandryClass... items);

	/**
	*	Select only those Reminder entries which are for these HusbandryClass entries.
	*	May be called repeatedly to add to the list of HusbandryClass entries to be selected
	*
	*	@param items	the list of HusbandryClass items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IReminderLister husbandryClass(List<IHusbandryClass> items);

	/**
	*	Select only those Reminder entries which are for these PlantSpecies entries.
	*	May be called repeatedly to add to the list of PlantSpecies entries to be selected.
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Reminder entries matching either will be selected.
	*
	*	@param items	the list of PlantSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IReminderLister plantSpecies(IPlantSpecies... items);

	/**
	*	Select only those Reminder entries which are for these PlantSpecies entries.
	*	May be called repeatedly to add to the list of PlantSpecies entries to be selected
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Reminder entries matching either will be selected.
	*
	*	@param items	the list of PlantSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IReminderLister plantSpecies(List<IPlantSpecies> items);

	/**
	*	Select only those Reminder entries which are for these PlantVariety entries.
	*	May be called repeatedly to add to the list of PlantVariety entries to be selected.
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Reminder entries matching either will be selected.
	*
	*	@param items	the list of PlantVariety items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IReminderLister plantVariety(IPlantVariety... items);

	/**
	*	Select only those Reminder entries which are for these PlantVariety entries.
	*	May be called repeatedly to add to the list of PlantVariety entries to be selected
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Reminder entries matching either will be selected.
	*
	*	@param items	the list of PlantVariety items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IReminderLister plantVariety(List<IPlantVariety> items);

	/**
	 *	Select only those Reminder entries which are for these Husbandry entries.
	 *	May be called repeatedly to add to the list of Husbandry entries to be selected.
	 *
	 *	@param items	the list of Husbandry items.  A null value or empty list is a null-op
	 *	@return	this Lister
	 */
	public IReminderLister husbandry(IHusbandry... items);

	/**
	 *	Select only those Reminder entries which are for these Husbandry entries.
	 *	May be called repeatedly to add to the list of Husbandry entries to be selected
	 *
	 *	@param items	the list of Husbandry items.  A null value or empty list is a null-op
	 *	@return	this Lister
	 */
	public IReminderLister husbandry(List<IHusbandry> items);

	/**
	 *	Select only those Reminder entries which are for these Purchase items.
	 *	May be called repeatedly to add to the list of Purchase items to be selected.
	 *
	 *	@param items	the list of HPurchase items.  A null value or empty list is a null-op
	 *	@return	this Lister
	 */
	public IReminderLister purchaseItem(IPurchaseItem... items);

	/**
	 *	Select only those Reminder entries which are for these Purchase items.
	 *	May be called repeatedly to add to the list of Purchase items to be selected
	 *
	 *	@param items	the list of Purchase items.  A null value or empty list is a null-op
	 *	@return	this Lister
	 */
	public IReminderLister purchaseItem(List<IPurchaseItem> items);

	/**
	*	Select only those Reminder entries which are on or after this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no earliest date will be used in selection
	*
	*	@param	date	the earliest date to match
	*	@return	this Lister
	*/
	public IReminderLister fromShowFrom(LocalDate date);
	/**
	*	Select only those Reminder entries which are on or before this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no latest date will be used in selection
	*
	*	@param	date	the latest date to match
	*	@return	this Lister
	*/
	public IReminderLister toShowFrom(LocalDate date);

	/**
	*	Select only those Reminder entries which are on or after this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no earliest date will be used in selection
	*
	*	@param	date	the earliest date to match
	*	@return	this Lister
	*/
	public IReminderLister fromRepeatUntil(LocalDate date);
	/**
	*	Select only those Reminder entries which are on or before this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no latest date will be used in selection
	*
	*	@param	date	the latest date to match
	*	@return	this Lister
	*/
	public IReminderLister toRepeatUntil(LocalDate date);

}
