/*
 * Copyright (C) 2018, 2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change Log
	3.0.0	Add CroppingActual handling
 */

package uk.co.gardennotebook.spi;

import java.beans.PropertyChangeListener;
import java.util.List;
import java.time.*;

import java.util.Optional;

/**
*	A particular variety of a plant species.
 *	Note that the common name is NOT a unique value as different species may have varieties with the same name - e.g. Money Maker tomato AND aubergine.<BR>
Note the use of a Trigger to reflect changes to commonName into Product.
*
*	@author	Andy Gegg
*	@version	3.0.0
*	@since	1.0
*/
public interface IPlantVariety extends INotebookEntry
{
	/**
	*	The owning PlantSpecies.
	*
	*	@return	the PlantSpecies
	*/
	public IPlantSpecies getPlantSpecies();

	/**
	*	Get the value of commonName.
	*
	*	@return	the value of commonName
	*/
	public String getCommonName();

	/**
	*	Get the value of latinName (if present).
	*
	*	@return	the value of latinName
	*/
	public Optional<String> getLatinName();

	/**
	*	Get the value of synonymSet (if present).
	*
	*	@return	the value of synonymSet
	*/
	public Optional<String> getSynonymSet();

	/**
	*	Get the value of description (if present).
	*
	*	@return	the value of description
	*/
	public Optional<String> getDescription();

	/**
	*	Get the value of utility (if present).
	*
	*	@return	the value of utility
	*/
	public Optional<String> getUtility();

	/**
	*	Get the value of hardiness (if present).
	*
	*	@return	the value of hardiness
	*/
	public Optional<String> getHardiness();

	/**
	*	Get the value of lifeType (if present).
	*
	*	@return	the value of lifeType
	*/
	public Optional<String> getLifeType();

	/**
	*	Get the value of plantType (if present).
	*
	*	@return	the value of plantType
	*/
	public Optional<String> getPlantType();

	/**
	*	Get the value of lastUpdated.
	*
	*	@return	the value of lastUpdated
	*/
	public LocalDateTime getLastUpdated();

	/**
	*	Get the value of created.
	*
	*	@return	the value of created
	*/
	public LocalDateTime getCreated();


	/**
	*	Get the AfflictionEvent entries for this PlantVariety.
	*
	*	@return	 a lister for the AfflictionEvent entries
	*/
	public IAfflictionEventLister getAfflictionEvent();

	/**
	*	Get the Groundwork entries for this PlantVariety.
	*
	*	@return	 a lister for the Groundwork entries
	*/
	public IGroundworkLister getGroundwork();

	/**
	*	Get the Husbandry entries for this PlantVariety.
	*
	*	@return	 a lister for the Husbandry entries
	*/
	public IHusbandryLister getHusbandry();

	/**
	*	Get the PlantNote entries for this PlantVariety.
	*
	*	@return	 a lister for the PlantNote entries
	*/
	public IPlantNoteLister getPlantNote();

	/**
	*	Get the Product entries for this PlantVariety.
	*
	*	@return	 a lister for the Product entries
	*/
	public IProductLister getProduct();

	/**
	*	Get the Reminder entries for this PlantVariety.
	*
	*	@return	 a lister for the Reminder entries
	*/
	public IReminderLister getReminder();

	/**
	*	Get the SaleItem entries for this PlantVariety.
	*
	*	@return	 a lister for the SaleItem entries
	*/
	public ISaleItemLister getSaleItem();

	/**
	*	Get the ToDoList entries for this PlantVariety.
	*
	*	@return	 a lister for the ToDoList entries
	*/
	public IToDoListLister getToDoList();

	/**
	 *	Get the CroppingActual entries for this PlantVariety.
	 *
	 *	@return	 a lister for the CroppingActual entries
	 */
	public ICroppingActualLister getCroppingActual();

	/**
	*	Add a change listener to the PlantVariety.
	*
	*	@param	propertyName	identifies the change being flagged.  One of:<UL>
	*			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
	*			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
	*			<LI>Comment	-	this comment has been added to or deleted from this PlantVariety, newValue is an added comment, oldValue is a deleted comment
	*			<LI>AfflictionEvent	-	this AfflictionEvent has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>Groundwork	-	this Groundwork has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>Husbandry	-	this Husbandry has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>PlantNote	-	this PlantNote has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>Product	-	this Product has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>Reminder	-	this Reminder has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>SaleItem	-	this SaleItem has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>ToDoList	-	this ToDoList has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	 *			<LI>CroppingActual	-	this CroppingActual has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	</UL>
	*	@param	listener	a PropertyChangeListener (may be a Lambda expression)
	*/
	public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener);

	/**
	*	Remove a change listener from the PlantVariety.
	*
	*	@param	propertyName	identifies the change being flagged.  One of:<UL>
	*			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
	*			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
	*			<LI>Comment	-	this comment has been added to or deleted from this PlantVariety, newValue is an added comment, oldValue is a deleted comment
	*			<LI>AfflictionEvent	-	this AfflictionEvent has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>Groundwork	-	this Groundwork has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>Husbandry	-	this Husbandry has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>PlantNote	-	this PlantNote has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>Product	-	this Product has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>Reminder	-	this Reminder has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>SaleItem	-	this SaleItem has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	*			<LI>ToDoList	-	this ToDoList has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	 *			<LI>CroppingActual	-	this CroppingActual has been added to or deleted from this PlantVariety's children, newValue is an added object, oldValue is the deleted object
	</UL>
	*	@param	listener	the listener to remove
	*/
	public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener);

	/**
	*	Signal to all listeners that this PlantVariety has been deleted.
	*
	*/
	public void flagDeleted();

	/**
	*	Signal to all listeners that this PlantVariety has been replaced.
	*
	*	@param	newValue	the replacement PlantVariety
	*/
	public void flagReplaced(IPlantVariety newValue);

	/**
	*	Signal to all listeners that a child has been deleted.
	*
	*	@param	child	this AfflictionEvent is no longer a child of this PlantVariety
	*/
	public void flagChildDeleted(IAfflictionEvent child);

	/**
	*	Signal to all listeners that a child has been added.
	*
	*	@param	child	this AfflictionEvent has been added as a child of this PlantVariety
	*/
	public void flagChildAdded(IAfflictionEvent child);

	/**
	*	Signal to all listeners that a child has been deleted.
	*
	*	@param	child	this Groundwork is no longer a child of this PlantVariety
	*/
	public void flagChildDeleted(IGroundwork child);

	/**
	*	Signal to all listeners that a child has been added.
	*
	*	@param	child	this Groundwork has been added as a child of this PlantVariety
	*/
	public void flagChildAdded(IGroundwork child);

	/**
	*	Signal to all listeners that a child has been deleted.
	*
	*	@param	child	this Husbandry is no longer a child of this PlantVariety
	*/
	public void flagChildDeleted(IHusbandry child);

	/**
	*	Signal to all listeners that a child has been added.
	*
	*	@param	child	this Husbandry has been added as a child of this PlantVariety
	*/
	public void flagChildAdded(IHusbandry child);

	/**
	*	Signal to all listeners that a child has been deleted.
	*
	*	@param	child	this PlantNote is no longer a child of this PlantVariety
	*/
	public void flagChildDeleted(IPlantNote child);

	/**
	*	Signal to all listeners that a child has been added.
	*
	*	@param	child	this PlantNote has been added as a child of this PlantVariety
	*/
	public void flagChildAdded(IPlantNote child);

	/**
	*	Signal to all listeners that a child has been deleted.
	*
	*	@param	child	this Product is no longer a child of this PlantVariety
	*/
	public void flagChildDeleted(IProduct child);

	/**
	*	Signal to all listeners that a child has been added.
	*
	*	@param	child	this Product has been added as a child of this PlantVariety
	*/
	public void flagChildAdded(IProduct child);

	/**
	*	Signal to all listeners that a child has been deleted.
	*
	*	@param	child	this Reminder is no longer a child of this PlantVariety
	*/
	public void flagChildDeleted(IReminder child);

	/**
	*	Signal to all listeners that a child has been added.
	*
	*	@param	child	this Reminder has been added as a child of this PlantVariety
	*/
	public void flagChildAdded(IReminder child);

	/**
	*	Signal to all listeners that a child has been deleted.
	*
	*	@param	child	this SaleItem is no longer a child of this PlantVariety
	*/
	public void flagChildDeleted(ISaleItem child);

	/**
	*	Signal to all listeners that a child has been added.
	*
	*	@param	child	this SaleItem has been added as a child of this PlantVariety
	*/
	public void flagChildAdded(ISaleItem child);

	/**
	*	Signal to all listeners that a child has been deleted.
	*
	*	@param	child	this ToDoList is no longer a child of this PlantVariety
	*/
	public void flagChildDeleted(IToDoList child);

	/**
	*	Signal to all listeners that a child has been added.
	*
	*	@param	child	this ToDoList has been added as a child of this PlantVariety
	*/
	public void flagChildAdded(IToDoList child);

	/**
	 *	Signal to all listeners that a child has been deleted.
	 *
	 *	@param	child	this CroppingActual has been added as a child of this PlantSpecies
	 */
	public void flagChildDeleted(ICroppingActual child);

	/**
	 *	Signal to all listeners that a child has been added.
	 *
	 *	@param	child	this CroppingActual has been added as a child of this PlantSpecies
	 */
	public void flagChildAdded(ICroppingActual child);

}

