/*
 * Copyright (C) 2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	First version
 */

package uk.co.gardennotebook.spi;

import java.util.List;

/**
These are the various locations in the garden.
 *
 *	@author	Andy Gegg
 *	@version	3.0.0
 *	@since	3.0.0
*/
public interface ILocationLister
{
	/**
	*	Returns the Location entries matching the given criteria.
	*	If no criteria have been specified all Location entries will be returned.
	*	Criteria are applied additively (i.e. criterion AND criterion).
	*
	*	@return	A list of Location entries matching previously specified criteria.
	*
	*	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
	*			The causal Exception can be retrieved via <code>getCause()</code>
	*/
	public List<? extends ILocation> fetch() throws GNDBException;

	/**
	 *	Returns the immediate children of the Location entries matching the given criteria.
	 *	If no criteria have been specified all Location entries will be returned.
	 *	Criteria are applied additively (i.e. criterion AND criterion).
	 *
	 *	@return	A list of Location entries matching previously specified criteria.
	 *
	 *	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
	 *			The causal Exception can be retrieved via <code>getCause()</code>
	 */
	public List<? extends ILocation> fetchSubLocations() throws GNDBException;

	/**
	 *	Select those Location entries which are sublocations of the Location entries matching the given criteria, at any depth.
	 *	(I.E. in effect, call fetchSubLocations() for the list, then call fetchSubLocations() for the resultant list, etc.)
	 *
	 * @apiNote
	 *	Note the use of this in the JSONdump routine where a tree order list is required so that mother loccations
	 *	are written before their daughters, or there is a foreign key reference error on restore
	 *
	 *	@return	this Lister
	 */
	public List<? extends ILocation> fetchDescendantLocations() throws GNDBException;

	/**
	 *	Returns the parent(s) of the Location entries matching the given criteria.
	 *	If no criteria have been specified all non-leaf Location entries will be returned.
	 *	Criteria are applied additively (i.e. criterion AND criterion).
	 *
	 *	@return	A list of Location entries matching previously specified criteria.
	 *
	 *	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
	 *			The causal Exception can be retrieved via <code>getCause()</code>
	 */
	public List<? extends ILocation> fetchSuperLocation() throws GNDBException;

	/**
	*	Select only these Location entries
	*	May be called repeatedly to add to the list of Affliction entries to be returned
	*
	*	@param items	the list of Location entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public ILocationLister location(final ILocation... items);

	/**
	*	Select only these Location entries
	*	May be called repeatedly to add to the list of Location entries to be returned
	*
	*	@param items	the list of Location entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public ILocationLister location(final List<ILocation> items);

	/**
	 *	Select only the top level Location(s).
	 *	Will be ignored by fetchSubLocation().
	 *
	 * @return	this Lister
	 */
	public ILocationLister forceTopLevel();


	/**
	*	Select only the Location entries with these names
	*	May be called repeatedly to add to the list of names to be selected
	*
	*	@param items	the list of name values.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public ILocationLister name(final String... items);

}
