/*
 * Copyright (C) 2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	First version
 */

package uk.co.gardennotebook.spi;

import java.beans.PropertyChangeListener;
import java.time.LocalDateTime;
import java.util.Optional;

/**
*	A physical location, such as a bed or a greenhouse.
 *	The installation scripts include a single entry 'The Garden' which is a top-level parent for other Locations, such as flower beds, lawns, greenhouses, etc.
 *	'The Garden' is given a default geometry of a 100x100 square.
 *	You may have more than one top-level Location, such as 'The Garden', 'The Allotment';
 *	these will be treated as physically distinct areas, all the children of each top level Location must be physically within that top level parent if a geometry value is supplied.
 *
 * A location can also be 'virtual' in that it is just a convenience for grouping other locations.
*
*	@author	Andy Gegg
*	@version	3.0.0
*	@since	3.0.0
*/
public interface ILocation extends INotebookEntry
{
	/**
	*	The parent Location.
	*	The parent can be physical or virtual.
	*
	*
	*	@return	the parent Location
	*/
	public Optional<ILocation> getParentLocation();

	/**
	 * Check if this Location has a parent Location
	 *
	 * @return	true	if this Location has a parent
	 */
	public boolean hasParent();

	/**
	*	Get the value of name.
	*
	*	@return	the value of name
	*/
	public String getName();

	/**
	*	Get the value of description (if present).
	*
	*	@return	the value of description
	*/
	public Optional<String> getDescription();

	/**
	 *	Get the 'undercover' flag
	 */
	public boolean isUnderCover();

	/**
	 *	Get the Geometry for thw Location
	 */
	public Optional<String> getGeometry();

	/**
	*	Get the value of lastUpdated.
	*
	*	@return	the value of lastUpdated
	*/
	public LocalDateTime getLastUpdated();

	/**
	*	Get the value of created.
	*
	*	@return	the value of created
	*/
	public LocalDateTime getCreated();


	/**
	*	Get the AfflictionEvents for this Location.
	*
	*	@return	 a lister for the AfflictionEvent entries
	*/
	public IAfflictionEventLister getAfflictionEvent();

	/**
	 *	Get the Groundwork events for this Location.
	 *
	 *	@return	 a lister for the Groundwork entries
	 */
	public IGroundworkLister getGroundwork();

	/**
	*	Get the Husbandry entries for this Location.
	*
	*	@return	 a lister for the Husbandry entries
	*/
	public IHusbandryLister getHusbandry();

	// TODO	Add getReview

	/**
	*	Add a change listener to the Affliction.
	*
	*	@param	propertyName	identifies the change being flagged.  One of:<UL>
	*			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
	*			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
	*			<LI>Comment	-	this comment has been added to or deleted from this Affliction, newValue is an added comment, oldValue is a deleted comment
	*			<LI>AfflictionEvent	-	this AfflictionEvent has been added to or deleted from this Affliction's children, newValue is an added object, oldValue is the deleted object
	*			<LI>Husbandry	-	this Husbandry has been added to or deleted from this Affliction's children, newValue is an added object, oldValue is the deleted object
	*			</UL>
	*	@param	listener	a PropertyChangeListener (may be a Lambda expression)
	*/
	public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener);

	/**
	*	Remove a change listener from the Affliction.
	*
	*	@param	propertyName	identifies the change being flagged.  One of:<UL>
	*			<LI>deleted	-	this object has been deleted from the database, oldValue is the deleted object
	*			<LI>replaced	-	this object has been replaced in the database, oldValue is the deleted object, newValue its replacement
	*			<LI>Comment	-	this comment has been added to or deleted from this Affliction, newValue is an added comment, oldValue is a deleted comment
	*			<LI>AfflictionEvent	-	this AfflictionEvent has been added to or deleted from this Affliction's children, newValue is an added object, oldValue is the deleted object
	*			<LI>Husbandry	-	this Husbandry has been added to or deleted from this Affliction's children, newValue is an added object, oldValue is the deleted object
	*			</UL>
	*	@param	listener	the listener to remove
	*/
	public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener);

	/**
	*	Signal to all listeners that this Affliction has been deleted.
	*
	*/
	public void flagDeleted();

	/**
	*	Signal to all listeners that this Affliction has been replaced.
	*
	*	@param	newValue	the replacement Affliction
	*/
	public void flagReplaced(ILocation newValue);

	/**
	 *	Signal to all listeners that a child has been deleted.
	 *
	 *	@param	child	this Location is no longer a child of this Location
	 */
	public void flagChildDeleted(ILocation child);

	/**
	 *	Signal to all listeners that a child has been added.
	 *
	 *	@param	child	this Location has been added as a child of this Location
	 */
	public void flagChildAdded(ILocation child);

	/**
	*	Signal to all listeners that a child has been deleted.
	*
	*	@param	child	this AfflictionEvent is no longer a child of this Affliction
	*/
	public void flagChildDeleted(IAfflictionEvent child);

	/**
	*	Signal to all listeners that a child has been added.
	*
	*	@param	child	this AfflictionEvent has been added as a child of this Affliction
	*/
	public void flagChildAdded(IAfflictionEvent child);

	/**
	 *	Signal to all listeners that a child has been deleted.
	 *
	 *	@param	child	this Groundwork activity is no longer a child of this Affliction
	 */
	public void flagChildDeleted(IGroundwork child);

	/**
	 *	Signal to all listeners that a child has been added.
	 *
	 *	@param	child	this Groundwork activity has been added as a child of this Affliction
	 */
	public void flagChildAdded(IGroundwork child);

	/**
	*	Signal to all listeners that a child has been deleted.
	*
	*	@param	child	this Husbandry activity is no longer a child of this Affliction
	*/
	public void flagChildDeleted(IHusbandry child);

	/**
	*	Signal to all listeners that a child has been added.
	*
	*	@param	child	this Husbandry activity has been added as a child of this Affliction
	*/
	public void flagChildAdded(IWildlife child);

	/**
	 *	Signal to all listeners that a child has been deleted.
	 *
	 *	@param	child	this Husbandry activity is no longer a child of this Affliction
	 */
	public void flagChildDeleted(IWildlife child);

	/**
	 *	Signal to all listeners that a child has been added.
	 *
	 *	@param	child	this Location activity has been added as a child of this Affliction
	 */
	public void flagChildAdded(IHusbandry child);

	/**
	 *	Signal to all listeners that a child has been deleted.
	 *
	 *	@param	child	this Location is no longer a child of this Affliction
	 */
	public void flagChildDeleted(IReview child);

	/**
	 *	Signal to all listeners that a child has been added.
	 *
	 *	@param	child	this Location has been added as a child of this Affliction
	 */
	public void flagChildAdded(IReview child);

	/**
	 *	Signal to all listeners that a child has been deleted.
	 *
	 *	@param	child	this Location is no longer a child of this Affliction
	 */
	public void flagChildDeleted(ICroppingPlan child);

	/**
	 *	Signal to all listeners that a child has been added.
	 *
	 *	@param	child	this Location has been added as a child of this Affliction
	 */
	public void flagChildAdded(ICroppingPlan child);

}

