/*
 * Copyright (C) 2018, 2021 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	Support Locations and Quantity field
 */

package uk.co.gardennotebook.spi;

import java.util.List;

import java.time.*;

/**
*   Diary entries for events in the life of a crop.
*
*	@author	Andy Gegg
*	@version	3.0.0
*	@since	1.0
*/
public interface IHusbandryLister
{
	/**
	*	Returns the Husbandry entries matching the given criteria.
	*	If no criteria have been specified all Husbandry entries will be returned.
	*	Criteria are applied additively (i.e. criterion AND criterion).
	*
	*	@return	A list of Husbandry entries matching previously specified criteria.
	*
	*	@throws	GNDBException	If the underlying persistence store (e.g. a database) throws an exception.
	*			The causal Exception can be retrieved via <code>getCause()</code>
	*/
	public List<IHusbandry> fetch() throws GNDBException;

	/**
	*	Select only these Husbandry entries
	*	May be called repeatedly to add to the list of Husbandry entries to be returned
	*
	*	@param items	the list of Husbandry entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IHusbandryLister husbandry(IHusbandry... items);

	/**
	*	Select only these Husbandry entries
	*	May be called repeatedly to add to the list of Husbandry entries to be returned
	*
	*	@param items	the list of Husbandry entries.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IHusbandryLister husbandry(List<IHusbandry> items);

	/**
	*	Select only those Husbandry entries which are for these Affliction entries.
	*	May be called repeatedly to add to the list of Affliction entries to be selected.
	*
	*	@param items	the list of Affliction items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IHusbandryLister affliction(IAffliction... items);

	/**
	*	Select only those Husbandry entries which are for these Affliction entries.
	*	May be called repeatedly to add to the list of Affliction entries to be selected
	*
	*	@param items	the list of Affliction items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IHusbandryLister affliction(List<IAffliction> items);

	/**
	 *	Select only those Husbandry entries which are for these Locations.
	 *	May be called repeatedly to add to the list of Locations to be selected.
	 *
	 *	@param items	the list of Locations.  A null value or empty list is a null-op
	 *	@return	this Lister
	 */
	public IHusbandryLister location(ILocation... items);

	/**
	 *	Select only those Husbandry entries which are for these Locations.
	 *	May be called repeatedly to add to the list of Locations to be selected
	 *
	 *	@param items	the list of Locations.  A null value or empty list is a null-op
	 *	@return	this Lister
	 */
	public IHusbandryLister location(List<ILocation> items);

	/**
	*	Select only those Husbandry entries which are for these PlantSpecies entries.
	*	May be called repeatedly to add to the list of PlantSpecies entries to be selected.
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Husbandry entries matching either will be selected.
	*
	*	@param items	the list of PlantSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IHusbandryLister plantSpecies(IPlantSpecies... items);

	/**
	*	Select only those Husbandry entries which are for these PlantSpecies entries.
	*	May be called repeatedly to add to the list of PlantSpecies entries to be selected
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Husbandry entries matching either will be selected.
	*
	*	@param items	the list of PlantSpecies items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IHusbandryLister plantSpecies(List<IPlantSpecies> items);

	/**
	*	Select only those Husbandry entries which are for these HusbandryClass entries.
	*	May be called repeatedly to add to the list of HusbandryClass entries to be selected.
	*
	*	@param items	the list of HusbandryClass items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IHusbandryLister husbandryClass(IHusbandryClass... items);

	/**
	*	Select only those Husbandry entries which are for these HusbandryClass entries.
	*	May be called repeatedly to add to the list of HusbandryClass entries to be selected
	*
	*	@param items	the list of HusbandryClass items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IHusbandryLister husbandryClass(List<IHusbandryClass> items);

	/**
	*	Select only those Husbandry entries which are for these PlantVariety entries.
	*	May be called repeatedly to add to the list of PlantVariety entries to be selected.
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Husbandry entries matching either will be selected.
	*
	*	@param items	the list of PlantVariety items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IHusbandryLister plantVariety(IPlantVariety... items);

	/**
	*	Select only those Husbandry entries which are for these PlantVariety entries.
	*	May be called repeatedly to add to the list of PlantVariety entries to be selected
	*	If PlantSpecies and PlantVariety lists are BOTH supplied, Husbandry entries matching either will be selected.
	*
	*	@param items	the list of PlantVariety items.  A null value or empty list is a null-op
	*	@return	this Lister
	*/
	public IHusbandryLister plantVariety(List<IPlantVariety> items);

	/**
	*	Select only those Husbandry entries which are on or after this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no earliest date will be used in selection
	*
	*	@param	date	the earliest date to match
	*	@return	this Lister
	*/
	public IHusbandryLister fromDate(LocalDate date);
	/**
	*	Select only those Husbandry entries which are on or before this date
	*	A null value will be ignored
	*	If called repeatedly only the last value given will be used
	*	If this method is not called no latest date will be used in selection
	*
	*	@param	date	the latest date to match
	*	@return	this Lister
	*/
	public IHusbandryLister toDate(LocalDate date);

}
