/*
 * Copyright (C) 2018-2020, 2022 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.1.0   Implement functionality
    2.4.0   Support tab-out on comments
    2.8.0   Permit edit of Product based items, in line with ShoppingListCat
    3.0.1	CommentAdder handling moved to separate class
    3.0.2	Check before Cancel if changes have been made
    3.0.4	Use new convenience class NotebookBeanDeleter for deletion.
    		Use new convenience class NotebookBeanCanceller to handle edit cancellation.
    		Use new convenience class EditorCommentTableHandler to handle Comment table construction.
    		Set focus on first field.
 */

package uk.co.gardennotebook;

import java.io.IOException;
//import java.util.Optional;
import java.util.ResourceBundle;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
//import java.time.format.DateTimeFormatter;
//import java.time.format.FormatStyle;

import javafx.application.Platform;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleObjectProperty;
//import javafx.beans.value.ChangeListener;
import javafx.event.ActionEvent;
import javafx.event.Event;
import javafx.fxml.FXML;
import javafx.fxml.FXMLLoader;
import javafx.scene.Node;
//import javafx.scene.text.Text;
//import javafx.scene.control.Alert;
//import javafx.scene.control.ButtonType;
import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.TableColumn;
import javafx.scene.control.TableView;
import javafx.scene.control.TextField;
import javafx.scene.layout.GridPane;
import uk.co.gardennotebook.fxbean.CommentBean;
import uk.co.gardennotebook.spi.GNDBException;

import uk.co.gardennotebook.fxbean.ShoppingListBean;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;

/**
	*	Controller class for create/update of ShoppingList items
	*
	*	@author Andy Gegg
	*	@version	3.0.4
	*	@since	1.0
 */
final public class ShoppingListEditor extends GridPane implements INotebookLoadable
{

	private static final Logger LOGGER = LogManager.getLogger();

	private final ShoppingListBean thisValueBean;

	private Consumer<Node> loadSplit;
	private Consumer<Node> clearSplit;
	private BiConsumer<String, Node> loadTab;
	private Consumer<Node> clearTab;

	//	an observable property that the caller can monitor when a new item is being created
	private SimpleObjectProperty<ShoppingListBean> newBean;
	private final boolean addingItem;

	//	an observable property that the caller can monitor when an item is deleted
	private SimpleObjectProperty<Object> deletedBean;

//	private boolean addingBean;

    // use to  monitor if the current product's category is plant-like.  Is bound TO but updated explicitly
	private final SimpleBooleanProperty isPlantLike = new SimpleBooleanProperty(this, "Plant like", false);

    @FXML
	private ResourceBundle resources;

	@FXML
	private Button btnSave;
	@FXML
	private Button btnDelete;
    
    @FXML
    private ProductCategoryCombo cmbProductCategory;
    @FXML
    private ProductBrandCombo cmbProductBrand;
    
	@FXML
	private Label lblPlantSpecies;
    @FXML
    private PlantSpeciesCombo cmbPlantSpecies;
	@FXML
	private Label lblPlantVariety;
    @FXML
    private PlantVarietyCombo cmbPlantVariety;

	@FXML
	private Label lblProductName;
    @FXML
	private ProductNameCombo cmbProductName;
	@FXML
	private Label lblProductDetail;
	@FXML
	private ProductDetailCombo cmbProductDetail;
    
	@FXML
	private TextField fldNonspecificItem;

	@FXML
	private TableView<CommentBean> tbvComment;
	@FXML
	private TableColumn<CommentBean, CommentBean> tbvCommentDate;
	@FXML
	private TableColumn<CommentBean, String> tbvCommentText;

	ShoppingListEditor()
	{
		this(null);
	}

	ShoppingListEditor(ShoppingListBean initialVal)
	{
		this.thisValueBean = (initialVal != null ? initialVal : new ShoppingListBean());
        this.thisValueBean.setSaveRequired(true);   //  different behaviour of ShoppingListBean
		this.addingItem = (initialVal == null);
        LOGGER.debug("constructor: after beans, before FXML loader");
		FXMLLoader fxmlLoader = new FXMLLoader(getClass().getResource("/fxml/ShoppingListEditor.fxml"),
			ResourceBundle.getBundle("notebook") );
		fxmlLoader.setRoot(this);
		fxmlLoader.setController(this);
		try {
			fxmlLoader.load();	// NB initialize() is called from in here
		} catch (IOException exception) {
			throw new RuntimeException(exception);
		}
	}

	/*
	*	Initialises the controller class.
	*/
	@FXML
	private void initialize()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("initialize()");
//		tbvCommentDate.setCellValueFactory(e-> new SimpleObjectProperty<>( e.getValue()));
//		tbvCommentDate.setCellFactory(x -> new EditorCommentDateTableCell(resources));
//		tbvCommentText.setCellValueFactory(e->e.getValue().commentProperty());
//		tbvCommentText.setCellFactory(x -> new EditorCommentTextTableCell(resources));  //  2.4.0
//		tbvComment.setColumnResizePolicy(NotebookResizer.using(tbvComment));
//		tbvComment.getItems().setAll(thisValueBean.getComments());
//		CommentBean cb_new = new CommentBean(this.thisValueBean);
//
//		ChangeListener<String> commentAdder = new EditorCommentAdder<>(thisValueBean, tbvComment);
//		cb_new.commentProperty().addListener(commentAdder);
//		tbvComment.getItems().add(cb_new);
//		CommentBean cb_one = tbvComment.getItems().get(0);
//		Text t = new Text(cb_one.getDate().format(DateTimeFormatter.ofLocalizedDate(FormatStyle.SHORT)));
//		double wid = t.getLayoutBounds().getWidth();
//		tbvCommentDate.setPrefWidth(wid+10);

		final EditorCommentTableHandler<ShoppingListBean> cth = new EditorCommentTableHandler<>(resources);
		cth.constructCommentTable(tbvComment, tbvCommentDate, tbvCommentText, thisValueBean);

		thisValueBean.setSaveRequired(true);
        
        cmbProductCategory.valueProperty().bindBidirectional(thisValueBean.productCategoryProperty());
		cmbProductCategory.setDisable(!addingItem );
		isPlantLike.set(thisValueBean.hasProductCategory() && thisValueBean.productCategoryProperty().get().isPlantLike());

        cmbProductBrand.valueProperty().bindBidirectional(thisValueBean.productBrandProperty());
        
		cmbPlantSpecies.valueProperty().bindBidirectional(thisValueBean.plantSpeciesProperty());
		cmbPlantSpecies.setDisable(!addingItem);
        cmbPlantSpecies.visibleProperty().bind(isPlantLike);
        lblPlantSpecies.visibleProperty().bind(cmbPlantSpecies.visibleProperty());

        cmbPlantVariety.valueProperty().bindBidirectional(thisValueBean.plantVarietyProperty());
		cmbPlantVariety.plantSpeciesProperty().bind(cmbPlantSpecies.valueProperty());
		cmbPlantVariety.setDisable(!addingItem);
        cmbPlantVariety.visibleProperty().bind(isPlantLike);
        lblPlantVariety.visibleProperty().bind(cmbPlantVariety.visibleProperty());
        
        cmbProductName.productCategoryProperty().bind(thisValueBean.productCategoryProperty());
        LOGGER.debug("thisValueBean: {}", thisValueBean);
        LOGGER.debug("thisValueBean.nameProperty: {}", thisValueBean.nameProperty());
        cmbProductName.valueProperty().bindBidirectional(thisValueBean.nameProperty());
		cmbProductName.visibleProperty().bind(isPlantLike.not());
		cmbProductName.setDisable(!addingItem);
		lblProductName.visibleProperty().bind(cmbProductName.visibleProperty());
        
        cmbProductDetail.productCategoryProperty().bind(thisValueBean.productCategoryProperty());
        cmbProductDetail.productNameProperty().bind(thisValueBean.nameProperty());
        cmbProductDetail.valueProperty().bindBidirectional(thisValueBean.nameDetailProperty());
		cmbProductDetail.visibleProperty().bind(isPlantLike.not());
		cmbProductDetail.setDisable(!addingItem);
		lblProductDetail.visibleProperty().bind(cmbProductDetail.visibleProperty());

		fldNonspecificItem.textProperty().bindBidirectional(thisValueBean.nonspecificItemProperty());
		fldNonspecificItem.setTextFormatter(MaxLengthTextFormatter.getFormatter(255));
        fldNonspecificItem.setDisable(thisValueBean.hasProductCategory());

        /*
            For a new item, set category and non-specific both editable.
            If the item has a product category, then the non-specific value may not be set.
            Equally, if the non-specific value is set, the category may not be set.
            Plant variety may only be set when species is known.
            Name detail may only be set if the name is set.
        */
		cmbProductCategory.disableProperty().bind(thisValueBean.nonspecificItemProperty().isNotNull().and(thisValueBean.nonspecificItemProperty().isNotEmpty()));
		cmbProductBrand.disableProperty().bind(thisValueBean.hasProductCategoryProperty().not());
        cmbPlantSpecies.disableProperty().bind(isPlantLike.not());
        cmbPlantVariety.disableProperty().bind(isPlantLike.not().or(thisValueBean.hasPlantSpeciesProperty().not()));
        cmbProductName.disableProperty().bind(thisValueBean.hasProductCategoryProperty().not().or(isPlantLike));
        cmbProductDetail.disableProperty().bind(thisValueBean.hasProductCategoryProperty().not().or(isPlantLike).or(thisValueBean.nameProperty().isEmpty()));
        fldNonspecificItem.disableProperty().bind(thisValueBean.hasProductCategoryProperty());
        
		try
		{
			btnDelete.setDisable(addingItem || !(this.thisValueBean.canDelete()));
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}
        
        //  Can only save if the non-specific value is set OR the category and name/species is set
        //  Remember, setting plant species ALSO sets the Name property
        btnSave.disableProperty().bind( ((thisValueBean.hasProductCategoryProperty().and(thisValueBean.nameProperty().isNotEmpty()))
                                            .or(thisValueBean.nonspecificItemProperty().isNotNull().and(thisValueBean.nonspecificItemProperty().isNotEmpty()))
                                        ).not() );

		Platform.runLater(() -> cmbProductCategory.requestFocus());

		LOGGER.traceExit(log4jEntryMsg);
	}	//	initialize()

	@Override
	public void setLoadSplit(Consumer<Node> code)
	{
		loadSplit = code;
	}

	@Override
	public void setClearSplit(Consumer<Node> code)
	{
		clearSplit = code;
	}

	@Override
	public void setLoadTab(BiConsumer<String, Node> code)
	{
		loadTab = code;
	}

	@Override
	public void setClearTab(Consumer<Node> code)
	{
		clearTab = code;
	}

	@Override
	public void clearUpOnClose(Event e)
	{
		if (isCancelDenied())
		{
			e.consume();
			return;
		}

		thisValueBean.cancelEdit();
	}

    @FXML
    private void parentCategoryOnAction(ActionEvent event)
    {
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("parentCategoryOnAction()");
        boolean nullCat = (cmbProductCategory.valueProperty() == null || cmbProductCategory.valueProperty().get() == null);
        boolean plantLike = !nullCat && cmbProductCategory.valueProperty().get().isPlantLike();

        isPlantLike.set(plantLike);
        
        LOGGER.traceExit(log4jEntryMsg);
    }
    
	@FXML
	private void btnCancelOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnCancelOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

		if (isCancelDenied())
		{
			return;
		}

		thisValueBean.cancelEdit();
		clearTab.accept(this);
		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnCancelOnAction()

	/**
	 * Check if user really wants to quit if changes have been made
	 *
	 * @return	true	user does NOT want to quit
	 */
	private boolean isCancelDenied()
	{
		if (thisValueBean.needSave())
		{
//			Alert checkQuit = new Alert(Alert.AlertType.CONFIRMATION, resources.getString("alert.confirmquit"), ButtonType.NO, ButtonType.YES);
//			Optional<ButtonType> result = checkQuit.showAndWait();
//			LOGGER.debug("after delete dialog: result:{}, result.get:{}",result, result.orElse(null));
//			if (result.isPresent() && result.get() == ButtonType.NO)
//			{
//				LOGGER.debug("after quit denied");
//				return true;
//			}
			final NotebookEditorCanceller<ShoppingListBean> cancelChecker = new NotebookEditorCanceller<>(resources);
			return cancelChecker.isCancelDenied(thisValueBean);
		}
		return false;
	}

	@FXML
	private void btnDeleteOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnDeleteOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

//		boolean canDelete = false;
//		try
//		{
//			canDelete = thisValueBean.canDelete();
//		} catch (GNDBException ex) {
//			PanicHandler.panic(ex);
//		}
//		if (!canDelete)
//		{
//			Alert checkDelete = new Alert(Alert.AlertType.INFORMATION, resources.getString("alert.cannotdelete"), ButtonType.OK);
//			Optional<ButtonType> result = checkDelete.showAndWait();
//			LOGGER.debug("item cannot be deleted");
//			return;
//		}
//		Alert checkDelete = new Alert(Alert.AlertType.CONFIRMATION, resources.getString("alert.confirmdelete"), ButtonType.NO, ButtonType.YES);
//		Optional<ButtonType> result = checkDelete.showAndWait();
//		LOGGER.debug("after delete dialog: result:{}, result.get:{}",result, result.get());
//		if (result.isPresent() && result.get() == ButtonType.YES)
//		{
//			LOGGER.debug("after delete confirmed");
//			try
//			{
//				thisValueBean.delete();
//			} catch (GNDBException ex) {
//				PanicHandler.panic(ex);
//			}
//			deletedBeanProperty().setValue(new Object());
//			clearTab.accept(this);
//		}
		NotebookBeanDeleter<ShoppingListBean> deleterImpl = new NotebookBeanDeleter<>(resources);
		if (deleterImpl.deleteItemImpl(thisValueBean))
		{
			deletedBeanProperty().setValue(new Object());
			clearTab.accept(this);
		}

		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnDeleteOnAction()

	@FXML
	private void btnSaveOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnSaveOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}
		try
		{
			thisValueBean.save();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}
		newBeanProperty().setValue(thisValueBean);
		clearTab.accept(this);
		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnSaveOnAction()

	SimpleObjectProperty<ShoppingListBean> newBeanProperty()
	{
		if (newBean == null)
		{
			newBean = new SimpleObjectProperty<>();
		}
		return newBean;
	}

	SimpleObjectProperty<Object> deletedBeanProperty()
	{
		if (deletedBean == null)
		{
			deletedBean = new SimpleObjectProperty<>();
		}
		return deletedBean;
	}

}
