/*
 * Copyright (C) 2021-2023 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
    3.0.0	First version
    3.0.1	CommentAdder handling moved to separate class
    3.0.2	Check before Cancel if changes have been made
    3.0.4	Use new convenience class NotebookBeanDeleter for deletion.
    		Use new convenience class NotebookBeanCanceller to handle edit cancellation.
    		Use new convenience class EditorCommentTableHandler to handle Comment table construction.
    		Set focus on Title field.
    3.1.0	Use SafeDatePicker.
 */

package uk.co.gardennotebook;

import javafx.application.Platform;
import javafx.beans.property.SimpleObjectProperty;
//import javafx.beans.value.ChangeListener;
import javafx.event.ActionEvent;
import javafx.event.Event;
import javafx.fxml.FXML;
import javafx.fxml.FXMLLoader;
import javafx.scene.Node;
import javafx.scene.control.*;
import javafx.scene.layout.GridPane;
//import javafx.scene.text.Text;
import javafx.scene.web.HTMLEditor;
import javafx.scene.web.HTMLEditorSkin;
import javafx.scene.web.WebView;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.CommentBean;
import uk.co.gardennotebook.fxbean.ReviewBean;
import uk.co.gardennotebook.spi.GNDBException;

import java.io.IOException;
import java.time.Year;
//import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
//import java.time.format.FormatStyle;
//import java.util.Optional;
import java.util.ResourceBundle;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
	*	Controller class for create/update of Review items
	*
	*	@author Andy Gegg
	*	@version	3.1.0
	*	@since	3.0.0
 */
final public class ReviewEditor extends GridPane implements INotebookLoadable
{

	private static final Logger LOGGER = LogManager.getLogger();

	private final ReviewBean thisValueBean;

	private Consumer<Node> loadSplit;
	private Consumer<Node> clearSplit;
	private BiConsumer<String, Node> loadTab;
	private Consumer<Node> clearTab;

	//	an observable property that the caller can monitor when a new item is being created
	private SimpleObjectProperty<ReviewBean> newBean;
	private final boolean addingItem;

	//	an observable property that the caller can monitor when an item is deleted
	private SimpleObjectProperty<Object> deletedBean;

	private boolean addingBean;

	@FXML
	private ResourceBundle resources;

	@FXML
	private Button btnSave;
	@FXML
	private Button btnDelete;
	@FXML
	private SafeDatePicker dtpDate;
	@FXML
	private Spinner<Year> numYearInReview;
	@FXML
	private SafeDatePicker dtpCoverFrom;
	@FXML
	private SafeDatePicker dtpCoverTo;
	@FXML
	private TextField fldTitle;
	@FXML
//	private TextArea fldDescription;
	private HTMLEditor fldDescription;

	@FXML
	private TableView<CommentBean> tbvComment;
	@FXML
	private TableColumn<CommentBean, CommentBean> tbvCommentDate;
	@FXML
	private TableColumn<CommentBean, String> tbvCommentText;

	ReviewEditor()
	{
		this(null, null);
	}

	ReviewEditor(Year yearInReview)
	{
		this(yearInReview, null);
	}

	ReviewEditor(ReviewBean review)
	{
		this(null, review);
	}

	ReviewEditor(Year yearInReview, ReviewBean initialVal)
	{
		this.thisValueBean = (initialVal != null ? initialVal : new ReviewBean());
		this.addingItem = (initialVal == null);
		if (initialVal == null && yearInReview != null)
		{
			this.thisValueBean.setYearInReview(yearInReview);
		}
		FXMLLoader fxmlLoader = new FXMLLoader(getClass().getResource("/fxml/ReviewEditor.fxml"),
			ResourceBundle.getBundle("notebook") );
		fxmlLoader.setRoot(this);
		fxmlLoader.setController(this);
		try {
			fxmlLoader.load();	// NB initialize() is called from in here
		} catch (IOException exception) {
			throw new RuntimeException(exception);
		}
	}

	/*
	*	Initialises the controller class.
	*/
	@FXML
	private void initialize()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("initialize()");

		final EditorCommentTableHandler<ReviewBean> cth = new EditorCommentTableHandler<>(resources);
		cth.constructCommentTable(tbvComment, tbvCommentDate, tbvCommentText, thisValueBean);

		thisValueBean.setSaveRequired(true);

		dtpDate.setConverter(null);	// resets to default converter
		dtpDate.valueProperty().bindBidirectional(thisValueBean.dateProperty());
		dtpDate.setMandatory();
//        //  if user tabs out after editing, make sure the DatePicker updates
//        dtpDate.getEditor().focusedProperty().addListener((obj, wasFocused, isFocused)->{
//            if (wasFocused && !isFocused)
//            {
//                try
//                {
//                    dtpDate.setValue(dtpDate.getConverter().fromString(dtpDate.getEditor().getText()));
//                } catch (DateTimeParseException e) {
//                    dtpDate.getEditor().setText(dtpDate.getConverter().toString(dtpDate.getValue()));
//                }
//            }
//        });

		SpinnerValueFactory<Year> yearSpinnerValueFactory = new SpinnerValueFactory<Year>()
		{
			@Override
			public void decrement(int i)
			{
				if (getValue() != null)
				{
					setValue(getValue().minusYears(i));
				}
				else
				{
					setValue(Year.now());
				}
			}

			@Override
			public void increment(int i)
			{
				if (getValue() != null)
				{
					setValue(getValue().plusYears(i));
				}
				else
				{
					setValue(Year.now());
				}
			}
		};

		yearSpinnerValueFactory.setValue(thisValueBean.getYearInReview());
		numYearInReview.setValueFactory(yearSpinnerValueFactory);
		thisValueBean.yearInReviewProperty().bind(numYearInReview.valueProperty());	// NB bidirectional binding NOT allowed on Spinners!

		dtpCoverFrom.setConverter(null);	// resets to default converter
		dtpCoverFrom.valueProperty().bindBidirectional(thisValueBean.coverFromProperty());
//		//  if user tabs out after editing, make sure the DatePicker updates
//		dtpCoverFrom.getEditor().focusedProperty().addListener((obj, wasFocused, isFocused)->{
//			if (wasFocused && !isFocused)
//			{
//				try
//				{
//					dtpCoverFrom.setValue(dtpCoverFrom.getConverter().fromString(dtpCoverFrom.getEditor().getText()));
//				} catch (DateTimeParseException e) {
//					dtpCoverFrom.getEditor().setText(dtpCoverFrom.getConverter().toString(dtpCoverFrom.getValue()));
//				}
//			}
//		});

		dtpCoverTo.setConverter(null);	// resets to default converter
		dtpCoverTo.valueProperty().bindBidirectional(thisValueBean.coverToProperty());
//		//  if user tabs out after editing, make sure the DatePicker updates
//		dtpCoverTo.getEditor().focusedProperty().addListener((obj, wasFocused, isFocused)->{
//			if (wasFocused && !isFocused)
//			{
//				try
//				{
//					dtpCoverTo.setValue(dtpCoverTo.getConverter().fromString(dtpCoverTo.getEditor().getText()));
//				} catch (DateTimeParseException e) {
//					dtpCoverTo.getEditor().setText(dtpCoverTo.getConverter().toString(dtpCoverTo.getValue()));
//				}
//			}
//		});

		fldTitle.textProperty().bindBidirectional(thisValueBean.titleProperty());
		fldTitle.setTextFormatter(MaxLengthTextFormatter.getFormatter(255));

		//	HTMLEditor does not have a text property which can be bound to a field in the bean
		fldDescription.setHtmlText(thisValueBean.getDescription());
		//	HTMLEditor does not generate lost focus events on exit
//		fldDescription.focusedProperty().addListener((obj, wasFocused, isFocused)->{
//			System.out.println("fldDescription.focusedProperty(): was: "+wasFocused+", is: "+isFocused);
//			if (wasFocused && !isFocused)
//			{
//				thisValueBean.setDescription(fldDescription.getHtmlText());
//				System.out.println("fldDescription: lost focus: summary: "+thisValueBean.getSummary());
//			}
//
//		});

//		btnSave.disableProperty().bind(dtpDate.valueProperty().isNull() );

		fldTitle.textProperty().addListener((obs, old, nval) -> {
			btnSave.setDisable((nval == null) || nval.isBlank());
		});
		dtpDate.valueProperty().addListener((obs, old, nval) -> {
			btnSave.setDisable(nval == null);
		});
		btnSave.setDisable((fldTitle.textProperty().get() == null) || fldTitle.textProperty().get().isBlank());

		try
		{
			btnDelete.setDisable(addingItem || !(this.thisValueBean.canDelete()));
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}

		Platform.runLater(() -> {fldTitle.requestFocus(); fldTitle.deselect();});

		LOGGER.traceExit(log4jEntryMsg);
	}	//	initialize()

	@Override
	public void setLoadSplit(Consumer<Node> code)
	{
		loadSplit = code;
	}

	@Override
	public void setClearSplit(Consumer<Node> code)
	{
		clearSplit = code;
	}

	@Override
	public void setLoadTab(BiConsumer<String, Node> code)
	{
		loadTab = code;
	}

	@Override
	public void setClearTab(Consumer<Node> code)
	{
		clearTab = code;
	}

	@Override
	public void clearUpOnClose(Event e)
	{
		if (isCancelDenied())
		{
			e.consume();
			return;
		}

		// need to explicitly unbind the spinner or thisValueBean.cancelEdit() faults: 'cannot set bound value' in thisValueBean.setValues()
		thisValueBean.yearInReviewProperty().unbind();
		thisValueBean.cancelEdit();
	}

	@FXML
	private void btnCancelOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnCancelOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

		if (isCancelDenied())
		{
			return;
		}

		// need to explicitly unbind the spinner or thisValueBean.cancelEdit() faults: 'cannot set bound value' in thisValueBean.setValues()
		thisValueBean.yearInReviewProperty().unbind();
		thisValueBean.cancelEdit();
		clearTab.accept(this);
		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnCancelOnAction()

	/**
	 * Check if user really wants to quit if changes have been made
	 *
	 * @return	true	user does NOT want to quit
	 */
	private boolean isCancelDenied()
	{
		//	the HTML text is NOT implemented as a property so cannot be bound to, hence the need to explicitly update before checking
		thisValueBean.setDescription(fldDescription.getHtmlText());
		if (thisValueBean.needSave())
		{
			final NotebookEditorCanceller<ReviewBean> cancelChecker = new NotebookEditorCanceller<>(resources);
			return cancelChecker.isCancelDenied(thisValueBean);
		}
		return false;
	}

	@FXML
	private void btnDeleteOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnDeleteOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

		NotebookBeanDeleter<ReviewBean> deleterImpl = new NotebookBeanDeleter<>(resources, ()->thisValueBean.yearInReviewProperty().unbind());
		if (deleterImpl.deleteItemImpl(thisValueBean))
		{
			deletedBeanProperty().setValue(new Object());
			clearTab.accept(this);
		}

		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnDeleteOnAction()

	@FXML
	private void btnSaveOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnSaveOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}
		if (fldTitle.textProperty().isNull().get() || fldTitle.textProperty().get().isBlank() )
		{
			return;
		}

		thisValueBean.setDescription(fldDescription.getHtmlText());
		// need to explicitly unbind the spinner or thisValueBean.save() faults: 'cannot set bound value' in thisValueBean.setValues()
		thisValueBean.yearInReviewProperty().unbind();
		try
		{
			thisValueBean.save();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}
		newBeanProperty().setValue(thisValueBean);
		clearTab.accept(this);
		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnSaveOnAction()

	SimpleObjectProperty<ReviewBean> newBeanProperty()
	{
		if (newBean == null)
		{
			newBean = new SimpleObjectProperty<>();
		}
		return newBean;
	}

	SimpleObjectProperty<Object> deletedBeanProperty()
	{
		if (deletedBean == null)
		{
			deletedBean = new SimpleObjectProperty<>();
		}
		return deletedBean;
	}

}
