/*
 * Copyright (C) 2018, 2019, 2022 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	2.0.1   Remove nameDetail_2
    2.3.0   Add facility to specify Category of new Product
    2.4.0   Support tab-out on comments
    3.0.1	CommentAdder handling moved to separate class
    3.0.2	Check before Cancel if changes have been made
    3.0.4	Use new convenience class NotebookBeanDeleter for deletion.
    		Use new convenience class NotebookBeanCanceller to handle edit cancellation.
    		Use new convenience class EditorCommentTableHandler to handle Comment table construction.
    		Set focus on first field.
 */

package uk.co.gardennotebook;

import java.io.IOException;
//import java.util.Optional;
import java.util.ResourceBundle;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
//import java.time.format.DateTimeFormatter;
//import java.time.format.FormatStyle;

import javafx.application.Platform;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleObjectProperty;
//import javafx.beans.value.ChangeListener;
//import javafx.beans.value.ObservableValue;
import javafx.event.ActionEvent;
import javafx.event.Event;
import javafx.fxml.FXML;
import javafx.fxml.FXMLLoader;
import javafx.scene.Node;
//import javafx.scene.text.Text;
//import javafx.scene.control.Alert;
//import javafx.scene.control.ButtonType;
import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.TableColumn;
import javafx.scene.control.TableView;
import javafx.scene.control.TextField;
import uk.co.gardennotebook.fxbean.CommentBean;
import javafx.scene.layout.GridPane;
import uk.co.gardennotebook.fxbean.ProductBean;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.ProductCategoryBean;
import uk.co.gardennotebook.spi.GNDBException;

/**
	*	Controller class for create/update of Product items
	*
	*	@author Andy Gegg
	*	@version	3.0.4
	*	@since	1.0
 */
final public class ProductEditor extends GridPane implements INotebookLoadable
{

	private static final Logger LOGGER = LogManager.getLogger();

	private final ProductBean thisValueBean;

	private Consumer<Node> loadSplit;
	private Consumer<Node> clearSplit;
	private BiConsumer<String, Node> loadTab;
	private Consumer<Node> clearTab;

	//	an observable property that the caller can monitor when a new item is being created
	private SimpleObjectProperty<ProductBean> newBean;
	private final boolean addingItem;

	//	an observable property that the caller can monitor when an item is deleted
	private SimpleObjectProperty<Object> deletedBean;

//	private boolean addingBean;
	
    // use to  monitor if the current product's category is plant-like.  Is bound TO but updated explicitly
	private final SimpleBooleanProperty isPlantLike = new SimpleBooleanProperty(this, "Plant like", false);

	@FXML
	private ResourceBundle resources;

	@FXML
	private Button btnSave;
	@FXML
	private Button btnDelete;
	@FXML
	private ProductCategoryCombo parentProductCategory;
	@FXML
	private Label lblName;
	@FXML
	private TextField fldName;
	@FXML
	private Label lblPlantSpecies;
	@FXML
	private PlantSpeciesCombo parentPlantSpecies;
	@FXML
	private Label lblPlantVariety;
	@FXML
	private PlantVarietyCombo parentPlantVariety;
	@FXML
	private ProductBrandCombo parentProductBrand;
	@FXML
	private Label lblNameDetail_1;
	@FXML
	private TextField fldNameDetail_1;
//	@FXML
//	private Label lblNameDetail_2;
//	@FXML
//	private TextField fldNameDetail_2;
	@FXML
	private TextField fldDescription;

	@FXML
	private TableView<CommentBean> tbvComment;
	@FXML
	private TableColumn<CommentBean, CommentBean> tbvCommentDate;
	@FXML
	private TableColumn<CommentBean, String> tbvCommentText;

	ProductEditor()
	{
		this(null, null);
	}

	ProductEditor(ProductCategoryBean initialCat)
	{
		this(initialCat, null);
	}

	ProductEditor(ProductBean initialVal)
	{
		this(null, initialVal);
	}

	ProductEditor(ProductCategoryBean initialCat, ProductBean initialVal)
	{
		this.thisValueBean = (initialVal != null ? initialVal : new ProductBean());
		this.addingItem = (initialVal == null);
        if (addingItem && initialCat != null)
        {
            this.thisValueBean.setProductCategory(initialCat);
        }
		FXMLLoader fxmlLoader = new FXMLLoader(getClass().getResource("/fxml/ProductEditor.fxml"),
			ResourceBundle.getBundle("notebook") );
		fxmlLoader.setRoot(this);
		fxmlLoader.setController(this);
		try {
			fxmlLoader.load();	// NB initialize is called from in here
		} catch (IOException exception) {
			throw new RuntimeException(exception);
		}
	}

	/**
	* Initializes the controller class.
	*/
	@FXML
	private void initialize()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("initialize()");
//		tbvCommentDate.setCellValueFactory(e-> new SimpleObjectProperty<>( e.getValue()));
//		tbvCommentDate.setCellFactory(x -> new EditorCommentDateTableCell(resources));
//		tbvCommentText.setCellValueFactory(e->e.getValue().commentProperty());
//		tbvCommentText.setCellFactory(x -> new EditorCommentTextTableCell(resources));  //  2.4.0
//		tbvComment.setColumnResizePolicy(NotebookResizer.using(tbvComment));
//		tbvComment.getItems().setAll(thisValueBean.getComments());
//		CommentBean cb_new = new CommentBean(this.thisValueBean);
//
//		ChangeListener<String> commentAdder = new EditorCommentAdder<>(thisValueBean, tbvComment);
//		cb_new.commentProperty().addListener(commentAdder);
//		tbvComment.getItems().add(cb_new);
//		CommentBean cb_one = tbvComment.getItems().get(0);
//		Text t = new Text(cb_one.getDate().format(DateTimeFormatter.ofLocalizedDate(FormatStyle.SHORT)));
//		double wid = t.getLayoutBounds().getWidth();
//		tbvCommentDate.setPrefWidth(wid+10);

		final EditorCommentTableHandler<ProductBean> cth = new EditorCommentTableHandler<>(resources);
		cth.constructCommentTable(tbvComment, tbvCommentDate, tbvCommentText, thisValueBean);

		thisValueBean.setSaveRequired(true);
		
		parentProductCategory.valueProperty().bindBidirectional(thisValueBean.productCategoryProperty());
		parentProductCategory.setDisable(!addingItem);
		isPlantLike.set(thisValueBean.productCategoryProperty().get().isPlantLike());
		
		fldName.textProperty().bindBidirectional(thisValueBean.nameProperty());
		fldName.setTextFormatter(MaxLengthTextFormatter.getFormatter(255));
		fldName.visibleProperty().bind(isPlantLike.not());
		fldName.setDisable(!addingItem);
		lblName.visibleProperty().bind(fldName.visibleProperty());
		parentPlantSpecies.valueProperty().bindBidirectional(thisValueBean.plantSpeciesProperty());
		parentPlantSpecies.visibleProperty().bind(isPlantLike);
		parentPlantSpecies.setDisable(!addingItem);
		lblPlantSpecies.visibleProperty().bind(parentPlantSpecies.visibleProperty());

		
		fldNameDetail_1.textProperty().bindBidirectional(thisValueBean.nameDetail_1Property());
		fldNameDetail_1.setTextFormatter(MaxLengthTextFormatter.getFormatter(255));
		fldNameDetail_1.visibleProperty().bind(isPlantLike.not());
		fldNameDetail_1.setDisable(!addingItem);
		lblNameDetail_1.visibleProperty().bind(fldNameDetail_1.visibleProperty());
		parentPlantVariety.valueProperty().bindBidirectional(thisValueBean.plantVarietyProperty());
		parentPlantVariety.plantSpeciesProperty().bind(parentPlantSpecies.valueProperty());
		parentPlantVariety.visibleProperty().bind(isPlantLike);
		parentPlantVariety.setDisable(!addingItem);
		lblPlantVariety.visibleProperty().bind(parentPlantVariety.visibleProperty());

//		fldNameDetail_2.textProperty().bindBidirectional(thisValueBean.nameDetail_2Property());
//		fldNameDetail_2.setTextFormatter(MaxLengthTextFormatter.getFormatter(255));
//		fldNameDetail_2.visibleProperty().bind(isPlantLike.not());
//		fldNameDetail_2.setDisable(!addingItem);
//		lblNameDetail_2.visibleProperty().bind(fldNameDetail_1.visibleProperty());

		parentProductBrand.valueProperty().bindBidirectional(thisValueBean.productBrandProperty());
		parentProductBrand.setDisable(!addingItem);

		fldDescription.textProperty().bindBidirectional(thisValueBean.descriptionProperty());
		fldDescription.setTextFormatter(MaxLengthTextFormatter.getFormatter(255));
		btnSave.disableProperty().bind(parentProductCategory.valueProperty().isNull()
			.or(fldName.textProperty().isNull().or(fldName.textProperty().isEmpty())));
		try {
			btnDelete.setDisable(addingItem || !(this.thisValueBean.canDelete()));
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}

		Platform.runLater(() -> {fldName.requestFocus(); fldName.deselect();});

		LOGGER.traceExit(log4jEntryMsg);
	}	//	initialize()

	@Override
	public void setLoadSplit(Consumer<Node> code)
	{
		loadSplit = code;
	}

	@Override
	public void setClearSplit(Consumer<Node> code)
	{
		clearSplit = code;
	}

	@Override
	public void setLoadTab(BiConsumer<String, Node> code)
	{
		loadTab = code;
	}

	@Override
	public void setClearTab(Consumer<Node> code)
	{
		clearTab = code;
	}

	@Override
	public void clearUpOnClose(Event e)
	{
		if (isCancelDenied())
		{
			e.consume();
			return;
		}

		thisValueBean.cancelEdit();
	}

	@FXML
	private void btnCancelOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnCancelOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

		if (isCancelDenied())
		{
			return;
		}

		thisValueBean.cancelEdit();
		clearTab.accept(this);
		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnCancelOnAction()

	/**
	 * Check if user really wants to quit if changes have been made
	 *
	 * @return	true	user does NOT want to quit
	 */
	private boolean isCancelDenied()
	{
		if (thisValueBean.needSave())
		{
//			Alert checkQuit = new Alert(Alert.AlertType.CONFIRMATION, resources.getString("alert.confirmquit"), ButtonType.NO, ButtonType.YES);
//			Optional<ButtonType> result = checkQuit.showAndWait();
//			LOGGER.debug("after delete dialog: result:{}, result.get:{}",result, result.orElse(null));
//			if (result.isPresent() && result.get() == ButtonType.NO)
//			{
//				LOGGER.debug("after quit denied");
//				return true;
//			}
			final NotebookEditorCanceller<ProductBean> cancelChecker = new NotebookEditorCanceller<>(resources);
			return cancelChecker.isCancelDenied(thisValueBean);
		}
		return false;
	}

	@FXML
	private void btnDeleteOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnDeleteOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}
		
//		boolean canDelete = false;
//		try {
//			canDelete = thisValueBean.canDelete();
//		} catch (GNDBException ex) {
//			PanicHandler.panic(ex);
//		}
//		if (!canDelete)
//		{
//			Alert checkDelete = new Alert(Alert.AlertType.INFORMATION, resources.getString("alert.cannotdelete"), ButtonType.OK);
//			Optional<ButtonType> result = checkDelete.showAndWait();
//			LOGGER.debug("item cannot be deleted");
//			return;
//		}
//		Alert checkDelete = new Alert(Alert.AlertType.CONFIRMATION, resources.getString("alert.confirmdelete"), ButtonType.NO, ButtonType.YES);
//		Optional<ButtonType> result = checkDelete.showAndWait();
//		LOGGER.debug("after delete dialog: result:{}, result.get:{}",result, result.get());
//		if (result.isPresent() && result.get() == ButtonType.YES)
//		{
//			LOGGER.debug("after delete confirmed");
//			try {
//				thisValueBean.delete();
//			} catch (GNDBException ex) {
//				PanicHandler.panic(ex);
//			}
//			deletedBeanProperty().setValue(new Object());
//			clearTab.accept(this);
//		}
		NotebookBeanDeleter<ProductBean> deleterImpl = new NotebookBeanDeleter<>(resources);
		if (deleterImpl.deleteItemImpl(thisValueBean))
		{
			deletedBeanProperty().setValue(new Object());
			clearTab.accept(this);
		}
		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnDeleteOnAction()

	@FXML
	private void btnSaveOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnSaveOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}
		try {
			thisValueBean.save();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}
		newBeanProperty().setValue(thisValueBean);
		clearTab.accept(this);
		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnSaveOnAction()
	
	@FXML
	private void parentProductCategoryOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("parentProductCategoryOnAction()");
		if (parentProductCategory.valueProperty().get() == null)
		{
			isPlantLike.set(false);
			return;
		}
		isPlantLike.set(parentProductCategory.valueProperty().get().isPlantLike());

	}

	SimpleObjectProperty<ProductBean> newBeanProperty()
	{
		if (newBean == null)
		{
			newBean = new SimpleObjectProperty<>();
		}
		return newBean;
	}

	SimpleObjectProperty<Object> deletedBeanProperty()
	{
		if (deletedBean == null)
		{
			deletedBean = new SimpleObjectProperty<>();
		}
		return deletedBean;
	}

}
