/*
 * Copyright (C) 2021, 2022 Andrew Gegg
 *
 *	This file is part of the Garden Notebook application
 *
 * The Garden Notebook application is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 */

/*
	Change log
	3.0.0	First version
    3.0.1	CommentAdder handling moved to separate class
    3.0.2	Check before Cancel if changes have been made
    3.0.4	Use new convenience class NotebookBeanDeleter for deletion.
    		Use new convenience class NotebookBeanCanceller to handle edit cancellation.
    		Use new convenience class EditorCommentTableHandler to handle Comment table construction.
    		Set focus on first field.
 */

package uk.co.gardennotebook;

import javafx.application.Platform;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.event.ActionEvent;
import javafx.event.Event;
import javafx.fxml.FXML;
import javafx.fxml.FXMLLoader;
import javafx.scene.Node;
import javafx.scene.control.*;
import javafx.scene.layout.GridPane;
//import javafx.scene.text.Text;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.CommentBean;
import uk.co.gardennotebook.fxbean.LocationBean;
import uk.co.gardennotebook.spi.GNDBException;
//import uk.co.gardennotebook.spi.ILocation;
//import uk.co.gardennotebook.spi.TrugServer;

import java.io.IOException;
//import java.time.format.DateTimeFormatter;
//import java.time.format.FormatStyle;
import java.util.List;
//import java.util.Optional;
import java.util.ResourceBundle;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
	*	Controller class for create/update of \Location items
	*
	*	@author Andy Gegg
	*	@version	3.0.4
	*	@since	3.0.0
 */
final public class LocationEditor extends GridPane implements INotebookLoadable
{

	private static final Logger LOGGER = LogManager.getLogger();

	private final LocationBean thisValueBean;

	private Consumer<Node> loadSplit;
	private Consumer<Node> clearSplit;
	private BiConsumer<String, Node> loadTab;
	private Consumer<Node> clearTab;

	//	an observable property that the caller can monitor when a new item is being created
	private SimpleObjectProperty<LocationBean> newBean;
	private final boolean addingItem;

	//	an observable property that the caller can monitor when an item is deleted
	private SimpleObjectProperty<Object> deletedBean;

//	private boolean addingBean;

	@FXML
	private ResourceBundle resources;

	@FXML
	private Button btnSave;
	@FXML
	private Button btnDelete;
	@FXML
	private LocationCombo parentLocation;
	@FXML
	private TextField fldName;
	@FXML
	private TextField fldDescription;
	@FXML
	private CheckBox chkUnderCover;
	@FXML
	private TextField fldGeometry;

	@FXML
	private TableView<CommentBean> tbvComment;
	@FXML
	private TableColumn<CommentBean, CommentBean> tbvCommentDate;
	@FXML
	private TableColumn<CommentBean, String> tbvCommentText;

	LocationEditor()
	{
		this(null);
	}

	LocationEditor(LocationBean initialVal)
	{
		this(initialVal, null);
	}

	/**
	 *	Can only specify the parent for a new Location
	 *
	 * @param initialVal	the Location to edit.
	 * @param parent	the parent of a new Location.  Ignored in initialVal is non-null.
	 */
	LocationEditor(LocationBean initialVal, LocationBean parent)
	{
		if (initialVal == null)
		{	// new value
			if (parent == null)
			{
				this.thisValueBean = new LocationBean();
			}
			else
			{
				this.thisValueBean = new LocationBean();
				this.thisValueBean.setParentLocation(parent);
			}
		}
		else
		{
			this.thisValueBean = initialVal;
		}

		this.addingItem = (initialVal == null);
		FXMLLoader fxmlLoader = new FXMLLoader(getClass().getResource("/fxml/LocationEditor.fxml"),
				ResourceBundle.getBundle("notebook") );
		fxmlLoader.setRoot(this);
		fxmlLoader.setController(this);
		try {
			fxmlLoader.load();	// NB initialize() is called from in here
		} catch (IOException exception) {
			throw new RuntimeException(exception);
		}
	}

	/*
	*	Initialises the controller class.
	*/
	@FXML
	private void initialize()
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("initialize(): thisValueBean: {}", thisValueBean);
//		tbvCommentDate.setCellValueFactory(e-> new SimpleObjectProperty<>( e.getValue()));
//		tbvCommentDate.setCellFactory(x -> new EditorCommentDateTableCell(resources));
//		tbvCommentText.setCellValueFactory(e->e.getValue().commentProperty());
//		tbvCommentText.setCellFactory(x -> new EditorCommentTextTableCell(resources));  //  2.4.0
//		tbvComment.setColumnResizePolicy(NotebookResizer.using(tbvComment));
//		tbvComment.getItems().setAll(thisValueBean.getComments());
//		CommentBean cb_new = new CommentBean(this.thisValueBean);
//
//		ChangeListener<String> commentAdder = new EditorCommentAdder<>(thisValueBean, tbvComment);
//		cb_new.commentProperty().addListener(commentAdder);
//		tbvComment.getItems().add(cb_new);
//		CommentBean cb_one = tbvComment.getItems().get(0);
//		Text t = new Text(cb_one.getDate().format(DateTimeFormatter.ofLocalizedDate(FormatStyle.SHORT)));
//		double wid = t.getLayoutBounds().getWidth();
//		tbvCommentDate.setPrefWidth(wid+10);

		final EditorCommentTableHandler<LocationBean> cth = new EditorCommentTableHandler<>(resources);
		cth.constructCommentTable(tbvComment, tbvCommentDate, tbvCommentText, thisValueBean);

		thisValueBean.setSaveRequired(true);

		parentLocation.valueProperty().bindBidirectional(thisValueBean.parentLocationProperty());

		fldName.textProperty().bindBidirectional(thisValueBean.nameProperty());
		fldName.setTextFormatter(MaxLengthTextFormatter.getFormatter(255));

        fldName.focusedProperty().addListener( new ChangeListener<>() {
            @Override
            public void changed(ObservableValue obs, Boolean oldVal, Boolean newVal)
            {
                if (oldVal && !newVal)
                {
                    LOGGER.debug("fldName lost focus");
                    thisValueBean.checkForDuplicateName(fldName.getText());
                }
            }
        });

		fldDescription.textProperty().bindBidirectional(thisValueBean.descriptionProperty());
		fldDescription.setTextFormatter(MaxLengthTextFormatter.getFormatter(255));

		chkUnderCover.selectedProperty().bindBidirectional(thisValueBean.underCoverProperty());

		fldGeometry.textProperty().bindBidirectional(thisValueBean.geometryProperty());

		btnSave.disableProperty().bind(fldName.textProperty().isNull().or(fldName.textProperty().isEmpty()).or(thisValueBean.duplicateNameProperty())); //  2.2.0
		try
		{
			btnDelete.setDisable(addingItem || !(this.thisValueBean.canDelete()));
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}

		try
		{
			List<LocationBean> excludeLocs = LocationBean.fetchDescendantLocations(this.thisValueBean);
			LOGGER.debug("exclusion list: {}", excludeLocs);
			parentLocation.excludeLocations(excludeLocs);
			parentLocation.excludeLocations(thisValueBean);
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}

		Platform.runLater(() -> {fldName.requestFocus(); fldName.deselect();});

		LOGGER.traceExit(log4jEntryMsg);
	}	//	initialize()

	@Override
	public void setLoadSplit(Consumer<Node> code)
	{
		loadSplit = code;
	}

	@Override
	public void setClearSplit(Consumer<Node> code)
	{
		clearSplit = code;
	}

	@Override
	public void setLoadTab(BiConsumer<String, Node> code)
	{
		loadTab = code;
	}

	@Override
	public void setClearTab(Consumer<Node> code)
	{
		clearTab = code;
	}

	@Override
	public void clearUpOnClose(Event e)
	{
		if (isCancelDenied())
		{
			e.consume();
			return;
		}

		thisValueBean.cancelEdit();
	}

	@FXML
	private void btnCancelOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnCancelOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

		if (isCancelDenied())
		{
			return;
		}

		thisValueBean.cancelEdit();
		clearTab.accept(this);
		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnCancelOnAction()

	/**
	 * Check if user really wants to quit if changes have been made
	 *
	 * @return	true	user does NOT want to quit
	 */
	private boolean isCancelDenied()
	{
		if (thisValueBean.needSave())
		{
//			Alert checkQuit = new Alert(Alert.AlertType.CONFIRMATION, resources.getString("alert.confirmquit"), ButtonType.NO, ButtonType.YES);
//			Optional<ButtonType> result = checkQuit.showAndWait();
//			LOGGER.debug("after delete dialog: result:{}, result.get:{}",result, result.orElse(null));
//			if (result.isPresent() && result.get() == ButtonType.NO)
//			{
//				LOGGER.debug("after quit denied");
//				return true;
//			}
			NotebookEditorCanceller<LocationBean> cancelChecker = new NotebookEditorCanceller<>(resources);
			return cancelChecker.isCancelDenied(thisValueBean);
		}
		return false;
	}

	@FXML
	private void btnDeleteOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnDeleteOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}

//		boolean canDelete = false;
//		try
//		{
//			canDelete = thisValueBean.canDelete();
//		} catch (GNDBException ex) {
//			PanicHandler.panic(ex);
//		}
//		if (!canDelete)
//		{
//			Alert checkDelete = new Alert(Alert.AlertType.INFORMATION, resources.getString("alert.cannotdelete"), ButtonType.OK);
//			Optional<ButtonType> result = checkDelete.showAndWait();
//			LOGGER.debug("item cannot be deleted");
//			return;
//		}
//		Alert checkDelete = new Alert(Alert.AlertType.CONFIRMATION, resources.getString("alert.confirmdelete"), ButtonType.NO, ButtonType.YES);
//		Optional<ButtonType> result = checkDelete.showAndWait();
//		LOGGER.debug("after delete dialog: result:{}, result.get:{}",result, result.get());
//		if (result.isPresent() && result.get() == ButtonType.YES)
//		{
//			LOGGER.debug("after delete confirmed");
//			try
//			{
//				thisValueBean.delete();
//			} catch (GNDBException ex) {
//				PanicHandler.panic(ex);
//			}
//			deletedBeanProperty().setValue(new Object());
//			clearTab.accept(this);
//		}
		NotebookBeanDeleter<LocationBean> deleterImpl = new NotebookBeanDeleter<>(resources);
		if (deleterImpl.deleteItemImpl(thisValueBean))
		{
			deletedBeanProperty().setValue(new Object());
			clearTab.accept(this);
		}

		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnDeleteOnAction()

	@FXML
	private void btnSaveOnAction(ActionEvent event)
	{
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("btnSaveOnAction()");
		if (thisValueBean == null)
		{
			LOGGER.debug("thisValueBean is null");
			LOGGER.traceExit(log4jEntryMsg);
			return;
		}
		try
		{
			thisValueBean.save();
		} catch (GNDBException ex) {
			PanicHandler.panic(ex);
		}
		newBeanProperty().setValue(thisValueBean);
		clearTab.accept(this);
		LOGGER.traceExit(log4jEntryMsg);
	}	//	btnSaveOnAction()

	SimpleObjectProperty<LocationBean> newBeanProperty()
	{
		if (newBean == null)
		{
			newBean = new SimpleObjectProperty<>();
		}
		return newBean;
	}

	SimpleObjectProperty<Object> deletedBeanProperty()
	{
		if (deletedBean == null)
		{
			deletedBean = new SimpleObjectProperty<>();
		}
		return deletedBean;
	}

}
