/*
 *
 *  Copyright (C) 2022 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook;

import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.Property;
import javafx.beans.property.SimpleObjectProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.CommentBean;
import uk.co.gardennotebook.fxbean.INotebookBean;
import uk.co.gardennotebook.fxbean.PlantSpeciesBean;
import uk.co.gardennotebook.fxbean.PlantVarietyBean;
import uk.co.gardennotebook.fxbean.SaleItemBean;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.INotebookEntry;
import uk.co.gardennotebook.spi.NotebookEntryType;
import uk.co.gardennotebook.util.SimpleMoney;

import java.time.LocalDate;
import java.util.Objects;
import java.util.ResourceBundle;

/**
 *	A specific implementation of DiaryBean for SaleItem diary entries.
 *
 * @author Andy Gegg
 *	@version	3.0.5
 *	@since	3.0.5
 */
final class DiaryBeanSaleItem extends DiaryBean
{
    private static final Logger LOGGER = LogManager.getLogger();

    private final SaleItemBean saleItem;

    private Integer itemKey = 0;

    final ObjectProperty<LocalDate> dateProperty;

    private final ObjectProperty<String> mainTitleProperty;	//	the title - null here
    private final StringBinding mainTitleText;
    private final Property<PlantSpeciesBean> subTitleProperty;	//	the PlantSpecies
    private final StringBinding subTitleText;
    private final ObjectProperty<PlantVarietyBean> varietyProperty;	// the PlantVariety
    private final StringBinding varietyText;
    private final ObjectProperty<SimpleMoney> detailProperty;    //  the price at which sold
    private final StringBinding detailText;

    private final SimpleObjectProperty<ObservableList<CommentBean>> comments =
            new SimpleObjectProperty<>(this, "comments", FXCollections.emptyObservableList());

    private final ResourceBundle resources = ResourceBundle.getBundle("notebook");

    DiaryBeanSaleItem(SaleItemBean saleItemBean)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("DiaryBeanSaleItem: constructor(): SaleBean:{}", saleItemBean);

        saleItem = saleItemBean;
        itemKey = saleItemBean.getKey();

        dateProperty = saleItem.getSale().dateProperty();

        mainTitleProperty = null;
        mainTitleText = new StringBinding()
        {
            @Override
            protected String computeValue()
            {
                return "";
            }
        };
        subTitleProperty = saleItem.plantSpeciesProperty();
        subTitleText = new StringBinding()
        {
            {
                super.bind(subTitleProperty);
            }
            @Override
            protected String computeValue()
            {
                return saleItem.getPlantSpecies().getCommonName();
            }

            @Override
            public void dispose()
            {
                super.unbind(subTitleProperty);
            }
        };

        varietyProperty = saleItem.plantVarietyProperty();
        varietyText = new StringBinding()
        {
            {
                super.bind(varietyProperty);
            }
            @Override
            protected String computeValue()
            {
                if (saleItem.hasPlantVariety()) return saleItem.getPlantVariety().getCommonName();
                return "";
            }

            @Override
            public void dispose()
            {
                super.unbind(varietyProperty);
            }
        };

        detailProperty = saleItem.itemPriceProperty();
        detailText = new StringBinding()
        {
            {
                super.bind(detailProperty);
            }
            @Override
            protected String computeValue()
            {
                return saleItemBean.getItemPrice().toString();
            }

            @Override
            public void dispose()
            {
                super.unbind(detailProperty);
            }
        };

        LOGGER.traceExit(log4jEntryMsg);
    }

    @Override
    INotebookBean getItem()
    {
        return saleItem;
    }

    @Override
    INotebookEntry getBaseItem()
    {
        return saleItem.get().orElse(null);
    }

    @Override
    Integer getKey()
    {
        return itemKey;
    }

    Integer getSaleKey()
    {
        return saleItem.getSale().getKey();
    }

    @Override
    NotebookEntryType getItemType()
    {
        return NotebookEntryType.SALEITEM;
    }

    @Override
    boolean canDelete() throws GNDBException
    {
        return saleItem.canDelete();
    }

    /**
     * The date of this diary entry
     *
     * @return	the (value of) the dateProperty
     */
    @Override
    LocalDate getDate()
    {
        return dateProperty.getValue();
    }

    @Override
    void setDate(LocalDate date)
    {
        dateProperty().setValue(date);
    }

    @Override
    ObjectProperty<LocalDate> dateProperty()
    {
        return dateProperty;
    }

    /**
     * The title for this Diary entry - here, a null String (to group under the Sale)
     *
     * @return	the (value of) the mainTitleProperty
     */
    @Override
    StringBinding getMainTitleText()
    {
        return mainTitleText;
    }

    @Override
    ObjectProperty<String> mainTitleProperty()
    {
        return mainTitleProperty;
    }

    /**
     * The first detail for this Diary entry - here, the PlantSpecies
     *
     * @return	the (value of) the subTitleProperty
     */
    @Override
    StringBinding getSubTitleText()
    {
        return subTitleText;
    }

    @Override
    Property<PlantSpeciesBean> subTitleProperty()
    {
        return subTitleProperty;
    }

    /**
     * The second detail for this Diary entry - here, the PlantVariety
     *
     * @return	the (value of) the varietyProperty
     */
    @Override
    StringBinding getVarietyText()
    {
        return varietyText;
    }

    @Override
    ObjectProperty<PlantVarietyBean> varietyProperty()
    {
        return varietyProperty;
    }

    /**
     * The third detail for this Diary entry - here, the item price
     *
     * @return	the (value of) the detailProperty
     */
    @Override
    StringBinding getDetailText()
    {
        return detailText;
    }

    @Override
    ObjectProperty<SimpleMoney> detailProperty()
    {
        return detailProperty;
    }

    private ObservableList<CommentBean> getCommentList()
    {
        return saleItem.getComments();
    }

    @Override
    ObservableList<CommentBean> getComments()
    {
        LOGGER.debug("getComments(): getValue(): {}", commentProperty().getValue());
        return commentProperty().getValue();
    }

    @Override
    SimpleObjectProperty<ObservableList<CommentBean>> commentProperty()
    {
        LOGGER.debug("commentProperty(): getCommentList(): {}", getCommentList());
        comments.set(getCommentList());	//	2.9.6
        return comments;
    }

    @Override
    /**
     * Implement the comparison for sorting.
     * Items are first sorted in date order.
     * if same date, then by item type
     *      -   weather then husbandry, then affliction,
     * 		-	then groundwork, then wildlife,
     * 		-	then purchases and items within purchases,
     * 		-	then sales and sale items within sales
     * 		-	and journal entries
     * 	Items of same type on same day are sorted by key to give a stable ordering
     * 	    (this will usually correspond to sorting by order of creation).
     */
    public int compareTo(DiaryBean o)
    {
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("compareTo(): this: {}. other: {}", this.toString(), o.toString());

//        LOGGER.debug("this.dateProperty: {}", this.dateProperty.get());
//        LOGGER.debug("other.dateProperty: {}", o.dateProperty().get());

        if (!this.dateProperty.get().equals(o.dateProperty().get()))
        {//sort on date as primary key
            return this.dateProperty.get().compareTo(o.dateProperty().get());
        }

        int retVal = switch (o.getItemType())
                {
                    case SALEITEM -> this.getSaleKey().compareTo(((DiaryBeanSaleItem)o).getSaleKey()); //  keep SIs for same SA together
                    case WEATHER, HUSBANDRY, AFFLICTIONEVENT, GROUNDWORK, WILDLIFE, PURCHASE -> 1;
                    case SALE ->
                            (getSaleKey().equals(o.getKey())) ? +1 : getSaleKey().compareTo(o.getKey());
                    default -> -1;
                };
        return LOGGER.traceExit(retVal);
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        DiaryBeanSaleItem that = (DiaryBeanSaleItem) o;
        return saleItem.equals(that.saleItem) && itemKey.equals(that.itemKey) && Objects.equals(mainTitleProperty, that.mainTitleProperty) && Objects.equals(mainTitleText, that.mainTitleText) && subTitleProperty.equals(that.subTitleProperty) && subTitleText.equals(that.subTitleText) && Objects.equals(varietyProperty, that.varietyProperty) && Objects.equals(varietyText, that.varietyText) && Objects.equals(detailProperty, that.detailProperty) && Objects.equals(detailText, that.detailText) && Objects.equals(comments, that.comments) && Objects.equals(resources, that.resources);
    }

    @Override
    public int hashCode()
    {
        return Objects.hash(super.hashCode(), saleItem, itemKey, mainTitleProperty, mainTitleText, subTitleProperty, subTitleText, varietyProperty, varietyText, detailProperty, detailText, comments, resources);
    }

    @Override
    public  String toString()
    {
        return ("DiaryBeanSaleItem of: hash: " + this.hashCode() + ", " + saleItem.toString());
    }

}
