/*
 *
 *  Copyright (C) 2022 Andrew Gegg
 *
 * 	This file is part of the Gardeners Notebook application
 *
 *  The Gardeners Notebook application is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 *
 */

package uk.co.gardennotebook;

import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.EntryMessage;
import uk.co.gardennotebook.fxbean.CommentBean;
import uk.co.gardennotebook.fxbean.INotebookBean;
import uk.co.gardennotebook.fxbean.ProductBean;
import uk.co.gardennotebook.fxbean.ProductCategoryBean;
import uk.co.gardennotebook.fxbean.PurchaseItemBean;
import uk.co.gardennotebook.spi.GNDBException;
import uk.co.gardennotebook.spi.INotebookEntry;
import uk.co.gardennotebook.spi.NotebookEntryType;

import java.time.LocalDate;
import java.util.Objects;
import java.util.ResourceBundle;

/**
 *	A specific implementation of DiaryBean for PurchaseItem diary entries.
 *
 * @author Andy Gegg
 *	@version	3.0.5
 *	@since	3.0.5
 */
final class DiaryBeanPurchaseItem extends DiaryBean
{
    private static final Logger LOGGER = LogManager.getLogger();

    private final PurchaseItemBean purchaseItem;

    private Integer itemKey = 0;

    final ObjectProperty<LocalDate> dateProperty;

    private final ObjectProperty<?> mainTitleProperty = null;	//	the title - null here
    private final StringBinding mainTitleText;
    private final ObjectProperty<ProductCategoryBean> subTitleProperty;	//	the ProductCategory
    private final StringBinding subTitleText;
    private final ObjectProperty<ProductBean> varietyProperty;	// the Product or PlantSpecies
    private final StringBinding varietyText;
    private final ObjectProperty<ProductBean> detailProperty;    //  ProductDetail_1 or PlantVariety
    private final StringBinding detailText;

    private final SimpleObjectProperty<ObservableList<CommentBean>> comments =
            new SimpleObjectProperty<>(this, "comments", FXCollections.emptyObservableList());

    private final ResourceBundle resources = ResourceBundle.getBundle("notebook");

    DiaryBeanPurchaseItem(PurchaseItemBean purchaseItemBean)
    {
        EntryMessage log4jEntryMsg = LOGGER.traceEntry("DiaryBeanPurchaseItem: constructor(): PurchaseBean:{}", purchaseItemBean);

        purchaseItem = purchaseItemBean;
        itemKey = purchaseItem.getKey();

        dateProperty = purchaseItem.getPurchase().dateProperty();

//        mainTitleProperty = null;
        mainTitleText = new StringBinding()
        {
            @Override
            protected String computeValue()
            {
                return "";
            }
        };

        subTitleProperty = purchaseItem.getProduct().productCategoryProperty();
        subTitleText = new StringBinding()
        {
            {
                super.bind(subTitleProperty);
            }
            @Override
            protected String computeValue()
            {
                return purchaseItem.getProduct().getProductCategory().getName();
            }

            @Override
            public void dispose()
            {
                super.unbind(subTitleProperty);
            }
        };

        varietyProperty = purchaseItem.productProperty();
        varietyText = new StringBinding()
        {
            {
                super.bind(varietyProperty);
            }
            @Override
            protected String computeValue()
            {
                if (purchaseItem.hasPlantSpecies()) return purchaseItem.getPlantSpecies().getCommonName();
                return purchaseItem.getProduct().getName();
            }

            @Override
            public void dispose()
            {
                super.unbind(varietyProperty);
            }
        };

        detailProperty = purchaseItem.productProperty();;
        detailText = new StringBinding()
        {
            {
                super.bind(detailProperty);
            }
            @Override
            protected String computeValue()
            {
                if (purchaseItem.hasPlantVariety()) return purchaseItem.getPlantVariety().getCommonName();
                return purchaseItem.getProduct().getNameDetail_1();
            }

            @Override
            public void dispose()
            {
                super.unbind(detailProperty);
            }
        };

        LOGGER.traceExit(log4jEntryMsg);
    }

    @Override
    INotebookBean getItem()
    {
        return purchaseItem;
    }

    @Override
    INotebookEntry getBaseItem()
    {
        return purchaseItem.get().orElse(null);
    }

    @Override
    Integer getKey()
    {
        return itemKey;
    }

    Integer getPurchaseKey()
    {
        return purchaseItem.getPurchase().getKey();
    }

    @Override
    NotebookEntryType getItemType()
    {
        return NotebookEntryType.PURCHASEITEM;
    }

    @Override
    boolean canDelete() throws GNDBException
    {
        return purchaseItem.canDelete();
    }

    /**
     * The date of this diary entry
     *
     * @return	the (value of) the dateProperty
     */
    @Override
    LocalDate getDate()
    {
        return dateProperty.getValue();
    }

    @Override
    void setDate(LocalDate date)
    {
        dateProperty().setValue(date);
    }

    @Override
    ObjectProperty<LocalDate> dateProperty()
    {
        return dateProperty;
    }

    /**
     * The title for this Diary entry - here, null (to group under the Purchase)
     *
     * @return	the (value of) the mainTitleProperty
     */
    @Override
    StringBinding getMainTitleText()
    {
        return mainTitleText;
    }

    @Override
    ObjectProperty<?> mainTitleProperty()
    {
        return mainTitleProperty;
    }

    /**
     * The first detail for this Diary entry - here, the ProductCategory name
     *
     * @return	the (value of) the subTitleProperty
     */
    @Override
    StringBinding getSubTitleText()
    {
        return subTitleText;
    }

    @Override
    ObjectProperty<ProductCategoryBean> subTitleProperty()
    {
        return subTitleProperty;
    }

    /**
     * The second detail for this Diary entry - here, the Product or PlantSpecies
     *
     * @return	the (value of) the varietyProperty
     */
    @Override
    StringBinding getVarietyText()
    {
        return varietyText;
    }

    @Override
    ObjectProperty<ProductBean> varietyProperty()
    {
        return varietyProperty;
    }

    /**
     * The third detail for this Diary entry - here, the ProductDetail_1 or PlantVariety
     *
     * @return	the (value of) the detailProperty
     */
    @Override
    StringBinding getDetailText()
    {
        return detailText;
    }

    @Override
    ObjectProperty<ProductBean> detailProperty()
    {
        return detailProperty;
    }

    private ObservableList<CommentBean> getCommentList()
    {
        return purchaseItem.getComments();
    }

    @Override
    ObservableList<CommentBean> getComments()
    {
        LOGGER.debug("getComments(): getValue(): {}", commentProperty().getValue());
        return commentProperty().getValue();
    }

    @Override
    SimpleObjectProperty<ObservableList<CommentBean>> commentProperty()
    {
        LOGGER.debug("commentProperty(): getCommentList(): {}", getCommentList());
        comments.set(getCommentList());	//	2.9.6
        return comments;
    }

    @Override
    /**
     * Implement the comparison for sorting.
     * Items are first sorted in date order.
     * if same date, then by item type
     *      -   weather then husbandry, then affliction,
     * 		-	then groundwork, then wildlife,
     * 		-	then purchases and items within purchases,
     * 		-	then sales and sale items within sales
     * 		-	and journal entries
     * 	Items of same type on same day are sorted by key to give a stable ordering
     * 	    (this will usually correspond to sorting by order of creation).
     */
    public int compareTo(DiaryBean o)
    {
		EntryMessage log4jEntryMsg = LOGGER.traceEntry("compareTo(): this: {}. other: {}", this.toString(), o.toString());

//        LOGGER.debug("this.dateProperty: {}", this.dateProperty.get());
//        LOGGER.debug("other.dateProperty: {}", o.dateProperty().get());

        if (!this.dateProperty.get().equals(o.dateProperty().get()))
        {//sort on date as primary key
            return this.dateProperty.get().compareTo(o.dateProperty().get());
        }

        int retVal = switch (o.getItemType())
                {
                    case PURCHASEITEM -> this.getPurchaseKey().compareTo(((DiaryBeanPurchaseItem)o).getPurchaseKey()); //  keep PIs for same PU together
                    case WEATHER, HUSBANDRY, AFFLICTIONEVENT, GROUNDWORK, WILDLIFE -> 1;
                    case PURCHASE ->
                            (getPurchaseKey().equals(o.getKey())) ? +1 : getPurchaseKey().compareTo(o.getKey());
                    default -> -1;
                };
        return LOGGER.traceExit(retVal);
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        DiaryBeanPurchaseItem that = (DiaryBeanPurchaseItem) o;
        return purchaseItem.equals(that.purchaseItem) && itemKey.equals(that.itemKey) && mainTitleProperty.equals(that.mainTitleProperty) && mainTitleText.equals(that.mainTitleText) && subTitleProperty.equals(that.subTitleProperty) && subTitleText.equals(that.subTitleText) && Objects.equals(varietyProperty, that.varietyProperty) && Objects.equals(varietyText, that.varietyText) && Objects.equals(detailProperty, that.detailProperty) && Objects.equals(detailText, that.detailText) && Objects.equals(comments, that.comments) && Objects.equals(resources, that.resources);
    }

    @Override
    public int hashCode()
    {
        return Objects.hash(super.hashCode(), purchaseItem, itemKey, mainTitleProperty, mainTitleText, subTitleProperty, subTitleText, varietyProperty, varietyText, detailProperty, detailText, comments, resources);
    }

    @Override
    public  String toString()
    {
        return ("DiaryBeanPurchaseItem of: hash: " + this.hashCode() + ", " + purchaseItem.toString());
    }

}
